﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Bridge;
using App.Controls;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using App.Command;
using System.Diagnostics;

namespace App.Bridge
{
    public class BridgeNotificationHandlerStringFormat : INotificationHandlerStringFormat
    {
        const string NoConversion = "{0}";

        public string UnsavedFile
        {
            get { return NoConversion; }
        }

        public string UnresolvedTexture
        {
            get { return NoConversion; }
        }
    }

    public class BridgeNotificationHandler : INotificationHandler
    {
        Dictionary<MessageContext, Tuple<DialogResult, bool>> MessageBoxConfig;

        public BridgeNotificationHandler()
        {
            // メッセージコンテキスト、ダイアログに対する応答、エラー扱いするかどうか？
            var items = new TupleList<MessageContext, DialogResult, bool>
            {
                {MessageContext.IO_ReloadFromMenuModified, DialogResult.Yes, false},
                {MessageContext.IO_ReloadModified, DialogResult.Yes, false},
                {MessageContext.IO_Reload, DialogResult.Yes, false},
                {MessageContext.IO_CloseSceneAnimationBoundToAnObject, DialogResult.Yes, false},
                {MessageContext.IO_CloseAnimationBoundToObjects, DialogResult.Yes, false},
                {MessageContext.IO_CloseAnimationBoundToAnObject, DialogResult.Yes, false},
                {MessageContext.IO_FileIsReferenced, DialogResult.Yes, false},
                {MessageContext.IO_NotMaterialShaderAssined, DialogResult.Yes, false},
                {MessageContext.DocumentManager_LoadFromFile_SkipReload, DialogResult.Yes, true},
                {MessageContext.IO_FailedToBindDialog, DialogResult.Yes, false},
                {MessageContext.IO_InvalidItems, DialogResult.Yes, false},
                {MessageContext.IO_Load_Failed, DialogResult.Yes, true},
                {MessageContext.ShaderConvert_InvalidItems, DialogResult.Yes, false},
                {MessageContext.IO_SkeletalAnimation_MoveRetagetingHostModelToProject, DialogResult.Yes, false},
                {MessageContext.NoFileLocation, DialogResult.No, false},
                {MessageContext.IO_SaveFileReadOnly, DialogResult.Cancel, true},
                {MessageContext.TexturePatternAnimation_OutOfRange, DialogResult.Yes, false},
                {MessageContext.IO_TextureReferenceCanNotResolved, DialogResult.Yes, false},
                {MessageContext.IO_PreOpenCommand, DialogResult.No, true},
                {MessageContext.IO_PreSaveCommand, DialogResult.No, true},
                {MessageContext.IO_PostSaveCommand, DialogResult.No, true},
                {MessageContext.IO_PreBinarizeCommand, DialogResult.No, true},
                {MessageContext.IO_PostCloseCommand, DialogResult.No, true},
            };

            // すべて登録されているべき
            Debug.Assert(items.Select(x => x.Item1).SequenceEqual(Enum.GetValues(typeof(MessageContext)).OfType<MessageContext>()));

            MessageBoxConfig = items.ToDictionary(x => x.Item1, x => new Tuple<DialogResult, bool>(x.Item2, x.Item3));
        }

        BridgeNotificationHandlerStringFormat stringFormat = new BridgeNotificationHandlerStringFormat();
        public INotificationHandlerStringFormat StringFormat
        {
            get { return stringFormat; }
        }

        public List<BridgeInfo.Result> results = new List<BridgeInfo.Result>();
        public void WriteMessageLog(MessageLog.LogType type, string message, Action onDoubleClick = null)
        {
            BridgeInfo.Result.ResultType resultType = BridgeInfo.Result.ResultType.Error;
            bool isError = false;
            switch (type)
            {
                case MessageLog.LogType.Error:
                case MessageLog.LogType.CriticalError:
                    resultType = BridgeInfo.Result.ResultType.Error;
                    isError = true;
                    break;
                case MessageLog.LogType.Warning:
                    resultType = BridgeInfo.Result.ResultType.Warning;
                    break;
                case MessageLog.LogType.Help:
                case MessageLog.LogType.Information:
                case MessageLog.LogType.Question:
                    resultType = BridgeInfo.Result.ResultType.Information;
                    break;
            }
            results.Add(new BridgeInfo.Result()
            {
                Type = resultType,
                Message = message,
                error = isError,
            });
        }

        public void MessageBoxInformation(string message)
        {
            results.Add(new BridgeInfo.Result()
            {
                Type = BridgeInfo.Result.ResultType.Information,
                Message = message,
                error = false,
            });
        }

        /// <summary>
        /// 情報メッセージ。
        /// </summary>
        public void MessageBoxInformation(string format, params object[] args)
        {
            MessageBoxInformation(string.Format(format, args));
        }

        /// <summary>
        /// 警告メッセージ。
        /// </summary>
        public void MessageBoxWarning(string message)
        {
            results.Add(new BridgeInfo.Result()
            {
                Type = BridgeInfo.Result.ResultType.Warning,
                Message = message,
                error = false,
            });
        }

        public void MessageBoxWarning(string format, params object[] args)
        {
            MessageBoxWarning(string.Format(format, args));
        }

        /// <summary>
        /// エラーメッセージ。
        /// </summary>
        public void MessageBoxError(string message)
        {
            results.Add(new BridgeInfo.Result()
            {
                Type = BridgeInfo.Result.ResultType.Error,
                Message = message,
                error = true,
            });
        }

        public void MessageBoxError(string format, params object[] args)
        {
            MessageBoxError(string.Format(format, args));
        }

        private string FormatDialogMessage(string message, DialogResult dialogResult)
        {
            return message + " -> " + UIText.EnumValue(dialogResult);
        }

        /// <summary>
        /// はい、いいえメッセージ。
        /// </summary>
        public bool MessageBoxYesNo(MessageContext context, string message)
        {
            var dialogResult = MessageBoxConfig[context].Item1;
            var error = MessageBoxConfig[context].Item2;
            results.Add(new BridgeInfo.Result()
                {
                    Type = BridgeInfo.Result.ResultType.Information,
                    Message = FormatDialogMessage(message, dialogResult),
                    error = error,
                });
            return dialogResult == DialogResult.Yes || dialogResult == DialogResult.OK;
        }

        public bool MessageBoxYesNo(MessageContext context, string format, params object[] args)
        {
            return MessageBoxYesNo(context, string.Format(format, args));
        }

        /// <summary>
        /// はい、いいえ、キャンセルメッセージ。
        /// </summary>
        public DialogResult MessageBoxYesNoCancel(MessageContext context, string message)
        {
            var dialogResult = MessageBoxConfig[context].Item1;
            var error = MessageBoxConfig[context].Item2;
            results.Add(new BridgeInfo.Result()
                {
                    Type = BridgeInfo.Result.ResultType.Information,
                    Message = FormatDialogMessage(message, dialogResult),
                    error = error,
                }
            );

            return dialogResult;
        }

        public DialogResult MessageBoxYesNoCancel(MessageContext context, string format, params object[] args)
        {
            return MessageBoxYesNoCancel(context, string.Format(format, args));
        }

        /// <summary>
        /// ＯＫ、キャンセルメッセージ。
        /// </summary>
        public bool MessageBoxOkCancel(MessageContext context, string message)
        {
            return MessageBoxYesNo(context, message);
        }

        public void SkeletalAnimBindDialog(SkeletalAnimBindDialog.ErrorInfo[] skeletalAnimError, bool canCancel)
        {
            foreach (var item in skeletalAnimError)
            {
                results.Add(new BridgeInfo.Result()
                {
                    Type = BridgeInfo.Result.ResultType.Information,
                    Message = string.Format(res.Strings.Bridge_SkeletalAnimBindError, item.animation.Name, item.model.Name),
                    error = false,
                });
            }
        }

        public IEnumerable<ShaderAssignUtility.MaterialAnimationError> FixShaderParamAnimDialog(ShaderAssignUtility.MaterialAnimationError[] errorMaterialAnims)
        {
            foreach (var item in errorMaterialAnims.GroupBy(x => x.anim))
            {
                results.Add(new BridgeInfo.Result()
                {
                    Type = BridgeInfo.Result.ResultType.Information,
                    Message = string.Format(res.Strings.Bridge_FixShaderParamAnim, ((Document)item.Key).FileName),
                    error = false,
                });
            }
            return errorMaterialAnims.ToArray();
        }

        public void OkListBoxDialog(MessageContext context, string text, string label, IEnumerable<string> lines)
        {
            var error = MessageBoxConfig[context].Item2;
            var builder = new StringBuilder();
            builder.AppendLine(label);
            foreach (var line in lines)
            {
                builder.AppendLine(line);
            }

            results.Add(new BridgeInfo.Result()
            {
                Type = BridgeInfo.Result.ResultType.Information,
                Message = builder.ToString(),
                error = error,
            });
        }

        public bool OkCancelListBoxDialog(MessageContext context, string text, string label, string format, IEnumerable<Document> documents)
        {
            var builder = new StringBuilder();
            builder.AppendLine(label);
            foreach (var document in documents)
            {
                builder.AppendLine(string.Format(format, document.FileName));
            }

            return MessageBoxOkCancel(context, builder.ToString());
        }

        public bool OkCancelListBoxDialog(MessageContext context, string text, string label, string format, IEnumerable<Tuple<string, System.Drawing.Image>> items)
        {
            var builder = new StringBuilder();
            builder.AppendLine(label);
            foreach (var item in items)
            {
                builder.AppendLine(string.Format(format, item.Item1));
            }

            return MessageBoxOkCancel(context, builder.ToString());
        }

        public bool OkCancelTextBoxDialog(MessageContext context, string text, string label, IEnumerable<string> lines)
        {
            var builder = new StringBuilder();
            builder.AppendLine(label);
            foreach (var line in lines)
            {
                builder.AppendLine(line);
            }

            return MessageBoxOkCancel(context, builder.ToString());
        }

        public bool SameFileReload(IEnumerable<Document> updateddocs)
        {
            // 何も言わずに再読み込み
            return true;
        }

        public void FileErrorDialog(MessageContext context, IEnumerable<Tuple<string, string, string>> errors, string message)
        {
            var error = MessageBoxConfig[context].Item2;
            var builder = new StringBuilder();
            builder.AppendLine(message);
            foreach (var item in errors)
            {
                builder.AppendLine(string.Format("{0}\t{1}\t{2}", item.Item1, item.Item2, item.Item3));
            }
            results.Add(new BridgeInfo.Result()
            {
                Type = BridgeInfo.Result.ResultType.Information,
                Message = builder.ToString(),
                error = error,
            });
        }

        public Model ModelSelectDialog(IEnumerable<Model> models, AnimationDocument animation, string chkText, ref bool applyAll)
        {
            // 何も選ばない
            return null;
        }

        public void FixLightAnimAssignDialog(LightAnimation[] errorLightAnims, out LightAnimation[] fixLightAnims, out LightAnimation[] notFixLightAnims)
        {
            fixLightAnims = errorLightAnims.ToArray();
            notFixLightAnims = new LightAnimation[0];
            DocumentManager.MessageLogFixLightAnimation(errorLightAnims, this);
        }

        public void FixShaderAssignDialog(Material[] errorMaterials, out Material[] fixMaterials, out Material[] notFixMaterials)
        {
            fixMaterials = errorMaterials.ToArray();
            notFixMaterials = new Material[0];
            DocumentManager.MessageLogFixMaterial(errorMaterials, this);
        }

        public List<Tuple<Document, string>> SaveDirectoryDialog(List<Document> savedDocuments)
        {
            // 来ないはず？
            Debug.Assert(false);

            return new List<Tuple<Document, string>>();
        }

        public string SaveFileDialog(string filter, string ext, string title, string fileName = "", string fileLocation = "", bool prompt = true)
        {
            results.Add(new BridgeInfo.Result()
            {
                Type = BridgeInfo.Result.ResultType.Information,
                Message = string.Format(res.Strings.Bridge_SaveCancel, fileName),
                error = true,
            });
            return "";
        }

        public bool DocumentsCloseDialog(IEnumerable<Document> modified, out IEnumerable<Document> saveRequired)
        {
            foreach (var item in modified)
            {
                results.Add(new BridgeInfo.Result()
                {
                    Type = BridgeInfo.Result.ResultType.Information,
                    Message = string.Format(res.Strings.Bridge_Modified, item.FileName),
                    error = false,
                });
            }
            saveRequired = null;
            return true;
        }
    }
}
