﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "EditBinaryBlock.h"
#include <algorithm>
#include <functional>

namespace nw { namespace g3d { namespace tool {

void StringPool::Finalize()
{
    this->Sort();

    this->CalcSize();
}

void StringPool::Sort()
{
    auto predLess = [](const std::string* lhs, const std::string* rhs) {
        return strcmp(lhs->c_str(), rhs->c_str()) < 0;
    };
    std::sort(m_Strings.begin(), m_Strings.end(), predLess);
    std::sort(m_Paths.begin(), m_Paths.end(), predLess);

    // 重複の除去
    auto predEqual = [](const std::string* lhs, const std::string* rhs) -> bool {
        if (lhs->size() != rhs->size())
        {
            return false;
        }

        // ソート済みの文字列が対象なので後ろから比較する。
        auto size = lhs->size();
        const char* strLhs = lhs->c_str() + size;
        const char* strRhs = rhs->c_str() + size;
        while (size-- > 0)
        {
            if (*(--strLhs) != *(--strRhs))
            {
                return false;
            }
        }
        return true;
    };
    m_Strings.erase(std::unique(m_Strings.begin(), m_Strings.end(), predEqual), m_Strings.end());
    m_Paths.erase(std::unique(m_Paths.begin(), m_Paths.end(), predEqual), m_Paths.end());
}

void StringPool::CalcSize()
{
    // ブロックサイズの計算
    size_t size = 0;
    auto funcCount = [&size](const std::string* pStr) {
        size_t sizeName = sizeof(ResNameData::LengthType) + pStr->length() + 1; // length + NUL終端文字列
        size += Align(sizeName, sizeof(ResNameData::LengthType));
    };
    std::for_each(m_Strings.cbegin(), m_Strings.cend(), funcCount);
    std::for_each(m_Paths.cbegin(), m_Paths.cend(), funcCount);
    m_Chunk.size = size;
}

void StringPool::Write(void* pBuffer)
{
    m_ResName.reserve(m_Strings.size());
    m_ResNamePath.reserve(m_Paths.size());

    u32* pCur = AddOffset<u32>(pBuffer, m_Chunk.offset);
    m_pAddress = pCur;
    auto writeName = [&](const std::string* pStr)
    {
        m_ResName.push_back(reinterpret_cast<ResNameData*>(pCur));

        u32 len = static_cast<u32>(pStr->size());
        *pCur++ = len;

        pCur[len >> 2] = 0;
        std::memcpy(pCur, pStr->c_str(), len);
        pCur += (len + 4) >> 2;
    };

    auto writePath = [&](const std::string* pStr)
    {
        m_ResNamePath.push_back(reinterpret_cast<ResNameData*>(pCur));

        u32 len = static_cast<u32>(pStr->size());
        *pCur++ = len;

        pCur[len >> 2] = 0;
        std::memcpy(pCur, pStr->c_str(), len);
        pCur += (len + 4) >> 2;
    };
    std::for_each(m_Strings.cbegin(), m_Strings.cend(), writeName);
    std::for_each(m_Paths.cbegin(), m_Paths.cend(), writePath);
}

const char* StringPool::GetPtr(const std::string& str) const
{
    auto funcEqual = [&str](const ResNameData* pName) -> bool {
        if (pName->len != str.size())
        {
            return false;
        }

        // とりあえず後ろから比較する。
        auto size = pName->len;
        const char* strLhs = pName->str + size;
        const char* strRhs = str.c_str() + size;
        while (size-- > 0)
        {
            if (*(--strLhs) != *(--strRhs))
            {
                return false;
            }
        }
        return true;
    };

    auto result = std::find_if(m_ResName.begin(), m_ResName.end(), funcEqual);
    return result == m_ResName.end() ? nullptr : (*result)->str;
}

const char* StringPool::GetPathPtr(const std::string& str) const
{
    auto funcEqual = [&str](const ResNameData* pName) -> bool {
        if (pName->len != str.size())
        {
            return false;
        }

        // とりあえず後ろから比較する。
        auto size = pName->len;
        const char* strLhs = pName->str + size;
        const char* strRhs = str.c_str() + size;
        while (size-- > 0)
        {
            if (*(--strLhs) != *(--strRhs))
            {
                return false;
            }
        }
        return true;
    };

    auto result = std::find_if(m_ResNamePath.begin(), m_ResNamePath.end(), funcEqual);
    return result == m_ResNamePath.end() ? nullptr : (*result)->str;
}

void EditBinaryBlock::CalcOffset(Context& ctx)
{
    for (int index = 0; index < NUM_CATEGORY; ++index)
    {
        if (m_Block[index].size)
        {
            if (index == BIN_TEXTURE)
            {
                int insertPos = 0;

                // ソート済みリストへの追加
                for (auto iter = ctx.textureChunks.begin(); iter != ctx.textureChunks.end(); ++iter)
                {
                    if ((*iter)->alignment < m_Block[BIN_TEXTURE].alignment)
                    {
                        break;
                    }
                    ++insertPos;
                }

                ctx.textureChunks.insert(ctx.textureChunks.begin() + insertPos, &m_Block[BIN_TEXTURE]);

                if (insertPos == 0)
                {
                    m_Block[BIN_TEXTURE].offset = 0;
                }
                else
                {
                    m_Block[BIN_TEXTURE].offset = ctx.textureChunks[insertPos-1]->offset + ctx.textureChunks[insertPos-1]->size;
                }

                size_t offset = 0;
                for (int i = insertPos + 1; i < static_cast<int>(ctx.textureChunks.size()); ++i)
                {
                    ctx.textureChunks[i]->offset = ctx.textureChunks[i-1]->offset + ctx.textureChunks[i-1]->size;
                }
            }
            else
            {
                m_Block[index].offset = ctx.chunk[index].size;
            }
            ctx.chunk[index].size += m_Block[index].size;
        }
    }
}

void* EditBinaryBlock::GetPtr(const Context& ctx, Category category, ptrdiff_t offset) const
{
    if (m_Block[category].size > 0)
    {
        return nw::g3d::ut::AddOffset(
            ctx.pBuf, ctx.chunk[category].offset + m_Block[category].offset + offset);
    }
    return nullptr;
}

}}} // namespace nw::g3d::tool
