﻿using System.Collections.Generic;
using System.ComponentModel;

namespace Nintendo.InGameEditing.UI
{
    /// <summary>
    /// 子要素を配置する方向を表します。
    /// </summary>
    public enum Orientation
    {
        /// <summary>
        /// 子要素を水平方向に配置します。
        /// </summary>
        Horizontal = 0,

        /// <summary>
        /// 子要素を垂直方向に配置します。
        /// </summary>
        Vertical = 1
    }

    /// <summary>
    /// パネルの種別を表します。
    /// </summary>
    public enum PanelType
    {
        /// <summary>
        /// 子要素を単一行に整列するパネルを表します。
        /// </summary>
        Stack = 0,

        /// <summary>
        /// 子要素を複数行に折り返して配置するパネルを表します。
        /// </summary>
        Wrap = 1
    }

    /// <summary>
    /// グループボックスコントロールです。
    /// </summary>
    public sealed class PanelModel : ControlModel, IHasChildren
    {
        private static readonly PropertyChangedEventArgs WidthEventArgs = new PropertyChangedEventArgs(nameof(Width));
        private static readonly PropertyChangedEventArgs HeightEventArgs = new PropertyChangedEventArgs(nameof(Height));
        private static readonly PropertyChangedEventArgs OrientationEventArgs = new PropertyChangedEventArgs(nameof(Orientation));
        private static readonly PropertyChangedEventArgs PanelTypeEventArgs = new PropertyChangedEventArgs(nameof(PanelType));
        private static readonly PropertyChangedEventArgs BackgroundEventArgs = new PropertyChangedEventArgs(nameof(Background));
        private const Orientation DefaultOrientation = Orientation.Vertical;
        private const PanelType DefaultPanelType = PanelType.Stack;
        internal static readonly string UiType = "Panel";

        private int? width = null;
        private int? height = null;
        private Orientation orientation = DefaultOrientation;
        private PanelType panelType = DefaultPanelType;
        private string background;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="node">元になるノード</param>
        /// <param name="attrs">メタデータに指定された属性の辞書</param>
        internal PanelModel(Node node, Dictionary<string, string> attrs) : base(node, attrs)
        {
            Children = new ControlModelList(node.Children);
        }

        /// <summary>
        /// グループボックス内に含まれる子要素のリストを取得します。
        /// </summary>
        public ControlModelList Children { get; }

        /// <summary>
        /// パネルの幅を指定します。
        /// null の場合、明示的に指定されていないことを示します。
        /// </summary>
        public int? Width
        {
            get { return width; }
            private set { SetPropertyValue(ref width, value, WidthEventArgs); }
        }

        /// <summary>
        /// パネルの高さを指定します。
        /// nullの場合、明示的に指定されていないことを示します。
        /// </summary>
        public int? Height
        {
            get { return height; }
            private set { SetPropertyValue(ref height, value, HeightEventArgs); }
        }

        /// <summary>
        /// パネルの子要素を並べる方向を取得します。
        /// </summary>
        public Orientation Orientation
        {
            get { return orientation; }
            private set { SetPropertyValue(ref orientation, value, OrientationEventArgs); }
        }

        /// <summary>
        /// パネルの種別を取得します。
        /// </summary>
        public PanelType PanelType
        {
            get { return panelType; }
            private set { SetPropertyValue(ref panelType, value, PanelTypeEventArgs); }
        }

        /// <summary>
        /// パネルの背景色を取得します。
        /// </summary>
        public string Background
        {
            get { return background; }
            private set { SetPropertyValue(ref background, value, BackgroundEventArgs); }
        }

        protected override void UpdateAttributes(Dictionary<string, string> attrs)
        {
            base.UpdateAttributes(attrs);

            var width = default(string);
            int widthValue = default(int);
            if (attrs.TryGetValue(nameof(width), out width) &&
                int.TryParse(width, out widthValue) &&
                widthValue >= 0)
            {
                Width = widthValue;
            }
            else
            {
                Width = null;
            }

            var height = default(string);
            int heightValue = default(int);
            if (attrs.TryGetValue(nameof(height), out height) &&
                int.TryParse(height, out heightValue) &&
                heightValue >= 0)
            {
                Height = heightValue;
            }
            else
            {
                Height = null;
            }

            var orientation = default(string);
            var orientationValue = default(bool);
            if (attrs.TryGetValue(nameof(orientation), out orientation) && bool.TryParse(orientation, out orientationValue))
            {
                // true == 1, false = 0;
                Orientation = orientationValue ? Orientation.Vertical : Orientation.Horizontal;
            }
            else
            {
                Orientation = DefaultOrientation;
            }

            var panelType = default(string);
            var panelTypeValue = default(bool);
            if (attrs.TryGetValue(nameof(panelType), out panelType) && bool.TryParse(panelType, out panelTypeValue))
            {
                // true == 1, false = 0;
                PanelType = panelTypeValue ? PanelType.Wrap : PanelType.Stack;
            }
            else
            {
                PanelType = DefaultPanelType;
            }

            var background = default(string);
            if (attrs.TryGetValue(nameof(background), out background))
            {
                Background = background;
            }
            else
            {
                Background = null;
            }
        }
    }
}
