﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reactive.Linq;
using Nintendo.InGameEditing.Messages;

namespace Nintendo.InGameEditing
{
    public class ValueNode<T> : Node, IValueNode
    {
        private readonly ITypeSerializer<T> genericSerializer;
        private T currentValue;

        internal ValueNode(EditService root, NodeMessage nodeInfo)
            : base(root, nodeInfo)
        {
            if (nodeInfo.NodeType != NodeType.Value) { throw new ArgumentException(nameof(nodeInfo)); }
            TypeName = nodeInfo.TypeName;

            if (!root.TryGetTypeSerializer(TypeName, out genericSerializer))
            {
                throw new InvalidOperationException($"型名 '{TypeName}' に対する TypeSerializer の取得に失敗しました。TypeSerializer を TypeSerializerProvider に登録してください。");
            }

            Receiver.OfType<ValueMessage>()
                .Select(message => genericSerializer.Deserialize(message.Value))
                .Subscribe(OnValueReceived, e => Service?.ErrorReceiver.OnNext(e));
        }

        /// <summary>
        /// 値の受信時に実行されます。
        /// </summary>
        public event EventHandler<ValueReceivedEventArgs<T>> ValueReceived;

        /// <summary>
        /// サーバーで指定された型名を取得します。
        /// </summary>
        public string TypeName { get; }

        /// <summary>
        /// 現在のノードの値を取得します。
        /// </summary>
        public T Value => currentValue;

        /// <summary>
        /// サーバーに最新の値を要求し、更新します。
        /// </summary>
        public void PullValue() => Service?.Sender.OnNext(new ValueRequestMessage(Id));

        /// <summary>
        /// サーバーに値を通知します。
        /// </summary>
        /// <param name="value">通知する値</param>
        public void PushValue(T value)
            => Service?.Sender.OnNext(new ValueMessage(Id, genericSerializer.Serialize(value)));

        /// <summary>
        /// 値の受信時に呼び出されます。
        /// </summary>
        /// <param name="value">受信した値</param>
        protected virtual void OnValueReceived(T value)
        {
            lock (LockObj)
            {
                currentValue = value;
                ValueReceived?.Invoke(this, new ValueReceivedEventArgs<T>(value));
            }
        }

        protected override void DisposeInternal()
        {
            ValueReceived = null;
            base.DisposeInternal();
        }
    }
}
