﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace Nintendo.InGameEditing.Utilities
{
    internal static class NullTerminatedStringHelper
    {
        /// <summary>
        /// MemoryStream から null 終端された文字列を読み取り、ストリームの現在位置を進めます。
        /// MemoryStream の publiclyVisible が true に設定されている必要があります。
        /// </summary>
        /// <param name="stream">対象のメモリストリーム</param>
        /// <param name="encoding">文字列のエンコーディング</param>
        /// <returns>読み取った文字列</returns>
        internal static string ReadNullTerminatedString(this MemoryStream stream, Encoding encoding)
        {
            if (stream == null) { throw new ArgumentNullException(nameof(stream)); }
            if (encoding == null) { throw new ArgumentNullException(nameof(encoding)); }

            var buffer = stream.GetBuffer();
            var position = (int)stream.Position;

            int index = position;
            for (; index < buffer.Length && buffer[index] != '\0'; ++index) { }

            stream.Position = Math.Min(index + 1, buffer.Length);
            return encoding.GetString(buffer, position, index - position);
        }

        /// <summary>
        /// MemoryStream から null 終端された UTF8 の文字列を読み取り、ストリームの現在位置を進めます。
        /// MemoryStream の publiclyVisible が true に設定されている必要があります。
        /// </summary>
        /// <param name="stream">対象のメモリストリーム</param>
        /// <returns>読み取った文字列</returns>
        internal static string ReadNullTerminatedUTF8String(this MemoryStream stream)
        {
            if (stream == null) { throw new ArgumentNullException(nameof(stream)); }
            return stream.ReadNullTerminatedString(Encoding.UTF8);
        }

        internal static string ReadNullTerminatedString(this BinaryReader reader, Encoding encoding)
        {
            if (reader == null) { throw new ArgumentNullException(nameof(reader)); }
            if (encoding == null) { throw new ArgumentNullException(nameof(encoding)); }

            var buffer = new List<byte>();

            while (true)
            {
                try
                {
                    var b = reader.ReadByte();
                    if (b == 0) { break; }
                    buffer.Add(b);
                }
                catch (EndOfStreamException)
                {
                    break;
                }
            }

            return encoding.GetString(buffer.ToArray());
        }

        internal static string ReadNullTerminatedUTF8String(this BinaryReader reader)
            => reader.ReadNullTerminatedString(Encoding.UTF8);

        internal static void WriteNullTerminatedString(this BinaryWriter writer, string str, Encoding encoding)
        {
            if (writer == null) { throw new ArgumentNullException(nameof(writer)); }
            if (encoding == null) { throw new ArgumentNullException(nameof(encoding)); }

            str = str ?? string.Empty;

            var buffer = encoding.GetBytes(str);

            writer.Write(buffer);
            writer.Write((byte)0);
        }

        internal static void WriteNullTerminatedUTF8String(this BinaryWriter writer, string str)
            => writer.WriteNullTerminatedString(str, Encoding.UTF8);
    }
}
