﻿using System;
using System.ComponentModel;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using System.Reactive.Subjects;
using Nintendo.InGameEditing.Messages;

namespace Nintendo.InGameEditing
{
    /// <summary>
    /// 編集ノードクラスです。
    /// </summary>
    public class Node
    {
        internal Node(EditService root, NodeMessage nodeInfo)
        {
            if (root == null) { throw new ArgumentNullException(nameof(root)); }
            if (nodeInfo == null) { throw new ArgumentNullException(nameof(nodeInfo)); }

            Service = root;

            Id = nodeInfo.Id;
            Metadata = nodeInfo.Metadata;
            NodeType = nodeInfo.NodeType;

            Receiver.OfType<MetadataMessage>()
                .Select(message => message.Value)
                .Subscribe(OnMetadataReceived, e => Service?.ErrorReceiver.OnNext(e));

            Disposer.Add((IDisposable)Receiver);
            Disposer.Add(Disposable.Create(DisposeInternal));
        }

        /// <summary>
        /// メタデータの受信時に実行されます。
        /// </summary>
        public event EventHandler<DataReceivedEventArgs> MetadataReceived;

        /// <summary>
        /// サーバーからノードが削除されたときに実行されます。
        /// </summary>
        public event EventHandler<RemovedEventArgs> Removed;

        protected object LockObj { get; } = new object();

        internal ISubject<NodeTargetMessage> Receiver { get; } = new Subject<NodeTargetMessage>();

        internal CompositeDisposable Disposer { get; } = new CompositeDisposable();

        internal uint Id { get; }

        /// <summary>
        /// メタデータを取得します。
        /// </summary>
        public byte[] Metadata { get; private set; }

        /// <summary>
        /// ノードの種別を取得します。
        /// </summary>
        public NodeType NodeType { get; }

        /// <summary>
        /// サーバーから受信した子ノードのリストを取得します。
        /// </summary>
        public NodeList Children { get; } = new NodeList();

        /// <summary>
        /// ノードがサーバーから削除された状態かどうかを取得します。
        /// </summary>
        public bool IsRemoved => Disposer.IsDisposed;

        /// <summary>
        /// サーバーに子ノードの情報を要求します。
        /// </summary>
        public void RequestChildren() => Service?.Sender.OnNext(new ChildNodeRequestMessage(Id));

        protected EditService Service { get; private set; }

        /// <summary>
        /// メタデータの受信時に呼び出されます。
        /// </summary>
        /// <param name="metadata">受信したメタデータ</param>
        protected virtual void OnMetadataReceived(byte[] metadata)
        {
            lock (LockObj)
            {
                Metadata = metadata ?? Const.EmptyBytes;
                MetadataReceived?.Invoke(this, new DataReceivedEventArgs(metadata));
            }
        }

        /// <summary>
        /// サーバーからノードが削除されたときに呼び出されます。
        /// </summary>
        protected virtual void OnRemoved(RemovedEventArgs args)
        {
            if (args == null) throw new ArgumentNullException(nameof(args));
            lock (LockObj) { Removed?.Invoke(this, args); }
        }

        protected virtual void DisposeInternal()
        {
            var isConnected = Service?.IsConnected == true;

            MetadataReceived = null;
            Service = null;

            OnRemoved(isConnected ? RemovedEventArgs.Deleted : RemovedEventArgs.Disconnected);
            Removed = null;

            Children.RemoveItems();
        }
    }
}
