﻿using System;
using System.Reactive.Disposables;
using System.Reactive.Linq;
using Nintendo.InGameEditing.Messages;

namespace Nintendo.InGameEditing
{
    /// <summary>
    /// 編集ノードクラスです。
    /// </summary>
    public class CommandNode : Node
    {
        internal CommandNode(EditService root, NodeMessage nodeInfo)
            : base(root, nodeInfo)
        {
            Disposer.Add(Disposable.Create(() => CommandReceived = null));

            Receiver.OfType<UserCommandMessage>()
                .Select(message => new DataReceivedEventArgs(message.Value))
                .Subscribe(OnCommandReceived, e => Service?.ErrorReceiver.OnNext(e));
        }

        /// <summary>
        /// コマンドの受信時に発行されるイベントです。
        /// </summary>
        public event EventHandler<DataReceivedEventArgs> CommandReceived;

        /// <summary>
        /// コマンドを発行します。
        /// </summary>
        /// <param name="data">任意のバイト列。null の場合、空配列に相当します。</param>
        public void PushCommand(byte[] data) => Service?.Sender.OnNext(new UserCommandMessage(Id, data));

        protected virtual void OnCommandReceived(DataReceivedEventArgs args)
        {
            if (args == null) { throw new ArgumentNullException(nameof(args)); }

            lock (LockObj)
            {
                CommandReceived?.Invoke(this, args);
            }
        }
    }
}
