﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using Nintendo.InGameEditing.Communication.Htcs;

namespace Nintendo.InGameEditing.Communication
{
    /// <summary>
    /// 接続状態が更新された時に通知される情報です。
    /// </summary>
    public class ConnectionInfoUpdatedEventArgs : EventArgs
    {
        /// <summary>
        /// ターゲットリストです。
        /// </summary>
        private readonly TargetList _targetList;

        /// <summary>
        /// ポート情報リストです。
        /// </summary>
        private readonly PortMap _portMap;

        /// <summary>
        /// ターゲットリストを取得します。
        /// </summary>
        public IReadOnlyList<TargetInfo> TargetInfos
        {
            get
            {
                return _targetList.Select(target => new TargetInfo(target, _portMap)).ToList();
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="targetList">ターゲットリスト</param>
        /// <param name="portMap">ポートマッピング情報</param>
        internal ConnectionInfoUpdatedEventArgs(TargetList targetList, PortMap portMap)
        {
            this._targetList = targetList;
            this._portMap = portMap;
        }
    }

    /// <summary>
    /// ポート名とエンドポイントのペアです。
    /// </summary>
    public class PortInfo
    {
        /// <summary>
        /// マッピングのキーとなるポート名を取得します。
        /// </summary>
        public string PortName { get; }

        /// <summary>
        /// WinSockのポート情報を取得します。
        /// </summary>
        public IPEndPoint EndPoint { get; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="port">パースされたポート情報</param>
        internal PortInfo(PortMapItem port)
        {
            this.PortName = port.HtcsPortDescriptor.HtcsPortName;
            this.EndPoint = port.EndPoint;
        }

        /// <summary>
        /// 文字列化します。
        /// </summary>
        /// <returns>インスタンスの内容を表す文字列</returns>
        public override string ToString()
        {
            return $"{this.PortName} -> {this.EndPoint}";
        }
    }

    /// <summary>
    /// ポート情報を内包したターゲット情報です。
    /// </summary>
    public class TargetInfo
    {
        /// <summary>
        /// 通信システム名を取得します。
        /// </summary>
        public string PeerType { get; }

        /// <summary>
        /// ハードウェア識別名を取得します。
        /// </summary>
        public string HtcsPeerName { get; }

        /// <summary>
        /// このターゲットに紐付くポート情報リストです。
        /// </summary>
        public IReadOnlyList<PortInfo> PortInfos { get; private set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target">ターゲット情報</param>
        /// <param name="portInfos">フィルタ前のポート情報リスト</param>
        internal TargetInfo(Target target, IEnumerable<PortMapItem> portInfos)
        {
            this.PeerType = target.PeerType;
            this.HtcsPeerName = target.HtcsPeerName;

            // PeerNameで紐付くもののみを引っ掛けてリストに格納
            this.PortInfos = portInfos.Where(p => p.HtcsPortDescriptor.HtcsPeerName.Value.Equals(
                this.HtcsPeerName)).Select(port => new PortInfo(port)).ToList();
        }

        /// <summary>
        /// 文字列化します。
        /// </summary>
        /// <returns>インスタンスの内容を表す文字列</returns>
        public override string ToString()
        {
            return $"{this.PeerType} : {this.HtcsPeerName}";
        }
    }
}
