﻿using Nintendo.Foundation.Contracts;
using System;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Runtime.InteropServices;

namespace Nintendo.Atk.Binary
{
    /// <summary>
    /// Opus ヘッダ内の基本情報です。
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
    public class AudioOpusBasicInfo
    {
        /// <summary>
        /// AudioOpusBasicInfo のサイズ。（ヘッダサイズを含みます）
        /// </summary>
        public static readonly int StructSize = Marshal.SizeOf(typeof(AudioOpusBasicInfo));

        /// <summary>
        /// バイナリデータから、AudioOpusBasicInfo を生成します。
        /// </summary>
        /// <param name="reader">BinaryReader。</param>
        /// <returns>生成した AudioOpusBasicInfo。</returns>
        public static AudioOpusBasicInfo FromBinary(BinaryReader reader)
        {
            Ensure.Argument.NotNull(reader);

            var result = new AudioOpusBasicInfo();

            result.Header = AudioOpusInfoHeader.FromBinary(reader);

            if (result.Header.Type != AudioOpusInfoType.BasicInfo ||
                result.Header.Size < StructSize - AudioOpusInfoHeader.StructSize)
            {
                throw new NotSupportedException();
            }

            result.Version = reader.ReadSByte();
            result.ChannelCount = reader.ReadSByte();
            result.FrameDataSize = reader.ReadInt16();
            result.SampleRate = reader.ReadInt32();
            result.DataInfoOffset = reader.ReadInt32();
            result.FrameDataOffsetInfoOffset = reader.ReadInt32();
            result.ContextInfoOffset = reader.ReadInt32();

            // 4byte パディング
            reader.ReadUInt32();

            return result;
        }

        /// <summary>
        /// OpusInfo ヘッダ。
        /// </summary>
        public AudioOpusInfoHeader Header { get; private set; }

        /// <summary>
        /// このプロパティには適切な値が設定されない可能性があるため、現在 Nintendo.Atk ではこのプロパティをサポートしていません。
        /// このプロパティを使用しないでください。
        /// </summary>
        public SByte Version { get; set; }

        /// <summary>
        /// チャンネル数
        /// </summary>
        public SByte ChannelCount { get; set; }

        /// <summary>
        /// 1 フレームのバイトサイズ（可変ビットレートの場合は 0）
        /// </summary>
        public Int16 FrameDataSize { get; set; }

        /// <summary>
        /// サンプルレート。
        /// </summary>
        public Int32 SampleRate { get; set; }

        /// <summary>
        /// opus データへのファイル先頭からのオフセット。
        /// </summary>
        public Int32 DataInfoOffset { get; set; }

        /// <summary>
        /// このプロパティには適切な値が設定されない可能性があるため、現在 Nintendo.Atk ではこのプロパティをサポートしていません。
        /// このプロパティを使用しないでください。
        /// </summary>
        public Int32 FrameDataOffsetInfoOffset { get; set; }

        /// <summary>
        /// このプロパティには適切な値が設定されない可能性があるため、現在 Nintendo.Atk ではこのプロパティをサポートしていません。
        /// このプロパティを使用しないでください。
        /// </summary>
        public Int32 ContextInfoOffset { get; set; }

        /// <summary>
        /// パディング。
        /// </summary>
        private Int32 Padding { get; set; }
    }
}
