﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Runtime.InteropServices;

namespace Nintendo.Atk.Binary
{
    [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
    internal static class AtkNative
    {
        private const string NintendoAtkNativeAssemblyFileName = @"Nintendo.Atk.Native.dll";
        private const string NintendoAtkNativeAssemblyPath = NintendoAtkNativeAssemblyFileName;
        private const string NintendoAtkNativeX86AssemblyPath = @"x86\" + NintendoAtkNativeAssemblyFileName;

        private static IntPtr moduleHandle;

        internal const uint ResultErrorFlag = 0x80000000;

        internal enum Result : uint
        {
            // 成功値
            Success = 0,

            // エラー値
            UnknownError = ResultErrorFlag | 0,
            InvalidArgument,
            OutOfMemory,
            NotSupported,
        }

        static AtkNative()
        {
            // P/Invoke により dll が読み込まれる前に明示的に LoadLibrary します。
            moduleHandle = LoadLibrary(GetFullPath());
        }

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result OpenBfwav(IntPtr bfwavData, out IntPtr outHandle);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result CloseBfwav(IntPtr handle);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result GetBfwavWaveBinaryInfo(IntPtr handle, ref AtkNativeWaveBinaryInfo info);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern UInt64 GetRequiredBufferSizeForDecodeBfwav(IntPtr handle);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result DecodeBfwav(IntPtr handle, Int32 channel, IntPtr outputBuffer, UInt64 outputBufferSize);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result OpenBfstm(IntPtr bfstmData, out IntPtr outHandle);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result CloseBfstm(IntPtr handle);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result GetBfstmWaveBinaryInfo(IntPtr handle, ref AtkNativeWaveBinaryInfo info);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern UInt64 GetRequiredBufferSizeForDecodeBfstm(IntPtr handle);

        [DllImport(NintendoAtkNativeAssemblyFileName, CallingConvention = CallingConvention.Cdecl)]
        internal static extern AtkNative.Result DecodeBfstm(IntPtr handle, Int32 channel, IntPtr outputBuffer, UInt64 outputBufferSize);

        private static string GetFullPath()
        {
            return Path.Combine(Path.GetDirectoryName(typeof(AtkNative).Assembly.Location), GetRelativePath());
        }

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        private static string GetRelativePath()
        {
            switch (IntPtr.Size)
            {
                case sizeof(Int64):
                    return NintendoAtkNativeAssemblyPath;

                case sizeof(Int32):
                    return NintendoAtkNativeX86AssemblyPath;

                default:
                    throw new InvalidOperationException("invalid platform target.");
            }
        }

        [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Ansi)]
        private static extern IntPtr LoadLibrary([MarshalAs(UnmanagedType.LPStr)]string lpFileName);
    }
}
