﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"
#include "atkTool_BfwavReader.h"
#include "atkTool_WaveDecoder.h"
#include <nn/nn_Assert.h>
#include <nn/atk.h>
#include <nn/atkTool/atkTool_WaveBinaryInfo.h>
#include <nn/util/util_StringUtil.h>

#define REQUIRES(exp, returnValue) if(!(exp)) { return returnValue; }

namespace nn { namespace atkTool { namespace detail {

BfwavReader::BfwavReader(const void* pBfwavData)
    : m_Reader(pBfwavData)
    , m_IsAvailable(false)
{
    if (m_Reader.IsAvailable())
    {
        m_IsAvailable = m_Reader.ReadWaveInfo(&m_WaveInfo);
    }
}

nnatktoolResult BfwavReader::GetWaveBinaryInfo(nnatktoolWaveBinaryInfo& info) const
{
    switch(m_WaveInfo.sampleFormat)
    {
    case nn::atk::SampleFormat_PcmS16:
        nn::util::Strlcpy(info.format.encoding, nnatktoolWaveEncodingPcm16, sizeof(info.format.encoding));
        break;

    case nn::atk::SampleFormat_DspAdpcm:
        nn::util::Strlcpy(info.format.encoding, nnatktoolWaveEncodingAdpcm, sizeof(info.format.encoding));
        break;

    default:
        return ::ResultInvalidWaveEncoding;
    }

    info.format.channelCount = m_WaveInfo.channelCount;
    info.format.samplerate = m_WaveInfo.sampleRate;
    info.sampleCount = GetSampleCount();
    info.hasLoop = m_WaveInfo.loopFlag;

    if(info.hasLoop)
    {
        // bfwav はループを伸長しないので、originalLoopEndFrame は計算で求める
        // サンプルインデックスなので -1 する
        info.loopStartFrame = m_WaveInfo.loopStartFrame;
        info.loopEndFrame = m_WaveInfo.loopEndFrame - 1;
        info.originalLoopStartFrame = m_WaveInfo.originalLoopStartFrame;
        info.originalLoopEndFrame = info.originalLoopStartFrame + info.loopEndFrame - info.loopStartFrame;
    }
    else
    {
        info.loopStartFrame = 0;
        info.loopEndFrame = 0;
        info.originalLoopStartFrame = 0;
        info.originalLoopEndFrame = 0;
    }

    return ::ResultSuccess;
}

nnatktoolResult BfwavReader::DecodeAll(int channel, void* outputBuffer, size_t outputBufferSize) const
{
    REQUIRES(channel >= 0, ResultInvalidArgument);
    REQUIRES(channel < m_WaveInfo.channelCount, ResultInvalidArgument);
    REQUIRES(outputBuffer != nullptr, ResultInvalidArgument);
    REQUIRES(outputBufferSize >= GetRequiredBufferSizeForDecodeAll(), ResultBufferTooShort);

    int16_t* outputs = reinterpret_cast<int16_t*>(outputBuffer);
    const void* dataAddr = m_WaveInfo.channelParam[channel].dataAddress;

    nnatktoolResult result = ::ResultSuccess;

    switch (m_WaveInfo.sampleFormat)
    {
    case nn::atk::SampleFormat_DspAdpcm:
        {
            nn::codec::AdpcmContext adpcmContext = { 0 };
            adpcmContext._predScale = m_WaveInfo.channelParam[channel].adpcmParam.predScale;

            nn::codec::AdpcmParameter adpcmParameter;
            for (int i = 0; i < 16; i++)
            {
                adpcmParameter._coefficients[i] = m_WaveInfo.channelParam[channel].adpcmParam.coef[i / 2][i % 2];
            }

            DecodeAdpcm(outputs, adpcmContext, adpcmParameter, dataAddr, GetSampleCount());
        }
        break;

    case nn::atk::SampleFormat_PcmS16:
        DecodePcm16LE(dataAddr, outputs, GetSampleCount(), outputBufferSize);
        break;

    default:
        result = ::ResultInvalidWaveEncoding;
        break;
    }

    return result;
}

} } }
