﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/atk.h>
#include <nn/atk/atk_DeviceOutRecorder.h>
#include <nn/atk/fnd/io/atkfnd_Stream.h>

//! @brief  サウンド出力を録音します
class SoundRecorder
{
public:
    //! @brief  必要なメモリ量を取得します
    size_t GetRequiredMemorySize() NN_NOEXCEPT;
    //! @brief  初期化します
    void Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    //! @brief  終了します
    void Finalize() NN_NOEXCEPT;

    //! @brief  sampleCout だけサンプルを読み取ります
    //          返り値は読み込めたサンプル数です
    size_t ReadSamples(int16_t* pSampleBuffer, size_t sampleCount) NN_NOEXCEPT;
    //! @brief  1 秒あたりのサンプル数を取得します
    uint32_t GetSampleCountPerSecond() const NN_NOEXCEPT;
    //! @brief  録音中であるかどうかを取得します
    bool IsRecording() const NN_NOEXCEPT;

    //! @brief  録音を開始します
    void Start() NN_NOEXCEPT;
    //! @brief  録音を停止します
    void Stop() NN_NOEXCEPT;

private:
    //! @brief  オーディオレンダラから録音します
    class Recorder : public nn::atk::DeviceOutRecorder
    {
    public:
        //  DeviceOutRecorder の override
        virtual uint32_t OnProcessSamples(int16_t* samples, uint32_t sampleCount) NN_NOEXCEPT NN_OVERRIDE;
        virtual uint32_t GetMaxFrameLength() const NN_NOEXCEPT NN_OVERRIDE;
        virtual uint32_t GetSamplesPerSec() const NN_NOEXCEPT NN_OVERRIDE;
        virtual uint32_t GetValidChannels(uint32_t channels) const NN_NOEXCEPT NN_OVERRIDE;
    };
    //! @brief  Recorder が録音するストリームです
    class Stream : public nn::atk::detail::fnd::FileStream
    {
    public:
        //! @brief  必要なメモリ量を取得します
        size_t GetRequiredMemorySize() NN_NOEXCEPT;
        //! @brief  初期化します
        void Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT;
        //! @brief  録音を開始する準備をします
        void PrepareRecording() NN_NOEXCEPT;

        //  nn::atk::detail::fnd::FileStream の実装
        virtual nn::atk::detail::fnd::FndResult Open(const char* filePath, AccessMode openMode) NN_NOEXCEPT NN_OVERRIDE;
        virtual void Close() NN_NOEXCEPT NN_OVERRIDE;
        virtual void Flush() NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsOpened() const NN_NOEXCEPT NN_OVERRIDE;
        virtual bool CanRead() const NN_NOEXCEPT NN_OVERRIDE;
        virtual bool CanWrite() const NN_NOEXCEPT NN_OVERRIDE;
        virtual bool CanSeek() const NN_NOEXCEPT NN_OVERRIDE;
        virtual size_t GetSize() const NN_NOEXCEPT NN_OVERRIDE;
        virtual size_t Read(void* buffer, size_t length, nn::atk::detail::fnd::FndResult* result) NN_NOEXCEPT NN_OVERRIDE;
        virtual size_t Write(const void* buffer, size_t length, nn::atk::detail::fnd::FndResult* result) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::atk::detail::fnd::FndResult Seek(nn::atk::detail::fnd::position_t offset, SeekOrigin origin) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::atk::detail::fnd::position_t GetCurrentPosition() const NN_NOEXCEPT NN_OVERRIDE;
        virtual void EnableCache(void* buffer, size_t length) NN_NOEXCEPT NN_OVERRIDE;
        virtual void DisableCache() NN_NOEXCEPT NN_OVERRIDE;
        virtual bool IsCacheEnabled() const NN_NOEXCEPT NN_OVERRIDE;
        virtual int GetIoBufferAlignment() const NN_NOEXCEPT NN_OVERRIDE;
        virtual bool CanSetFsAccessLog() const NN_NOEXCEPT NN_OVERRIDE;
        virtual void* SetFsAccessLog(nn::atk::detail::fnd::FsAccessLog* pFsAccessLog) NN_NOEXCEPT NN_OVERRIDE;
        virtual nn::atk::detail::fnd::position_t GetCachePosition() NN_NOEXCEPT NN_OVERRIDE;
        virtual size_t GetCachedLength() NN_NOEXCEPT NN_OVERRIDE;

    private:
        nn::atk::detail::fnd::CriticalSection m_CriticalSection;

        char*    m_Buffer;
        size_t   m_BufferSize;
        uint32_t m_WritePosition;
        uint32_t m_ReadPosition;
        size_t   m_ReadableSize;
        bool     m_IsHeaderWritten;
    };

private:
    Recorder m_Recorder;
    Stream   m_Stream;

    bool m_IsRecording;
};
