﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Vector.h>
#include <nn/nn_SdkLog.h>
#include "gfxutil/GfxContext.h"
#include "gfxutil/Panel.h"

namespace
{
    //  パネルの枠の幅
    const float PanelEdgeWidth = 4.0f;
    //  パネルのタイトルの縦幅
    const float PanelTitleHeight = 16.0f + PanelEdgeWidth;
#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    //  パネルの枠の色
    const nn::util::Uint8x4 FocusedPanelEdgeColorTop    = GetUint8x4( 16, 16, 128, 255 );
    const nn::util::Uint8x4 FocusedPanelEdgeColorBottom = GetUint8x4(  0,  0, 128, 255 );
    const nn::util::Uint8x4 PanelEdgeColorTop           = GetUint8x4( 16, 16,  32, 255 );
    const nn::util::Uint8x4 PanelEdgeColorBottom        = GetUint8x4(  0,  0,  32, 255 );
    //  パネルのクライアント領域の色
    const nn::util::Uint8x4 PanelClientColor            = GetUint8x4(  8,  8,   8, 255 );
    //  パネルのタイトルのテキストの色
    const nn::util::Unorm8x4 FocusedPanelTitleTextColor = GetUnorm8x4( 192, 192, 192, 255 );
    const nn::util::Unorm8x4 PanelTitleTextColor        = GetUnorm8x4( 128, 128, 128, 255 );
    //  パネルのタイトルのテキストのスケール
    const float PanelTitleTextScaleX = 1.0f;
    const float PanelTitleTextScaleY = 1.0f;
#endif  //  ATKPLAYER_BUILD_CONFIG_ENABLE_GFX

    //  クライアント領域の最小の幅, 高さ
    const float ClientWidthMin  = 16.0f;
    const float ClientHeightMin = 16.0f;
    //  クライアント領域のパディング
    const float ClientPaddingX = 8.0f;
    const float ClientPaddingY = 4.0f;

    //  パネルのデフォルトの幅, 高さ
    const float PanelDefaultWidth  = 320.0f;
    const float PanelDefaultHeight = 320.0f;
    //  パネルのデフォルトの位置
    const float PanelDefaultPositionX = 0.0f;
    const float PanelDefaultPositionY = 0.0f;
    //  パネルのデフォルトのタイトル
    const char* PanelDefaultTitle = "Panel";
}

namespace gfxutil
{

NN_DEFINE_STATIC_CONSTANT( const int Panel::TitleNameLengthMax );

//  コンストラクタです
Panel::Panel() NN_NOEXCEPT
{
    SetTitle( PanelDefaultTitle );
    SetSize( PanelDefaultWidth, PanelDefaultHeight );
    SetPosition( PanelDefaultPositionX, PanelDefaultPositionY );
    SetFocused( false );
}
//  描画します
void Panel::Draw(GfxContext& gfxContext) NN_NOEXCEPT
{
#if  defined( ATKPLAYER_BUILD_CONFIG_ENABLE_GFX )
    gfxutil::FontRenderer& fontRenderer = gfxContext.GetFontRenderer();

    //  枠の描画
    const nn::util::Uint8x4& topColor = m_IsFocused ? FocusedPanelEdgeColorTop : PanelEdgeColorTop;
    const nn::util::Uint8x4& bottomColor = m_IsFocused ? FocusedPanelEdgeColorBottom : PanelEdgeColorBottom;
    gfxContext.DrawQuad( m_Position, m_Size, topColor, bottomColor );

    //  クライアント領域の描画
    gfxContext.DrawQuad( m_ClientPosition, m_ClientSize, PanelClientColor );

    //  タイトルの描画
    const float titlePositionX = m_Position._v[0] + PanelEdgeWidth;
    const float titlePositionY = m_Position._v[1] + PanelEdgeWidth;
    const nn::util::Unorm8x4& textColor = m_IsFocused ? FocusedPanelTitleTextColor : PanelTitleTextColor;

    fontRenderer.SetPosition( titlePositionX, titlePositionY );
    fontRenderer.SetColor( textColor );
    fontRenderer.SetScale( PanelTitleTextScaleX, PanelTitleTextScaleY );
    fontRenderer.Print( m_TitleName );
#else
    NN_UNUSED( gfxContext );
#endif  //  ATKPLAYER_BUILD_CONFIG_ENABLE_GFX
}

//  クライアント領域の左上位置を返します
const nn::util::Float2 Panel::GetClientPositionLeftTop() const NN_NOEXCEPT
{
    nn::util::Float2 pos = GetFloat2(
        m_ClientPosition._v[0] + ClientPaddingX,
        m_ClientPosition._v[1] + ClientPaddingY
    );
    return pos;
}
//! @brief  クライアント領域の右下位置を返します
const nn::util::Float2 Panel::GetClientPositionRightBottom() const NN_NOEXCEPT
{
    nn::util::Float2 pos = GetFloat2(
        m_ClientPosition._v[0] + m_ClientSize.x - ClientPaddingX,
        m_ClientPosition._v[1] + m_ClientSize.y - ClientPaddingY
    );
    return pos;
}

//  タイトルを設定します
void Panel::SetTitle(const char* title) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( title );
    nn::util::SNPrintf( m_TitleName, TitleNameLengthMax, title );
}
//  幅, 高さを設定します
void Panel::SetSize(float width, float height) NN_NOEXCEPT
{
    //  サイズの最小値
    const float widthMin  = ClientWidthMin  + 2.0f * ( PanelEdgeWidth + ClientPaddingX );
    const float heightMin = ClientHeightMin + 2.0f * ( PanelEdgeWidth + ClientPaddingY ) + PanelTitleHeight;

    if( width < widthMin )
    {
        NN_LOG( "too short width.\n" );
        m_Size.x = widthMin;
    }
    else
    {
        m_Size.x = width;
    }
    if( height < heightMin )
    {
        NN_LOG( "too short height.\n" );
        m_Size.y = heightMin;
    }
    else
    {
        m_Size.y = height;
    }

    SetClientParams();
}
//  位置を設定します
void Panel::SetPosition(float x, float y) NN_NOEXCEPT
{
    nn::util::VectorSet( &m_Position, x, y, 0.0f );
    SetClientParams();
}
//  このパネルにフォーカスがあるかどうかを設定します
void Panel::SetFocused(bool isFocused) NN_NOEXCEPT
{
    m_IsFocused = isFocused;
}

//  このパネルにフォーカスがあるかどうかを取得します
bool Panel::IsFocused() const NN_NOEXCEPT
{
    return m_IsFocused;
}

//  クライアント領域のパラメータを設定します
void Panel::SetClientParams() NN_NOEXCEPT
{
    m_ClientPosition._v[0] = m_Position._v[0] + PanelEdgeWidth;
    m_ClientPosition._v[1] = m_Position._v[1] + PanelTitleHeight + PanelEdgeWidth;
    m_ClientPosition._v[2] = m_Position._v[2];

    m_ClientSize.x = m_Size.x - 2.0f * PanelEdgeWidth;
    m_ClientSize.y = m_Size.y - PanelTitleHeight - 2.0f * PanelEdgeWidth;
}

}
