﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "Util.h"
#include "SoundRecorder.h"

namespace
{
    //  波形を録音するバッファの大きさ
    const size_t RecordingBufferSize = 256 * 1024;
}


//  ------------------------------------------------------------
//
//                            SoundRecorder
//
//  ------------------------------------------------------------
//  必要なメモリ量を取得します
size_t SoundRecorder::GetRequiredMemorySize() NN_NOEXCEPT
{
    size_t memSize = 0;

    memSize += m_Recorder.GetRequiredMemorySizeForRecording();
    memSize += nn::os::ThreadStackAlignment + nn::atk::DeviceOutRecorder::RequiredThreadStackSize;

    memSize += m_Stream.GetRequiredMemorySize();

    return memSize;
}
//  初期化します
void SoundRecorder::Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    NN_ABORT_UNLESS_GREATER_EQUAL( bufferSize, GetRequiredMemorySize() );

    nn::util::BytePtr ptr( buffer );
    {
        void* recordingBuffer = ptr.Get();
        const size_t recordingBufferSize = m_Recorder.GetRequiredMemorySizeForRecording();
        ptr.Advance( recordingBufferSize );

        ptr.AlignUp( nn::os::ThreadStackAlignment );
        void* stackBuffer = ptr.Get();
        ptr.Advance( nn::atk::DeviceOutRecorder::RequiredThreadStackSize );

        m_Recorder.Initialize( recordingBuffer, recordingBufferSize, stackBuffer, nn::atk::DeviceOutRecorder::RequiredThreadStackSize );
    }

    {
        const size_t size = m_Stream.GetRequiredMemorySize();
        m_Stream.Initialize( ptr.Get(), size );
        ptr.Advance( size );
    }

    m_IsRecording = false;
}
//  終了します
void SoundRecorder::Finalize() NN_NOEXCEPT
{
    if( m_IsRecording )
    {
        Stop();

        //  録音用の AuxBus が取り除かれるのを待つ
        //  TODO: 取り除かれたかどうかを判断できるようになったとき、適切なコードに置き換える
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 5 * 10 ) );
    }

    m_Recorder.Finalize();

    m_Stream.Close();
}

//  sampleCout だけサンプルを読み取ります。返り値は読み込めたサンプル数です
size_t SoundRecorder::ReadSamples(int16_t* pSampleBuffer, size_t sampleCount) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pSampleBuffer );

    const size_t sampleSize = sizeof(int16_t);
    return m_Stream.Read( pSampleBuffer, sampleCount * sampleSize, nullptr ) / sampleSize;
}
//  1 秒あたりのサンプル数を取得します
uint32_t SoundRecorder::GetSampleCountPerSecond() const NN_NOEXCEPT
{
    return m_Recorder.GetSamplesPerSec();
}
//  録音中であるかどうかを取得します
bool SoundRecorder::IsRecording() const NN_NOEXCEPT
{
    return m_IsRecording;
}

//  録音を開始します
void SoundRecorder::Start() NN_NOEXCEPT
{
    if( !m_IsRecording )
    {
        m_Stream.PrepareRecording();

        nn::atk::DeviceOutRecorder::RecordingOptions options;

        options.SetLeadSilenceTrimmingEnabled( false );
        options.SetWriteBlockPerSamples( 1 );

        m_IsRecording = m_Recorder.Start( m_Stream, options );
        if( !m_IsRecording )
        {
            NN_LOG( "failed to start recording.\n" );
        }
    }
}
//  録音を停止します
void SoundRecorder::Stop() NN_NOEXCEPT
{
    if( m_IsRecording )
    {
        m_Recorder.Stop( true );
        m_IsRecording = false;
    }
}



//  ------------------------------------------------------------
//
//                           Recorder
//
//  ------------------------------------------------------------
uint32_t SoundRecorder::Recorder::OnProcessSamples(int16_t* samples, uint32_t sampleCount) NN_NOEXCEPT
{
    for( uint32_t i = 0; i < sampleCount; i++ )
    {
        samples[i] = ResolveSampleEndian( samples[i] );
    }
    return sampleCount;
}
uint32_t SoundRecorder::Recorder::GetMaxFrameLength() const NN_NOEXCEPT
{
    return 1;
}
uint32_t SoundRecorder::Recorder::GetSamplesPerSec() const NN_NOEXCEPT
{
    return nn::atk::detail::driver::HardwareManager::DefaultRendererSampleRate;
}
uint32_t SoundRecorder::Recorder::GetValidChannels(uint32_t channels) const NN_NOEXCEPT
{
    NN_UNUSED( channels );
    return nn::atk::detail::driver::HardwareManager::GetInstance().GetChannelCount();
}


//  ------------------------------------------------------------
//
//                            Stream
//
//  ------------------------------------------------------------
//  必要なメモリ量を取得します
size_t SoundRecorder::Stream::GetRequiredMemorySize() NN_NOEXCEPT
{
    return RecordingBufferSize;
}
//  初期化します
void SoundRecorder::Stream::Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    NN_ABORT_UNLESS_GREATER_EQUAL( bufferSize, GetRequiredMemorySize() );

    m_Buffer = reinterpret_cast<char*>( buffer );
    m_BufferSize = bufferSize;

    m_WritePosition = 0;
    m_ReadPosition = 0;
    m_ReadableSize = 0;
    m_IsHeaderWritten = false;
}
//  録音を開始する準備をします
void SoundRecorder::Stream::PrepareRecording() NN_NOEXCEPT
{
    //  nn::atk::DeviceOutRecorder が録音を開始するときにヘッダが出力されるため、
    //  m_IsHeaderWritten を false にします。
    m_IsHeaderWritten = false;

    //  録音開始前に溜まっているバッファを空にします。
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_CriticalSection );
    m_WritePosition = 0;
    m_ReadPosition = 0;
    m_ReadableSize = 0;
}
nn::atk::detail::fnd::FndResult SoundRecorder::Stream::Open(const char* filePath, AccessMode openMode) NN_NOEXCEPT
{
    NN_UNUSED( filePath );
    NN_UNUSED( openMode );

    return nn::atk::detail::fnd::FndResult( nn::atk::detail::fnd::FndResultType_True );
}
void SoundRecorder::Stream::Close() NN_NOEXCEPT
{
    m_WritePosition = 0;
    m_ReadPosition = 0;
    m_ReadableSize = 0;
    m_IsHeaderWritten = false;
}
void SoundRecorder::Stream::Flush() NN_NOEXCEPT
{
}
bool SoundRecorder::Stream::IsOpened() const NN_NOEXCEPT
{
    return true;
}
bool SoundRecorder::Stream::CanRead() const NN_NOEXCEPT
{
    return true;
}
bool SoundRecorder::Stream::CanWrite() const NN_NOEXCEPT
{
    return true;
}
bool SoundRecorder::Stream::CanSeek() const NN_NOEXCEPT
{
    return false;
}
size_t SoundRecorder::Stream::GetSize() const NN_NOEXCEPT
{
    return 0;
}
size_t SoundRecorder::Stream::Read(void* buffer, size_t length, nn::atk::detail::fnd::FndResult* result) NN_NOEXCEPT
{
    NN_UNUSED( result );
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_CriticalSection );

    if( m_ReadableSize >= length )
    {
        char* buf = reinterpret_cast<char*>( buffer );

        for( size_t i = 0; i < length; ++i )
        {
            buf[i] = m_Buffer[m_ReadPosition++];

            if( m_ReadPosition == m_BufferSize )
            {
                m_ReadPosition = 0;
            }
        }
        m_ReadableSize -= length;

        return length;
    }
    else
    {
        return 0;
    }
}
size_t SoundRecorder::Stream::Write(const void* buffer, size_t length, nn::atk::detail::fnd::FndResult* result) NN_NOEXCEPT
{
    NN_UNUSED( result );
    NN_ABORT_UNLESS_NOT_NULL( buffer );
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock( m_CriticalSection );

    //  wav ファイル用のヘッダを無視する
    if( !m_IsHeaderWritten )
    {
        m_IsHeaderWritten = true;
        return length;
    }

    if( m_ReadableSize + length <= m_BufferSize )
    {
        const char* buf = reinterpret_cast<const char*>( buffer );
        for( size_t i = 0; i < length; ++i )
        {
            m_Buffer[m_WritePosition++] = buf[i];

            if( m_WritePosition == m_BufferSize )
            {
                m_WritePosition = 0;
            }
        }
        m_ReadableSize += length;
    }
    return length;
}
nn::atk::detail::fnd::FndResult SoundRecorder::Stream::Seek(nn::atk::detail::fnd::position_t offset, SeekOrigin origin) NN_NOEXCEPT
{
    NN_UNUSED( offset );
    NN_UNUSED( origin );
    NN_ABORT( "Not supported.\n" ); //  使う予定のない機能は未実装に
    return nn::atk::detail::fnd::FndResult( nn::atk::detail::fnd::FndResultType_Failed );
}
nn::atk::detail::fnd::position_t SoundRecorder::Stream::GetCurrentPosition() const NN_NOEXCEPT
{
    NN_ABORT( "Not supported.\n" ); //  使う予定のない機能は未実装に
    return 0;
}
void SoundRecorder::Stream::EnableCache(void* buffer, size_t length) NN_NOEXCEPT
{
    NN_UNUSED( buffer );
    NN_UNUSED( length );
    NN_ABORT( "Not supported.\n" ); //  使う予定のない機能は未実装に
}
void SoundRecorder::Stream::DisableCache() NN_NOEXCEPT
{
}
bool SoundRecorder::Stream::IsCacheEnabled() const NN_NOEXCEPT
{
    return false;
}
int SoundRecorder::Stream::GetIoBufferAlignment() const NN_NOEXCEPT
{
    return 1;
}
bool SoundRecorder::Stream::CanSetFsAccessLog() const NN_NOEXCEPT
{
    return false;
}
void* SoundRecorder::Stream::SetFsAccessLog(nn::atk::detail::fnd::FsAccessLog* pFsAccessLog) NN_NOEXCEPT
{
    NN_UNUSED(pFsAccessLog);
    NN_ABORT( "Not supported.\n" ); //  使う予定のない機能は未実装に
    return nullptr;
}

nn::atk::detail::fnd::position_t SoundRecorder::Stream::GetCachePosition() NN_NOEXCEPT
{
    return 0;
}

size_t SoundRecorder::Stream::GetCachedLength() NN_NOEXCEPT
{
    return 0;
}
