﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "gfxutil/GfxContext.h"
#include "SoundParametersPanel.h"
#include "FlightRecorder.h"

namespace
{
    //  パラメータの増加値 (減少値)
    const float ParameterIncrement_Volume          = 0.01f;
    const float ParameterIncrement_Pitch           = 0.01f;
    const int   ParameterIncrement_StartOffset     = 1;
    const float ParameterIncrement_Pan             = 0.01f;
    const float ParameterIncrement_SPan            = 0.01f;
    const float ParameterIncrement_MainSend        = 0.01f;
    const float ParameterIncrement_EffectSendA     = 0.01f;
    const float ParameterIncrement_EffectSendB     = 0.01f;
    const float ParameterIncrement_EffectSendC     = 0.01f;
    const float ParameterIncrement_LpfFrequency    = 0.01f;
    const int   ParameterIncrement_BqfType         = 1;
    const float ParameterIncrement_BqfValue        = 0.01f;
    const float ParameterIncrement_MainOutVolume   = 0.01f;
    const float ParameterIncrement_PanMain         = 0.01f;
    const float ParameterIncrement_SPanMain        = 0.01f;
    const float ParameterIncrement_MainSendMain    = 0.01f;
    const float ParameterIncrement_EffectSendAMain = 0.01f;
    const float ParameterIncrement_EffectSendBMain = 0.01f;
    const float ParameterIncrement_EffectSendCMain = 0.01f;

    //  パラメータの L キーによる加速された増加値 (減少値)
    const float ParameterIncrementBoost_Volume          = 0.1f;
    const float ParameterIncrementBoost_Pitch           = 0.1f;
    const int   ParameterIncrementBoost_StartOffset     = 1000;
    const float ParameterIncrementBoost_Pan             = 0.1f;
    const float ParameterIncrementBoost_SPan            = 0.1f;
    const float ParameterIncrementBoost_MainSend        = 0.1f;
    const float ParameterIncrementBoost_EffectSendA     = 0.1f;
    const float ParameterIncrementBoost_EffectSendB     = 0.1f;
    const float ParameterIncrementBoost_EffectSendC     = 0.1f;
    const float ParameterIncrementBoost_LpfFrequency    = 0.1f;
    const int   ParameterIncrementBoost_BqfType         = 1;
    const float ParameterIncrementBoost_BqfValue        = 0.1f;
    const float ParameterIncrementBoost_MainOutVolume   = 0.1f;
    const float ParameterIncrementBoost_PanMain         = 0.1f;
    const float ParameterIncrementBoost_SPanMain        = 0.1f;
    const float ParameterIncrementBoost_MainSendMain    = 0.1f;
    const float ParameterIncrementBoost_EffectSendAMain = 0.1f;
    const float ParameterIncrementBoost_EffectSendBMain = 0.1f;
    const float ParameterIncrementBoost_EffectSendCMain = 0.1f;
}
namespace
{
    //  パネルのタイトル
    const char* PanelTitle = "SoundParameters";
    //  トグルの表示位置のオフセット
    const float TogglePositionOffsetX = -39.0f;
    //  項目の名前の表示位置のオフセット
    const float ItemPositionOffsetX = 32.0f;
}
namespace
{
    //  BiquadFilter の表示する名前を返します
    const char* GetBiquadFilterName(int type)
    {
        switch( type )
        {
        case nn::atk::BiquadFilterType_Inherit:
            return "INHERIT";
        case nn::atk::BiquadFilterType_None:
            return "NONE";
        case nn::atk::BiquadFilterType_LowPassFilter:
            return "LPF";
        case nn::atk::BiquadFilterType_HighPassFilter:
            return "HPF";
        case nn::atk::BiquadFilterType_BandPassFilter512:
            return "BPF512";
        case nn::atk::BiquadFilterType_BandPassFilter1024:
            return "BPF1024";
        case nn::atk::BiquadFilterType_BandPassFilter2048:
            return "BPF2048";
        default:
            return "Invalid";
        }
    }
}

NN_DEFINE_STATIC_CONSTANT( const int SoundParametersPanel::SoundHandleCount );

//  初期化します
void SoundParametersPanel::Initialize(SoundArchiveContext& soundArchiveContext, float positionX, float positionY, float sizeX, float sizeY) NN_NOEXCEPT
{
    m_Panel.SetTitle( PanelTitle );
    m_Panel.SetPosition( positionX, positionY );
    m_Panel.SetSize( sizeX, sizeY );

    const nn::util::Float2 panelPos = m_Panel.GetClientPositionLeftTop();
    const float itemSizeX = m_Panel.GetClientPositionRightBottom().x - panelPos.x;
    m_ItemList.Initialize( itemSizeX - ItemPositionOffsetX );
    m_ItemList.SetSelectedItemIndex( ParameterType_Volume );

    //  名前の設定
    m_ItemList.GetNameLabel( ParameterType_Volume )         .SetText( "Volume:" );
    m_ItemList.GetNameLabel( ParameterType_Pitch )          .SetText( "Pitch:" );
    m_ItemList.GetNameLabel( ParameterType_StartOffset )    .SetText( "StartOffset:" );
    m_ItemList.GetNameLabel( ParameterType_Pan )            .SetText( "Pan:" );
    m_ItemList.GetNameLabel( ParameterType_SPan )           .SetText( "SPan:" );
    m_ItemList.GetNameLabel( ParameterType_MainSend )       .SetText( "MainSend:" );
    m_ItemList.GetNameLabel( ParameterType_EffectSendA )    .SetText( "EffectSendA:" );
    m_ItemList.GetNameLabel( ParameterType_EffectSendB )    .SetText( "EffectSendB:" );
    m_ItemList.GetNameLabel( ParameterType_EffectSendC )    .SetText( "EffectSendC:" );
    m_ItemList.GetNameLabel( ParameterType_LpfFrequency )   .SetText( "LpfFrequency:" );
    m_ItemList.GetNameLabel( ParameterType_BqfType )        .SetText( "BqfType:" );
    m_ItemList.GetNameLabel( ParameterType_BqfValue )       .SetText( "BqfValue:" );
    m_ItemList.GetNameLabel( ParameterType_MainOutVolume )  .SetText( "MainOutVolume:" );
    m_ItemList.GetNameLabel( ParameterType_PanMain )        .SetText( "Pan(Main):" );
    m_ItemList.GetNameLabel( ParameterType_SPanMain )       .SetText( "SPan(Main):" );
    m_ItemList.GetNameLabel( ParameterType_MainSendMain )   .SetText( "MainSend(Main):" );
    m_ItemList.GetNameLabel( ParameterType_EffectSendAMain ).SetText( "EffectSendA(Main):" );
    m_ItemList.GetNameLabel( ParameterType_EffectSendBMain ).SetText( "EffectSendB(Main):" );
    m_ItemList.GetNameLabel( ParameterType_EffectSendCMain ).SetText( "EffectSendC(Main):" );

    //  パラメータを反映させる
    for( int i = 0; i < SoundHandleCount; i++ )
    {
        WrappedSoundHandle& handle = soundArchiveContext.GetSoundHandle( i );
        handle.SetParameterIsDirty( true );
    }

    SetFocused( false );
}
//  処理します
void SoundParametersPanel::Update(const WrappedSoundHandle& handle) NN_NOEXCEPT
{
    m_ItemList.GetValueLabel( ParameterType_Volume )         .SetText( "%.2f", handle.GetVolume() );
    m_ItemList.GetValueLabel( ParameterType_Pitch )          .SetText( "%.2f", handle.GetPitch() );
    m_ItemList.GetValueLabel( ParameterType_StartOffset )    .SetText( "%d"  , handle.GetStartOffset() );
    m_ItemList.GetValueLabel( ParameterType_Pan )            .SetText( "%.2f", handle.GetPan() );
    m_ItemList.GetValueLabel( ParameterType_SPan )           .SetText( "%.2f", handle.GetSPan() );
    m_ItemList.GetValueLabel( ParameterType_MainSend )       .SetText( "%.2f", handle.GetMainSend() );
    m_ItemList.GetValueLabel( ParameterType_EffectSendA )    .SetText( "%.2f", handle.GetEffectSendA() );
    m_ItemList.GetValueLabel( ParameterType_EffectSendB )    .SetText( "%.2f", handle.GetEffectSendB() );
    m_ItemList.GetValueLabel( ParameterType_EffectSendC )    .SetText( "%.2f", handle.GetEffectSendC() );
    m_ItemList.GetValueLabel( ParameterType_LpfFrequency )   .SetText( "%.2f", handle.GetLpfFrequency() );
    m_ItemList.GetValueLabel( ParameterType_BqfType )        .SetText( "%s"  , GetBiquadFilterName( handle.GetBqfType() ) );
    m_ItemList.GetValueLabel( ParameterType_BqfValue )       .SetText( "%.2f", handle.GetBqfValue() );
    m_ItemList.GetValueLabel( ParameterType_MainOutVolume )  .SetText( "%.2f", handle.GetMainOutVolume() );
    m_ItemList.GetValueLabel( ParameterType_PanMain )        .SetText( "%.2f", handle.GetPanMain() );
    m_ItemList.GetValueLabel( ParameterType_SPanMain )       .SetText( "%.2f", handle.GetSPanMain() );
    m_ItemList.GetValueLabel( ParameterType_MainSendMain )   .SetText( "%.2f", handle.GetMainSendMain() );
    m_ItemList.GetValueLabel( ParameterType_EffectSendAMain ).SetText( "%.2f", handle.GetEffectSendAMain() );
    m_ItemList.GetValueLabel( ParameterType_EffectSendBMain ).SetText( "%.2f", handle.GetEffectSendBMain() );
    m_ItemList.GetValueLabel( ParameterType_EffectSendCMain ).SetText( "%.2f", handle.GetEffectSendCMain() );

    if( handle.GetOutputLine() & nn::atk::OutputLine_Main )
    {
        m_ToggleMainOutVolume.SetText( "[*]" );
    }
    else
    {
        m_ToggleMainOutVolume.SetText( "[ ]" );
    }
}
//  入力による更新を行います
void SoundParametersPanel::UpdateByHid(WrappedSoundHandle& handle, const HidPad& hidPad) NN_NOEXCEPT
{
    m_ItemList.UpdateByHid( hidPad );

    if( m_Panel.IsFocused() )
    {
        UpdateParametersByHid( handle, hidPad );
    }
}
//  描画します
void SoundParametersPanel::Draw(gfxutil::GfxContext& gfxContext) NN_NOEXCEPT
{
    m_Panel.Draw( gfxContext );

    const nn::util::Float2 panelPos = m_Panel.GetClientPositionLeftTop();
    m_ItemList.Draw( gfxContext, panelPos.x + ItemPositionOffsetX, panelPos.y );

    //  トグルの表示
    const nn::util::Float2 pos = m_ItemList.GetNameLabel( ParameterType_MainOutVolume ).GetPosition();
    m_ToggleMainOutVolume.SetPosition( pos.x + TogglePositionOffsetX, pos.y );
    m_ToggleMainOutVolume.Draw( gfxContext );
}

//  フォーカスを設定します
void SoundParametersPanel::SetFocused(bool isFocused) NN_NOEXCEPT
{
    m_Panel.SetFocused( isFocused );
    m_ItemList.SetFocused( isFocused );
}

//  入力によってパラメータを更新します
void SoundParametersPanel::UpdateParametersByHid(WrappedSoundHandle& handle, const HidPad& hidPad) NN_NOEXCEPT
{
    //  左右入力
    const bool isLeftContinue = hidPad.IsContinue( HidPad::Button_Left );
    const bool isRightContinue = hidPad.IsContinue( HidPad::Button_Right );
    if( isLeftContinue || isRightContinue )
    {
        const bool isBoost = hidPad.IsHold( HidPad::Button_L );
        switch( m_ItemList.GetSelectedItemIndex() )
        {
        case ParameterType_Volume:
            handle.SetVolume( ChangeParameterValue( handle.GetVolume(), ParameterIncrement_Volume, ParameterIncrementBoost_Volume, isBoost, isRightContinue, "Prm", "Volume" ) );
            break;
        case ParameterType_Pitch:
            handle.SetPitch( ChangeParameterValue( handle.GetPitch(), ParameterIncrement_Pitch, ParameterIncrementBoost_Pitch, isBoost, isRightContinue, "Prm", "Pitch" ) );
            break;
        case ParameterType_StartOffset:
            handle.SetStartOffset( ChangeParameterValue( handle.GetStartOffset(), ParameterIncrement_StartOffset, ParameterIncrementBoost_StartOffset, isBoost, isRightContinue, "Prm", "StartOffset" ) );
            break;
        case ParameterType_Pan:
            handle.SetPan( ChangeParameterValue( handle.GetPan(), ParameterIncrement_Pan, ParameterIncrementBoost_Pan, isBoost, isRightContinue, "Prm", "Pan" ) );
            break;
        case ParameterType_SPan:
            handle.SetSPan( ChangeParameterValue( handle.GetSPan(), ParameterIncrement_SPan, ParameterIncrementBoost_SPan, isBoost, isRightContinue, "Prm", "SPan" ) );
            break;
        case ParameterType_MainSend:
            handle.SetMainSend( ChangeParameterValue( handle.GetMainSend(), ParameterIncrement_MainSend, ParameterIncrementBoost_MainSend, isBoost, isRightContinue, "Prm", "MainSend" ) );
            break;
        case ParameterType_EffectSendA:
            handle.SetEffectSendA( ChangeParameterValue( handle.GetEffectSendA(), ParameterIncrement_EffectSendA, ParameterIncrementBoost_EffectSendA, isBoost, isRightContinue, "Prm", "EffectSendA" ) );
            break;
        case ParameterType_EffectSendB:
            handle.SetEffectSendB( ChangeParameterValue( handle.GetEffectSendB(), ParameterIncrement_EffectSendB, ParameterIncrementBoost_EffectSendB, isBoost, isRightContinue, "Prm", "EffectSendB" ) );
            break;
        case ParameterType_EffectSendC:
            handle.SetEffectSendC( ChangeParameterValue( handle.GetEffectSendC(), ParameterIncrement_EffectSendC, ParameterIncrementBoost_EffectSendC, isBoost, isRightContinue, "Prm", "EffectSendC" ) );
            break;
        case ParameterType_LpfFrequency:
            handle.SetLpfFrequency( ChangeParameterValue( handle.GetLpfFrequency(), ParameterIncrement_LpfFrequency, ParameterIncrementBoost_LpfFrequency, isBoost, isRightContinue, "Prm", "LpfFrequency" ) );
            break;
        case ParameterType_BqfType:
            handle.SetBqfType( ChangeParameterValue( handle.GetBqfType(), ParameterIncrement_BqfType, ParameterIncrementBoost_BqfType, isBoost, isRightContinue, "Prm", "BqfType" ) );
            break;
        case ParameterType_BqfValue:
            handle.SetBqfValue( ChangeParameterValue( handle.GetBqfValue(), ParameterIncrement_BqfValue, ParameterIncrementBoost_BqfValue, isBoost, isRightContinue, "Prm", "BqfValue" ) );
            break;
        case ParameterType_MainOutVolume:
            handle.SetMainOutVolume( ChangeParameterValue( handle.GetMainOutVolume(), ParameterIncrement_MainOutVolume, ParameterIncrementBoost_MainOutVolume, isBoost, isRightContinue, "Prm", "MainOutVolume" ) );
            break;
        case ParameterType_PanMain:
            handle.SetPanMain( ChangeParameterValue( handle.GetPanMain(), ParameterIncrement_PanMain, ParameterIncrementBoost_PanMain, isBoost, isRightContinue, "Prm", "PanMain" ) );
            break;
        case ParameterType_SPanMain:
            handle.SetSPanMain( ChangeParameterValue( handle.GetSPanMain(), ParameterIncrement_SPanMain, ParameterIncrementBoost_SPanMain, isBoost, isRightContinue, "Prm", "SPanMain" ) );
            break;
        case ParameterType_MainSendMain:
            handle.SetMainSendMain( ChangeParameterValue( handle.GetMainSendMain(), ParameterIncrement_MainSendMain, ParameterIncrementBoost_MainSendMain, isBoost, isRightContinue, "Prm", "MainSendMain" ) );
            break;
        case ParameterType_EffectSendAMain:
            handle.SetEffectSendAMain( ChangeParameterValue( handle.GetEffectSendAMain(), ParameterIncrement_EffectSendAMain, ParameterIncrementBoost_EffectSendAMain, isBoost, isRightContinue, "Prm", "EffectSendAMain" ) );
            break;
        case ParameterType_EffectSendBMain:
            handle.SetEffectSendBMain( ChangeParameterValue( handle.GetEffectSendBMain(), ParameterIncrement_EffectSendBMain, ParameterIncrementBoost_EffectSendBMain, isBoost, isRightContinue, "Prm", "EffectSendBMain" ) );
            break;
        case ParameterType_EffectSendCMain:
            handle.SetEffectSendCMain( ChangeParameterValue( handle.GetEffectSendCMain(), ParameterIncrement_EffectSendCMain, ParameterIncrementBoost_EffectSendCMain, isBoost, isRightContinue, "Prm", "EffectSendCMain" ) );
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    //  トグル切り替え
    if( hidPad.IsTrigger( HidPad::Button_X ) )
    {
        if( m_ItemList.GetSelectedItemIndex() == ParameterType_MainOutVolume )
        {
            uint32_t line = handle.GetOutputLine();
            line ^= nn::atk::OutputLine_Main;
            handle.SetOutputLine( line );
        }
    }
}
