﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/viewer/vfx_PacketProcedure.h>
#include <nn/vfx/viewer/vfx_Viewer.h>
#include <nn/vfx/viewer/vfx_Message.h>
#include <nn/vfx/viewer/vfx_Protocol.h>
#include <nn/vfx/viewer/vfx_ViewerData.h>

#include <nn/vfx/vfx_Misc.h>
#include <nn/vfx/vfx_Resource.h>

#ifndef DLOG
#if 0
#define DLOG( x, ... ) printf( x, ##__VA_ARGS__ )
#else
#define DLOG( x, ... )
#endif
#endif//DLOG

namespace nn {
namespace vfx {
namespace viewer {
namespace detail {

class VisiblityData
{
    NN_DISALLOW_COPY( VisiblityData );
public:
    nn::vfx::viewer::detail::AssetType assetType;
    nn::vfx::viewer::detail::Guid       guid;
    char                                targetName[ 64 ];
    bool                                isVisible;
};

//------------------------------------------------------------------------------
//      パケットを処理
//------------------------------------------------------------------------------
bool PacketProcedure::DoPacket( char* buffer, size_t bufferSize, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    NN_UNUSED( buffer );
    NN_UNUSED( bufferSize );

    bool ret = false;

    nn::vfx::viewer::detail::Message* header = reinterpret_cast< nn::vfx::viewer::detail::Message* >( buffer );

    DLOG( "MessageType = %d\n", header->type );
    DLOG( "BufferSize  = %d\n", header->bufferSize );
    buffer += sizeof( nn::vfx::viewer::detail::Message );

    switch( header->type )
    {
    case nn::vfx::viewer::detail::MessageType_Control:
    {
        // 制御メッセージ.
        HandleMessageControl( buffer );
    }
        break;

    case nn::vfx::viewer::detail::MessageType_SendBinary:
    {
        // バイナリ転送.
        HandleMessageSendBinary( buffer, header->bufferSize, pRegisterTextureSlotCallback, pUserData );
        ret = true;
    }
        break;

    case nn::vfx::viewer::detail::MessageType_Create:
    {
        // インスタンス生成.
        HandleMessageCreate( buffer );
    }
        break;

    case nn::vfx::viewer::detail::MessageType_Modify:
    {
        // リアルタイム編集.
        HandleMessageModify( buffer, pRegisterTextureSlotCallback );
    }
        break;

    case nn::vfx::viewer::detail::MessageType_Delete:
    {
        // インスタンス削除.
        HandleMessageDelete( buffer );
    }
        break;

    case nn::vfx::viewer::detail::MessageType_RemoveBinary:
    {
        // バイナリ削除.
        HandleMessageRemoveBinary( buffer );
    }
        break;
    default: break;
    }

    return ret;
}

//------------------------------------------------------------------------------
//      制御メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageControl( char* pBuffer ) NN_NOEXCEPT
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Control Message.\n" );
    DLOG( "--------------------------------------------\n\n" );

    nn::vfx::viewer::detail::MessageInfo info = ( *reinterpret_cast< nn::vfx::viewer::detail::MessageInfo* >( pBuffer ) );
    NN_UNUSED( info );

    DLOG( "Total Packet Size = %d\n", info.packetSize );
    DLOG( "Number Of Packets = %u\n", info.packetCount );
    DLOG( "Process ID        = %u\n", info.processId );
}


//------------------------------------------------------------------------------
//      バイナリ転送メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageSendBinary( char* pBuffer, size_t bufferSize, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback, void* pUserData ) NN_NOEXCEPT
{
    NN_UNUSED( bufferSize );

    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Send Binary Message.\n" );
    DLOG( "--------------------------------------------\n\n" );

    nn::vfx::viewer::detail::SendBinaryMessage* binMsg = ( nn::vfx::viewer::detail::SendBinaryMessage* )pBuffer;

    DLOG( "Send binary message %d \n", binMsg->assetType );

    switch( binMsg->assetType )
    {
    case nn::vfx::viewer::detail::AssetType_EmitterSet:
    {
        void* pBinaryTop = pBuffer + sizeof( nn::vfx::viewer::detail::SendBinaryMessage );
        char* pPath = reinterpret_cast< char* >( pBinaryTop );

        m_pViewerSystem->LoadEffectData( pPath, binMsg->guid, pRegisterTextureSlotCallback, pUserData );
        m_pViewerSystem->Play();
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Preview:
    {
        void* pBinaryTop = pBuffer + sizeof( nn::vfx::viewer::detail::SendBinaryMessage );
        char* pPath = reinterpret_cast< char* >( pBinaryTop );

        void* pFileData = NULL;
        size_t fileSize = 0;
        if( FileSystem::Load( &pFileData, &fileSize, pPath, m_pHeap ) == false )
        {
            break;
        }

        if( pFileData == NULL )
        {
            break;
        }

        const size_t offset = 96; // FileHeader + BinaryName + BinaryData = 32 + 32 + 32.
        void* pFileBinaryTop = reinterpret_cast< char* >( pFileData ) + offset;

        nn::vfx::detail::BinaryData* pBinary = static_cast< nn::vfx::detail::BinaryData* >( pFileBinaryTop );

        if( pBinary->GetBinaryTag() != VFX_MAKE_TAG( 'P', 'R', 'E', 'V' ) )
        {
            m_pHeap->Free( pFileData );
            break;
        }

        ResEffectPreviewPack* pResEffectPreviewPack = pBinary->GetBinaryDataWithFlip<ResEffectPreviewPack>();

        m_pViewerSystem->SetResEffectPreview( binMsg->guid, pResEffectPreviewPack );

        m_pHeap->Free( pFileData );
    }
        break;

    case nn::vfx::viewer::detail::AssetType_ViewerModel:
    {
        void* pBinaryTop = pBuffer + sizeof( nn::vfx::viewer::detail::SendBinaryMessage );
        char* pPath = reinterpret_cast< char* >( pBinaryTop );

        void* pFileData = NULL;
        size_t fileSize = 0;
        if( FileSystem::Load( &pFileData, &fileSize, pPath, m_pHeap ) == false )
        {
            break;
        }

        if( pFileData == NULL )
        {
            break;
        }

        // 通信システム修正用
        const size_t offset = 96; // FileHeader + BinaryName + BinaryData = 32 + 32 + 32.
        void* pFileBinaryTop = reinterpret_cast< char* >( pFileData ) + offset;

        nn::vfx::detail::BinaryData* pNodeBinary = static_cast< nn::vfx::detail::BinaryData* >( pFileBinaryTop );
        nn::vfx::detail::BinaryData* pModelBinary = pNodeBinary->GetNextData();

        if( pNodeBinary->GetBinaryTag() != VFX_MAKE_TAG( 'M', 'D', 'L', 'P' ) || pModelBinary == NULL )
        {
            m_pHeap->Free( pFileData );
            break;
        }

        // モデルプレビューを作成
        bool isCreated = m_pViewerSystem->CreateModelPreview( binMsg->guid, pModelBinary->GetBinaryData(), pModelBinary->GetBinarySize() );

        // ResPreviewを設定
        if( isCreated )
        {
            ResModelPreviewPack* pResModelPreviewPack = pNodeBinary->GetBinaryDataWithFlip<ResModelPreviewPack>();
            m_pViewerSystem->SetResModelPreview( binMsg->guid, pResModelPreviewPack );
        }

        m_pHeap->Free( pFileData );
        //DLOG( "Preview  Guid : %s\n", binMsg->guid.ToString() );
    }
        break;

    case nn::vfx::viewer::detail::AssetType_ViewerData:
    {
        void* binTop = pBuffer + sizeof( nn::vfx::viewer::detail::SendBinaryMessage );
        char* pPath = reinterpret_cast< char* >( binTop );

        void* pFileData = NULL;
        size_t fileSize = 0;
        if( FileSystem::Load( &pFileData, &fileSize, pPath, m_pHeap ) == false )
        {
            break;
        }

        if( pFileData == NULL )
        {
            break;
        }

        // 通信システム修正用
        const size_t offset = 96; // FileHeader + BinaryName + BinaryData = 32 + 32 + 32.
        void* fileBinTop = reinterpret_cast< char* >( pFileData ) + offset;
        size_t binarySize = fileSize - offset;
        NN_UNUSED( fileBinTop );

        // ビューアデータを保持するためのバッファを確保
        if( m_ViewerTempDataSize < binarySize )
        {
            if( m_pViewerTempData )
            {
                m_pHeap->Free( m_pViewerTempData );
            }
            m_pViewerTempData = m_pHeap->Alloc( binarySize );
            m_ViewerTempDataSize = binarySize;
        }

        // リロードのたびにコピる
        memcpy( m_pViewerTempData, fileBinTop, binarySize );

        // ハンドリング先でフリップされる
        if( m_ViewerDataCallback )
        {
            m_ViewerDataCallback( m_pHeap, m_pViewerDataCallbackObject, m_pViewerTempData, pRegisterTextureSlotCallback );
        }

        m_pHeap->Free( pFileData );
    }
        break;

    case nn::vfx::viewer::detail::AssetType_ViewerMessage:
    {
        void* pBinaryTop = pBuffer + sizeof( nn::vfx::viewer::detail::SendBinaryMessage );
        char* pData = reinterpret_cast< char* >( pBinaryTop );

        if( pData == NULL )
        {
            break;
        }

        // ビューアメッセージ用
        HandleMessageViewerControl( pData );
    }
        break;
    default:break;
    }
}// NOLINT(readability/fn_size)

//------------------------------------------------------------------------------
//      インスタンス生成メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageCreate( char* pBuffer ) NN_NOEXCEPT
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect CREATE Message.\n" );
    DLOG( "--------------------------------------------\n\n" );

    nn::vfx::viewer::detail::CreateMessage* binMsg = ( nn::vfx::viewer::detail::CreateMessage* )pBuffer;

#if 0
    m_pViewerSystem->CreateEffectPreview( binMsg->resGuid, binMsg->previewGuid );
#else
    switch( binMsg->assetType )
    {
    case nn::vfx::viewer::detail::AssetType_EmitterSet:
    {
        /* DO_NOTHING */
        /* このメッセージはエディタから送られません */
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Preview:
    {
        //DLOG( "Resource Guid : %s\n", binMsg->resGuid.ToString() );
        //DLOG( "Preview  Guid : %s\n", binMsg->previewGuid.ToString() );
        m_pViewerSystem->CreateEffectPreview( binMsg->resGuid, binMsg->previewGuid );
    }
        break;
    default:
        break;
    }
#endif
}

//------------------------------------------------------------------------------
//      リアルタイム編集メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageModify( char* pBuffer, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback ) NN_NOEXCEPT
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Modifty Message.\n" );
    DLOG( "--------------------------------------------\n" );

    nn::vfx::viewer::detail::ModifyMessage* msg = ( reinterpret_cast< nn::vfx::viewer::detail::ModifyMessage* >( pBuffer ) );

    DLOG( "AssetID      = %u\n", msg->assetType );
    //  DLOG( "GUID         = %s\n", msg.guid.ToString() );
    DLOG( "Offset       = %u\n", msg->offset );
    DLOG( "OverrideSize = %u\n", msg->overwriteSize );
    DLOG( "WithReset    = %u\n", msg->isRequiredReset );
    DLOG( "--------------------------------------------\n" );

    switch( msg->assetType )
    {
    case nn::vfx::viewer::detail::AssetType_EmitterSet:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect EmitterSetData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        // リソースのguidで編集を行う。
        nn::util::BytePtr ptr( pBuffer );
        ptr.Advance( sizeof( nn::vfx::viewer::detail::ModifyMessage ) );
        uint8_t* pSrc = reinterpret_cast< uint8_t* >( ptr.Get() );
        m_pViewerSystem->UpdateAssetEmitterSet( msg->guid, msg->offset, pSrc, msg->overwriteSize, static_cast< uint8_t >( msg->endian ) );
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Emitter:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect EmitterData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        nn::util::BytePtr ptr( pBuffer );
        ptr.Advance( sizeof( nn::vfx::viewer::detail::ModifyMessage ) );
        uint8_t* pSrc = reinterpret_cast< uint8_t* >( ptr.Get() );

        // リソースのguidとエミッタ名で編集を行う。
        if( msg->emitterName[ 0 ] != 0 )
        {
            const bool withReset = ( msg->isRequiredReset == 1 ) ? true : false;
            m_pViewerSystem->UpdateAssetEmitter(
                msg->guid, msg->emitterName, msg->offset, pSrc, msg->overwriteSize, withReset, static_cast< uint8_t >( msg->endian ) );
        }
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Texture:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect TextureData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        /* 未実装 */
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Primitive:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect PrimitiveData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        /* 未実装 */
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Shader:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect ShaderData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        /* 未実装 */
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Preview:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect PreviewData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        // リソースのguidで編集を行う。
        nn::util::BytePtr ptr( pBuffer );
        ptr.Advance( sizeof( nn::vfx::viewer::detail::ModifyMessage ) );
        uint8_t* pSrc = reinterpret_cast< uint8_t* >( ptr.Get() );
        m_pViewerSystem->UpdateResEffectPreview( msg->guid, msg->offset, pSrc, msg->overwriteSize, static_cast< uint8_t >( msg->endian ) );
    }
        break;

    case nn::vfx::viewer::detail::AssetType_ViewerData:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect ViewerData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        // このクラスで保持しているテンポラリデータに対して上書きを行う
        nn::util::BytePtr ptr( pBuffer );
        ptr.Advance( sizeof( nn::vfx::viewer::detail::ModifyMessage ) );
        uint8_t* pSrc = reinterpret_cast< uint8_t* >( ptr.Get() );
        UpdateViewerData( msg->offset, pSrc, msg->overwriteSize );
        if( m_ViewerDataCallback )
        {
            m_ViewerDataCallback( m_pHeap, m_pViewerDataCallbackObject, m_pViewerTempData, pRegisterTextureSlotCallback );
        }
    }
        break;

    case nn::vfx::viewer::detail::AssetType_ViewerModel:
    {
        DLOG( "--------------------------------------------\n" );
        DLOG( "Detect ViewerModelData!\n" );
        DLOG( "--------------------------------------------\n\n" );

        // リソースのguidで編集を行う。
        nn::util::BytePtr ptr( pBuffer );
        ptr.Advance( sizeof( nn::vfx::viewer::detail::ModifyMessage ) );
        uint8_t* pSrc = reinterpret_cast< uint8_t* >( ptr.Get() );
        m_pViewerSystem->UpdateResModelPreview( msg->guid, msg->offset, pSrc, msg->overwriteSize, static_cast< uint8_t >( msg->endian ) );
    }
        break;
    default: break;
    }
}

//------------------------------------------------------------------------------
//      インスタンス削除メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageDelete( char* pBuffer ) NN_NOEXCEPT
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Delete Message.\n" );
    DLOG( "--------------------------------------------\n" );
    nn::vfx::viewer::detail::DeleteMessage* msg = reinterpret_cast< nn::vfx::viewer::detail::DeleteMessage* >( pBuffer );

    switch( msg->assetType )
    {
    case nn::vfx::viewer::detail::AssetType_EmitterSet:
    {
        /* NOT_IMPLEMENT */
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Preview:
    {
        m_pViewerSystem->UnLoadEffectData( msg->resGuid );
    }
        break;
    default: break;
    }
}

//------------------------------------------------------------------------------
//      バイナリ削除メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageRemoveBinary( char* pBuffer ) NN_NOEXCEPT
{
    DLOG( "--------------------------------------------\n" );
    DLOG( "Detect Remove Binary Message.\n" );
    DLOG( "--------------------------------------------\n" );
    nn::vfx::viewer::detail::RemoveBinaryMessage* msg = reinterpret_cast< nn::vfx::viewer::detail::RemoveBinaryMessage* >( pBuffer );

    switch( msg->assetType )
    {
    case nn::vfx::viewer::detail::AssetType_EmitterSet:
    {
        /* NOT_IMPLEMENT */
    }
        break;

    case nn::vfx::viewer::detail::AssetType_Preview:
    {
        /* NOT_IMPLEMENT */
    }
        break;

    case nn::vfx::viewer::detail::AssetType_ViewerModel:
    {
        m_pViewerSystem->DestroyModelPreview( msg->guid );
    }
        break;
    default: break;
    }
}

//------------------------------------------------------------------------------
//      ビューアの再生制御メッセージを処理します.
//------------------------------------------------------------------------------
void PacketProcedure::HandleMessageViewerControl( char* pBuffer ) NN_NOEXCEPT
{
    void* pBinaryTop = pBuffer;
    uint32_t* pMessage = reinterpret_cast< uint32_t* >( pBinaryTop );
    uint32_t type = *pMessage;

    switch( type )
    {

    case nn::vfx::viewer::detail::ViewerMessageType_Config:
    {
        pMessage++;
        // Config Dataを設定する
        if( m_ViewerMessageCallback )
        {
            m_ViewerMessageCallback( m_pHeap, m_pViewerMessageCallbackObject, static_cast< void* >( pMessage ) );
        }
    }
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_ResetFrame:
        m_pViewerSystem->Play();
        m_pViewerSystem->ResetTime( false );
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_Play:
        m_pViewerSystem->Play();
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_Pause:
        if( m_pViewerSystem->IsPause() )
        {
            m_pViewerSystem->Play();
        }
        else
        {
            m_pViewerSystem->Pause();
        }
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_Stop:
        m_pViewerSystem->Stop();
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_SetRange:
    {
        pMessage++;
        int startFrame = *pMessage;
        pMessage++;
        int endFrame = *pMessage;
        pMessage++;
        int isLoop = *pMessage;

        m_pViewerSystem->SetFrameRange( static_cast< float >( startFrame ), static_cast< float >( endFrame ), isLoop > 0 ? 1 : 0 );
    }
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_StepFrame:
    {
        if( !m_pViewerSystem->IsPause() )
        {
            m_pViewerSystem->Pause();
        }

        m_pViewerSystem->StepFrame();
    }
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_ForceFade:
    {
        m_pViewerSystem->ForceFade();
    }
        break;

    case nn::vfx::viewer::detail::ViewerMessageType_Visibility:
    {
        pMessage++;

        VisiblityData* pData = reinterpret_cast< VisiblityData* >( pMessage );

        switch( pData->assetType )
        {

            // エミッタセットの表示/非表示
        case nn::vfx::viewer::detail::AssetType_EmitterSet:
        {
            m_pViewerSystem->SetEmitterSetVisibility( pData->guid, pData->isVisible );
        }
            break;

            // エミッタの表示/非表示
        case nn::vfx::viewer::detail::AssetType_Emitter:
        {
            m_pViewerSystem->SetEmitterVisibility( pData->guid, pData->targetName, pData->isVisible );
        }
            break;

            // プレビューの表示/非表示
        case nn::vfx::viewer::detail::AssetType_Preview:
        case nn::vfx::viewer::detail::AssetType_ViewerModel:
        {
            m_pViewerSystem->SetPreviewVisibility( pData->guid, pData->isVisible );
        }
            break;

        case nn::vfx::viewer::detail::AssetType_Texture:
            break;

        case nn::vfx::viewer::detail::AssetType_Primitive:
            break;

        case nn::vfx::viewer::detail::AssetType_Shader:
            break;

        case nn::vfx::viewer::detail::AssetType_ViewerData:
            break;

        case nn::vfx::viewer::detail::AssetType_ViewerMessage:
            break;
        default:
            break;
        }
    }
        break;

        // 描画順を取得
    case nn::vfx::viewer::detail::ViewerMessageType_DrawingOrder:
    {
        pMessage++;

        struct OrderData
        {
            uint8_t groupId;
            uint8_t order;
        };

        OrderData* pData = reinterpret_cast< OrderData* >( pMessage );

        // pDataが描画順データです。
        NN_UNUSED( pData );
#if 0
        for( int i = 0; i < 64; ++i )
        {
            printf( "%2d: %d, %d\n", i, pData[ i ].groupId, pData[ i ].order );
        }
#endif
    }
        break;

        // Captureをします。
    case nn::vfx::viewer::detail::ViewerMessageType_Capture:
    {
        pMessage++;
        size_t filePathLength = static_cast<size_t>( *pMessage );
        pMessage++;
        char* filePath = reinterpret_cast< char* >( pMessage );

        if ( m_ViewerCaptureMessageCallback )
        {
            m_ViewerCaptureMessageCallback( m_pViewerCaptureMessageCallbackObject, filePath, filePathLength );
        }
    }
        break;
    default: break;
    }
}// NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  ビューアデータを更新します。
//---------------------------------------------------------------------------
void PacketProcedure::UpdateViewerData( size_t offset,
    void*          pOverwriteData,
    size_t         overwriteDataSize ) NN_NOEXCEPT
{
    if( !m_pViewerTempData )
    {
        return;
    }

    // テンポラリデータからResCameraを抽出
    nn::vfx::detail::BinaryData* pNodeBinary = static_cast< nn::vfx::detail::BinaryData* >( m_pViewerTempData );
    ResCamera* pCamera = pNodeBinary->GetBinaryDataWithFlip<ResCamera>();

    // リアルタイム編集ですよフラグを立てる
    pCamera->isRealtimeEditing = 1;

#if defined( VFX_BI_ENDIAN )
    // いったんBEに戻して上書き準備
    if( pNodeBinary->endian != SystemEndianType )
    {
        pCamera->FlipEndian();
    }
#endif

    // 生のまま上書き
    nn::util::BytePtr ptr( pCamera );
    ptr.Advance( offset );
    uint8_t* pDst = reinterpret_cast< uint8_t* >( ptr.Get() );
    memcpy( pDst, pOverwriteData, overwriteDataSize );

#if defined( VFX_BI_ENDIAN )
    // 上書き後にLEに戻す
    if( pNodeBinary->endian != SystemEndianType )
    {
        pCamera->FlipEndian();
    }
#endif

    return;
}

} // namespace detail
} // namespace viewer
} // namespace vfx
} // namespace nn
