﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Primitive.h>
#include <nn/vfx/vfx_EndianUtil.h>


namespace nn     {
namespace vfx    {


//------------------------------------------------------------------------------
//  ResPrimitive
//------------------------------------------------------------------------------
struct ResVertexElementInfo
{
    int     Count;          //!< 頂点数.
    int     Element;        //!< 頂点あたりの構成要素数.

    void FlipEndian() NN_NOEXCEPT
    {
        detail::EndianUtil::Flip( &Count );
        detail::EndianUtil::Flip( &Element );
    }
};

struct ResPrimitive
{
    uint64_t             UniqueId;          //!< 識別用ID.
    ResVertexElementInfo Position;          //!< 位置座標.
    ResVertexElementInfo Normal;            //!< 法線ベクトル.
    ResVertexElementInfo Tangent;           //!< 接線ベクトル.
    ResVertexElementInfo Color;             //!< 頂点カラー.
    ResVertexElementInfo TexCoord0;         //!< テクスチャ座標0
    ResVertexElementInfo TexCoord1;         //!< テクスチャ座標1
    int             IndexCount;             //!< 頂点インデックス数.
    uint32_t        OffsetPosition;         //!< 座標配列オフセット
    uint32_t        OffsetNormal;           //!< 法線配列オフセット
    uint32_t        OffsetTangent;          //!< 接線配列オフセット
    uint32_t        OffsetColor;            //!< カラー配列オフセット
    uint32_t        OffsetTexCoord;         //!< UV配列オフセット
    uint32_t        OffsetIndex;            //!< インデックス配列オフセット

    uint8_t* GetPositionArray()
    {
        return CalcOffset(OffsetPosition);
    }

    uint8_t* GetNormalArray()
    {
        return CalcOffset(OffsetNormal);
    }

    uint8_t* GetTangentArray()
    {
        return CalcOffset(OffsetTangent);
    }

    uint8_t* GetColorArray()
    {
        return CalcOffset(OffsetColor);
    }

    uint8_t* GetTexCoordArray()
    {
        return CalcOffset(OffsetTexCoord);
    }

    uint8_t* GetIndexArray()
    {
        return CalcOffset(OffsetIndex);
    }

    void FlipEndian() NN_NOEXCEPT
    {
#if defined( VFX_ENDIAN_LITTLE )
        Position.FlipEndian();
        Normal.FlipEndian();
        Tangent.FlipEndian();
        Color.FlipEndian();
        TexCoord0.FlipEndian();
        TexCoord1.FlipEndian();
        detail::EndianUtil::Flip( &UniqueId );
        detail::EndianUtil::Flip( &IndexCount );
        detail::EndianUtil::Flip( &OffsetPosition );
        detail::EndianUtil::Flip( &OffsetNormal );
        detail::EndianUtil::Flip( &OffsetTangent );
        detail::EndianUtil::Flip( &OffsetColor );
        detail::EndianUtil::Flip( &OffsetTexCoord );
        detail::EndianUtil::Flip( &OffsetIndex );
        FlipVertexArray( GetPositionArray() );
        FlipVertexArray( GetNormalArray() );
        FlipVertexArray( GetTangentArray() );
        FlipVertexArray( GetColorArray() );
        FlipVertexArray( GetTexCoordArray() );
        FlipIndexArray();
#endif
    }

private:
    uint8_t* CalcOffset(uint32_t offset)
    {
        return offset == 0 ? NULL : reinterpret_cast<uint8_t*>(this) + offset;
    }

#if defined( VFX_ENDIAN_LITTLE )
    void FlipVertexArray(uint8_t* vertexArray)
    {
        if (vertexArray == NULL)
        {
            return;
        }

        nn::util::Float4* vec4Array = reinterpret_cast<nn::util::Float4*>(vertexArray);
        detail::EndianUtil::FlipArray(Position.Count, vec4Array);
    }

    void FlipIndexArray()
    {
        uint32_t* indexArray = reinterpret_cast<uint32_t*>(GetIndexArray());
        if (indexArray == NULL)
        {
            return;
        }

        detail::EndianUtil::FlipArray<uint32_t>(IndexCount, indexArray);
    }
#endif
};


//---------------------------------------------------------------------------
//  引数で指定されるプリミティブデータが必要とする頂点バッファサイズを計算する。
//---------------------------------------------------------------------------
size_t Primitive::CalculateBufferSize( detail::BinaryData* primData ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( primData );
    ResPrimitive* pResPrimitive = primData->GetBinaryDataWithFlip<ResPrimitive>();

    int elementCount = 0;
    if ( pResPrimitive->GetPositionArray() ) elementCount++;
    if ( pResPrimitive->GetNormalArray() ) elementCount++;
    if ( pResPrimitive->GetTangentArray() ) elementCount++;
    if ( pResPrimitive->GetColorArray() ) elementCount++;
    if ( pResPrimitive->GetTexCoordArray() ) elementCount++;

    size_t vertexBufferSize = sizeof( nn::util::Float4 ) * pResPrimitive->Position.Count * elementCount;
    size_t indexBufferSize  = sizeof( uint32_t ) * pResPrimitive->IndexCount;

    return ( vertexBufferSize + indexBufferSize );
}


//---------------------------------------------------------------------------
//  初期化処理。
//---------------------------------------------------------------------------
bool Primitive::Initialize( detail::ConstantBuffer* pBuffer, detail::BinaryData* primData ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pBuffer );
    NN_SDK_ASSERT_NOT_NULL( primData );

    ResPrimitive* pResPrimitive = primData->GetBinaryDataWithFlip<ResPrimitive>();

    // プリミティブを初期化
    bool ret = Initialize(
        pBuffer,
        pResPrimitive->UniqueId,
        pResPrimitive->IndexCount,
        pResPrimitive->Position.Count,
        pResPrimitive->GetPositionArray(),
        pResPrimitive->GetNormalArray(),
        pResPrimitive->GetTangentArray(),
        pResPrimitive->GetColorArray(),
        pResPrimitive->GetTexCoordArray(),
        pResPrimitive->GetIndexArray() );
    return ret;
}

//---------------------------------------------------------------------------
//  プリミティブ初期化処理
//---------------------------------------------------------------------------
bool Primitive::Initialize(
    detail::ConstantBuffer* pBuffer,
    uint64_t guid, int indexCount,
    int arrayCount,
    void* pPosSrc,
    void* pNorSrc,
    void* pTanSrc,
    void* pColSrc,
    void* pTexSrc,
    void* pIndexSrc ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pBuffer );

    m_ElementCount  = 0;
    m_IndexCount    = indexCount;
    m_Guid          = guid;
    m_IsInitialized = false;
    m_ArrayCount    = arrayCount;

    if( pPosSrc )
    {
        m_Offset[ PrimitiveDataElement_Position ] = 0;
        m_ElementCount++;
    }
    else
    {
        m_Offset[ PrimitiveDataElement_Position ] = 0;
    }
    if ( pNorSrc )
    {
        m_Offset[ PrimitiveDataElement_Normal ] = m_ElementCount * sizeof( nn::util::Float4 );
        m_ElementCount++;
    }
    else
    {
        m_Offset[ PrimitiveDataElement_Normal ] = 0;
    }
    if ( pTanSrc )
    {
        m_Offset[ PrimitiveDataElement_Tangent ] = m_ElementCount * sizeof( nn::util::Float4 );
        m_ElementCount++;
    }
    else
    {
        m_Offset[ PrimitiveDataElement_Tangent ] = 0;
    }
    if( pColSrc )
    {
        m_Offset[ PrimitiveDataElement_VertexColor ] = m_ElementCount * sizeof( nn::util::Float4 );
        m_ElementCount++;
    }
    else
    {
        m_Offset[ PrimitiveDataElement_VertexColor ] = 0;
    }
    if( pTexSrc )
    {
        m_Offset[ PrimitiveDataElement_TextureCoordinate ] = m_ElementCount * sizeof( nn::util::Float4 );
        m_ElementCount++;
    }
    else
    {
        m_Offset[ PrimitiveDataElement_TextureCoordinate ] = 0;
    }

    m_Stride            = sizeof( nn::util::Float4 ) * m_ElementCount;
    m_VertexBufferSize  = arrayCount * sizeof( nn::util::Float4 ) * m_ElementCount;
    m_IndexBufferSize   = sizeof( uint32_t ) * m_IndexCount;

    ptrdiff_t offset = pBuffer->GetCurrentOffset();
    nn::util::Float4* pVertexBuffer = static_cast< nn::util::Float4* >( pBuffer->Cut( m_VertexBufferSize + m_IndexBufferSize ) );
    if( pVertexBuffer == NULL )
    {
        // メモリ領域のカットに失敗
        return false;
    }

    void* top = pVertexBuffer;

    nn::util::Float4* pSrcPos4 = static_cast< nn::util::Float4* >( pPosSrc );
    nn::util::Float4* pSrcCol4 = static_cast< nn::util::Float4* >( pColSrc );
    nn::util::Float4* pSrcTex4 = static_cast< nn::util::Float4* >( pTexSrc );
    nn::util::Float4* pSrcNor4 = static_cast< nn::util::Float4* >( pNorSrc );
    nn::util::Float4* pSrcTan4 = static_cast< nn::util::Float4* >( pTanSrc );

    m_pSrcPosition4 = pSrcPos4;
    m_pSrcNormal4   = pSrcNor4;

    // 頂点配列生成
    for( int i = 0; i < arrayCount; i++ )
    {
        if( pPosSrc )
        {
            pVertexBuffer->x = pSrcPos4->x;
            pVertexBuffer->y = pSrcPos4->y;
            pVertexBuffer->z = pSrcPos4->z;
            pVertexBuffer->w = pSrcPos4->w;
            pSrcPos4++;
            pVertexBuffer++;
        }

        if( pNorSrc )
        {
            pVertexBuffer->x = pSrcNor4->x;
            pVertexBuffer->y = pSrcNor4->y;
            pVertexBuffer->z = pSrcNor4->z;
            pVertexBuffer->w = pSrcNor4->w;
            pSrcNor4++;
            pVertexBuffer++;
        }

        if( pTanSrc )
        {
            pVertexBuffer->x = pSrcTan4->x;
            pVertexBuffer->y = pSrcTan4->y;
            pVertexBuffer->z = pSrcTan4->z;
            pVertexBuffer->w = pSrcTan4->w;
            pSrcTan4++;
            pVertexBuffer++;
        }

        if( pColSrc )
        {
            pVertexBuffer->x = pSrcCol4->x;
            pVertexBuffer->y = pSrcCol4->y;
            pVertexBuffer->z = pSrcCol4->z;
            pVertexBuffer->w = pSrcCol4->w;
            pSrcCol4++;
            pVertexBuffer++;
        }

        if( pTexSrc )
        {
            pVertexBuffer->x = pSrcTex4->x;
            pVertexBuffer->y = pSrcTex4->y;
            pVertexBuffer->z = pSrcTex4->z;
            pVertexBuffer->w = pSrcTex4->w;
            pSrcTex4++;
            pVertexBuffer++;
        }
    }

    // インデックス
    detail::MemUtil::Copy( pVertexBuffer, pIndexSrc, m_IndexBufferSize );
    m_pIndexBuffer = pIndexSrc;

    pBuffer->GetGpuAddress( &m_VertexBufferAddress );
    pBuffer->GetGpuAddress( &m_IndexBufferAddress );

    m_VertexBufferAddress.Offset( offset );
    m_IndexBufferAddress.Offset( offset + nn::util::BytePtr( top ).Distance( pVertexBuffer ) );

    m_IsInitialized = true;
    return true;
}   // NOLINT(readability/fn_size);

//---------------------------------------------------------------------------
//  プリミティブ初期化処理( g3d使用プリミティブ版 )
//---------------------------------------------------------------------------
bool G3dPrimitive::Initialize( nn::g3d::ResModel* resModel, detail::ResG3dPrimitiveTableRecord* pPrimitiveTableRecord ) NN_NOEXCEPT
{
    m_pResModel = resModel;
    m_pResVertex = m_pResModel->GetVertex( 0 );
    NN_SDK_ASSERT_NOT_NULL( m_pResVertex );
    m_pResShape = m_pResModel->GetShape( 0 );
    NN_SDK_ASSERT_NOT_NULL( m_pResShape );
    m_pResMesh = m_pResShape->GetMesh( 0 );
    NN_SDK_ASSERT_NOT_NULL( m_pResMesh );

    const char* figureAttribNameG3d[] = { "_p0", "_n0", "_t0", "_c0", "_u0" };
    const int figureAttribNameG3dCount = sizeof( figureAttribNameG3d ) / sizeof( char* );

    // 位置情報は必ずある想定
    nn::g3d::ResVertexAttr* pDefaultResVertexAttr = m_pResVertex->FindVertexAttr( figureAttribNameG3d[ 0 ] );
    NN_SDK_ASSERT_NOT_NULL( pDefaultResVertexAttr );

    // フォーマット、オフセット初期化
    for( int i = 0; i < figureAttribNameG3dCount; i++ )
    {
        nn::g3d::ResVertexAttr* pResVertexAttr = m_pResVertex->FindVertexAttr( figureAttribNameG3d[ i ] );
        if ( pResVertexAttr == nullptr &&
             i == PrimitiveDataElement_TextureCoordinate &&
             pPrimitiveTableRecord->uv0Index != InvalidValueId_G3dAttributeIndex )
        {
            const char* pAttrName = m_pResVertex->GetVertexAttrName( pPrimitiveTableRecord->uv0Index );
            if ( pAttrName )
            {
                pResVertexAttr = m_pResVertex->FindVertexAttr( pAttrName );
            }
        }

        if( pResVertexAttr )
        {
            if( i == PrimitiveDataElement_TextureCoordinate )
            {
                nn::gfx::AttributeFormat format = pResVertexAttr->GetFormat();

                nn::g3d::ResVertexAttr* pU1ResVertexAttr = m_pResVertex->FindVertexAttr( "_u1" );
                if ( pU1ResVertexAttr == nullptr &&
                     pPrimitiveTableRecord->uv1Index != InvalidValueId_G3dAttributeIndex )
                {
                    const char* pAttrName = m_pResVertex->GetVertexAttrName( pPrimitiveTableRecord->uv1Index );
                    if ( pAttrName )
                    {
                        pU1ResVertexAttr = m_pResVertex->FindVertexAttr( pAttrName );
                    }
                }

                if( pU1ResVertexAttr )
                {
                    if ( format != pU1ResVertexAttr->GetFormat() )
                    {
                        nn::vfx::detail::OutputWarning( "%s.fmdb : A format of _u1 is different from _uo of the particle shape.\n", m_pResModel->GetName() );
                    }
                    else
                    {
                        if ( format == nn::gfx::AttributeFormat_8_8_Unorm )
                        {
                            format = nn::gfx::AttributeFormat_8_8_8_8_Unorm;
                        }
                        if ( format == nn::gfx::AttributeFormat_16_16_Unorm )
                        {
                            format = nn::gfx::AttributeFormat_16_16_16_16_Unorm;
                        }
                        if ( format == nn::gfx::AttributeFormat_16_16_Float )
                        {
                            format = nn::gfx::AttributeFormat_16_16_16_16_Float;
                        }
                        if ( format == nn::gfx::AttributeFormat_32_32_Float )
                        {
                            format = nn::gfx::AttributeFormat_32_32_32_32_Float;
                        }
                        if ( format == nn::gfx::AttributeFormat_8_8_Snorm )
                        {
                            format = nn::gfx::AttributeFormat_8_8_8_8_Snorm;
                        }
                        if ( format == nn::gfx::AttributeFormat_16_16_Snorm )
                        {
                            format = nn::gfx::AttributeFormat_16_16_16_16_Snorm;
                        }
                    }
                }
                m_AttrFormat[ i ] = format;
            }
            else
            {
                m_AttrFormat[ i ]  = pResVertexAttr->GetFormat();
            }
            m_BufferIndex[ i ] = pResVertexAttr->GetBufferIndex();
            m_Offset[ i ] = pResVertexAttr->GetOffset();
        }
        else
        {
            m_BufferIndex[ i ] = pDefaultResVertexAttr->GetBufferIndex();
            m_AttrFormat[ i ] = pDefaultResVertexAttr->GetFormat();
            m_Offset[ i ] = pDefaultResVertexAttr->GetOffset();
        }
    }


    m_pResMesh->GetIndexBuffer()->GetGpuAddress( &m_IndexBufferAddress );

    return true;
}   // NOLINT(readability/fn_size);

} // namespace vfx
} // namespace nn
