﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_EmitterCalc.h>
#include <nn/vfx/vfx_FieldRandom.h>
#include <nn/vfx/vfx_MemUtil.h>

namespace nn {
namespace vfx {
namespace detail {

void CalculateGpuNoise( nn::util::Vector3fType* pOutPos, Emitter* pEmitter, const ParticleProperty* pParticleProperty, int particleIndex, const nn::util::Vector3fType& randomVel, float time ) NN_NOEXCEPT
{
    // TODO : なるべく : CPUでも最適な状態で動作するように

    // GPUノイズ: 積算可能な疑似乱数を用いたフィールド
    ResFieldRandom* pGpuNoiseData = pEmitter->GetEmitterResource()->m_pFieldGpuNoiseData;

    const float frameRate = pEmitter->GetFrameRate();
    const float invRate = 1.0f - frameRate;
    const float ubAirRegist = pEmitter->GetEmitterResource()->m_ResEmitterStaticConstantBuffer->airRegist;
    const float airRegist = ubAirRegist + ( 1.0f - ubAirRegist ) * invRate;
    const float RandCycle = static_cast< float >( pGpuNoiseData->blank );

    float velTime = time;
    if( pGpuNoiseData->enableAirRegist )
    {
        velTime = ( airRegist == 1.0 ) ? time : ( 1 - ::std::powf( airRegist, time ) ) / ( 1 - airRegist );
    }

    const float speed = ( pGpuNoiseData->unifiedPhaseSpeed / 100.0f ) * RandCycle;
    const float randWidth = ( pGpuNoiseData->unifiedPhaseDistribution / 100.0f ) * RandCycle;

    nn::util::Vector4fType seed = NN_UTIL_VECTOR_4F_INITIALIZER( 0, 0, 0, 0 );
    nn::util::Vector3fType randomValue = NN_UTIL_VECTOR_3F_INITIALIZER(  0, 0, 0 );
    float deltaSpeed = 0;
    const float deltaVelTime = frameRate * 1.0f;

    if( pGpuNoiseData->enableUnifiedPhase )
    {
        // 位相を統一
        deltaSpeed = frameRate * speed;

        static const nn::util::Vector4fType Seed = NN_UTIL_VECTOR_4F_INITIALIZER( 0.0f, 0.31415f, 0.92653f, 0.58979f );
        seed = Seed;
        nn::util::VectorMultiply( &seed, seed, RandCycle );

        const float eTime = pEmitter->GetTime() * speed;
        const float temp = eTime - randWidth;
        const float temp2 = randWidth * 2;
        nn::util::VectorSetX( &seed, nn::util::VectorGetX( seed ) + temp + ( temp2 * pParticleProperty->random[particleIndex].x ) );
        nn::util::VectorSetY( &seed, nn::util::VectorGetY( seed ) + temp + ( temp2 * pParticleProperty->random[particleIndex].y ) );
        nn::util::VectorSetZ( &seed, nn::util::VectorGetZ( seed ) + temp + ( temp2 * pParticleProperty->random[particleIndex].z ) );
        nn::util::VectorSetW( &seed, nn::util::VectorGetW( seed ) + temp + ( temp2 * pParticleProperty->random[particleIndex].w ) );

        if( pGpuNoiseData->enableDetailedOption )
        {
            const float interval = detail::PiX2 / static_cast< float >( pGpuNoiseData->blank );               // 基本周期
            nn::util::VectorSetX( &randomValue, GetRand( nn::util::VectorGetX( seed ), velTime, deltaSpeed, deltaVelTime, interval, pGpuNoiseData ) );
            nn::util::VectorSetY( &randomValue, GetRand( nn::util::VectorGetY( seed ), velTime, deltaSpeed, deltaVelTime, interval, pGpuNoiseData ) );
            nn::util::VectorSetZ( &randomValue, GetRand( nn::util::VectorGetZ( seed ), velTime, deltaSpeed, deltaVelTime, interval, pGpuNoiseData ) );
        }
        else
        {
            const float interval = detail::PiX2 / static_cast< float >( pGpuNoiseData->blank );
            nn::util::VectorSetX( &randomValue, GetRandDefault( nn::util::VectorGetX( seed ), velTime, deltaSpeed, deltaVelTime, interval ) );
            nn::util::VectorSetY( &randomValue, GetRandDefault( nn::util::VectorGetY( seed ), velTime, deltaSpeed, deltaVelTime, interval ) );
            nn::util::VectorSetZ( &randomValue, GetRandDefault( nn::util::VectorGetZ( seed ), velTime, deltaSpeed, deltaVelTime, interval ) );
        }
    }
    else
    {
        // 位相の統一なし
        nn::util::VectorSet( &seed,
            pParticleProperty->random[particleIndex].x * RandCycle,
            pParticleProperty->random[particleIndex].y * RandCycle,
            pParticleProperty->random[particleIndex].z * RandCycle,
            pParticleProperty->random[particleIndex].w * RandCycle
            );

        if( pGpuNoiseData->enableDetailedOption )
        {
            const float interval = detail::PiX2 / static_cast< float >( pGpuNoiseData->blank );               // 基本周期
            nn::util::VectorSetX( &randomValue, GetRandWithNoPhaseShift( nn::util::VectorGetX( seed ), velTime, deltaVelTime, interval, pGpuNoiseData ) );
            nn::util::VectorSetY( &randomValue, GetRandWithNoPhaseShift( nn::util::VectorGetY( seed ), velTime, deltaVelTime, interval, pGpuNoiseData ) );
            nn::util::VectorSetZ( &randomValue, GetRandWithNoPhaseShift( nn::util::VectorGetZ( seed ), velTime, deltaVelTime, interval, pGpuNoiseData ) );
        }
        else
        {

            const float interval = detail::PiX2 / static_cast< float >( pGpuNoiseData->blank );
            nn::util::VectorSetX( &randomValue, GetRandWithNoPhaseShiftDefault( nn::util::VectorGetX( seed ), velTime, deltaVelTime, interval ) );
            nn::util::VectorSetY( &randomValue, GetRandWithNoPhaseShiftDefault( nn::util::VectorGetY( seed ), velTime, deltaVelTime, interval ) );
            nn::util::VectorSetZ( &randomValue, GetRandWithNoPhaseShiftDefault( nn::util::VectorGetZ( seed ), velTime, deltaVelTime, interval ) );
        }
    }

    nn::util::VectorSetX( pOutPos, nn::util::VectorGetX( *pOutPos ) + nn::util::VectorGetX( randomValue ) * nn::util::VectorGetX( randomVel ) );
    nn::util::VectorSetY( pOutPos, nn::util::VectorGetY( *pOutPos ) + nn::util::VectorGetY( randomValue ) * nn::util::VectorGetY( randomVel ) );
    nn::util::VectorSetZ( pOutPos, nn::util::VectorGetZ( *pOutPos ) + nn::util::VectorGetZ( randomValue ) * nn::util::VectorGetZ( randomVel ) );

    return;
}

} // namespace detail
} // namespace vfx
} // namespace nn
