﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_Callback.h>
#include <nn/vfx/vfx_SuperStripe.h>

namespace nn {
namespace vfx {

//---------------------------------------------------
// 引数ビューマトリクスからエミッタビルボード用マトリクスを生成し設定します。
//---------------------------------------------------
void EmitterMatrixSetArg::SetEmitterBillboardMatrix( const nn::util::Matrix4x3fType& viewMatrix ) NN_NOEXCEPT
{
    nn::util::Matrix4x3fType invViewMatrix;
    nn::util::MatrixInverse( &invViewMatrix, viewMatrix );

    const nn::util::Matrix4x3fType& esetRt  = pEmitter->GetEmitterSet()->GetMatrixRt();
    const nn::util::Matrix4x3fType& esetSrt = pEmitter->GetEmitterSet()->GetMatrixSrt();

    nn::util::Vector3fType pos;
    nn::util::MatrixGetAxisW( &pos, pEmitter->GetMatrixRt() );
    nn::util::MatrixSetAxisW( &invViewMatrix, pos );

    float sx = 1.f, sy = 1.f, sz = 1.f;

    nn::util::Vector3fType srtX, srtY, srtZ;
    nn::util::MatrixGetAxisX( &srtX, esetSrt );
    nn::util::MatrixGetAxisY( &srtY, esetSrt );
    nn::util::MatrixGetAxisZ( &srtZ, esetSrt );

    nn::util::Vector3fType rtX, rtY, rtZ;
    nn::util::MatrixGetAxisX( &rtX, esetRt );
    nn::util::MatrixGetAxisY( &rtY, esetRt );
    nn::util::MatrixGetAxisZ( &rtZ, esetRt );

    if ( nn::util::VectorGetX( rtX ) != 0.f )
    {
        sx = nn::util::VectorGetX( srtX ) / nn::util::VectorGetX( rtX );
    }

    if ( nn::util::VectorGetY( rtY ) != 0.f )
    {
        sy = nn::util::VectorGetY( srtY ) / nn::util::VectorGetY( rtY );
    }

    if ( nn::util::VectorGetZ( rtZ ) != 0.f )
    {
        sz = nn::util::VectorGetZ( srtZ ) / nn::util::VectorGetZ( rtZ );
    }

    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( sx, sy, sz );

    nn::util::Matrix4x3fType scaleMatrix;
    nn::util::MatrixIdentity( &scaleMatrix );
    nn::util::MatrixSetScale( &scaleMatrix, scale );

    nn::util::Matrix4x3fType invEsetRt;
    nn::util::MatrixInverse( &invEsetRt, esetRt );
    nn::util::Matrix4x3fType invEsetSrt;
    nn::util::MatrixInverse( &invEsetSrt, esetSrt );

    {
        nn::util::Matrix4x3fType temp;
        nn::util::MatrixMultiply( &temp, pEmitter->GetMatrixRt(), invEsetRt );
        nn::util::MatrixMultiply( &temp, temp, invViewMatrix );
        pEmitter->SetMatrixRt( temp );
    }
    {
        nn::util::Matrix4x3fType temp;
        nn::util::MatrixMultiply( &temp, pEmitter->GetMatrixSrt(), invEsetSrt );
        nn::util::MatrixMultiply( &temp, temp, scaleMatrix );
        nn::util::MatrixMultiply( &temp, temp, invViewMatrix );
        pEmitter->SetMatrixSrt( temp );
    }
}

//---------------------------------------------------
// 引数ビューマトリクスからエミッタY軸ビルボード用マトリクスを生成し設定します。
//---------------------------------------------------
void EmitterMatrixSetArg::SetEmitterYAxisBillboardMatrix( const nn::util::Matrix4x3fType& viewMatrix ) NN_NOEXCEPT
{
    const nn::util::Matrix4x3fType& esetRt  = pEmitter->GetEmitterSet()->GetMatrixRt();
    const nn::util::Matrix4x3fType& esetSrt = pEmitter->GetEmitterSet()->GetMatrixSrt();

    nn::util::Vector3fType   pos;
    nn::util::Vector3fType   eye;
    nn::util::Float4x3       matrix;

    nn::util::MatrixGetAxisW( &pos, esetRt );

    nn::util::MatrixStore( &matrix, viewMatrix );
    eye._v[0] = -matrix.m[0][0] * matrix.m[3][0] - matrix.m[0][1] * matrix.m[3][1] - matrix.m[0][2] * matrix.m[3][2];
    eye._v[1] = -matrix.m[1][0] * matrix.m[3][0] - matrix.m[1][1] * matrix.m[3][1] - matrix.m[1][2] * matrix.m[3][2];
    eye._v[2] = -matrix.m[2][0] * matrix.m[3][0] - matrix.m[2][1] * matrix.m[3][1] - matrix.m[2][2] * matrix.m[3][2];

    float billboardY = nn::util::Atan2Est( -( nn::util::VectorGetX( eye ) - nn::util::VectorGetX( pos ) ), ( nn::util::VectorGetZ( eye ) - nn::util::VectorGetZ( pos ) ) );

    nn::util::Vector3fType rot = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, -billboardY, 0.f );

    nn::util::Matrix4x3fType YAxisMatrix;
    nn::util::MatrixIdentity( &YAxisMatrix );
    nn::util::MatrixSetRotateXyz( &YAxisMatrix, rot );
    nn::util::MatrixSetAxisW( &YAxisMatrix, pos );

    float sx = 1.f, sy = 1.f, sz = 1.f;

    nn::util::Vector3fType srtX, srtY, srtZ;
    nn::util::MatrixGetAxisX( &srtX, esetSrt );
    nn::util::MatrixGetAxisY( &srtY, esetSrt );
    nn::util::MatrixGetAxisZ( &srtZ, esetSrt );

    nn::util::Vector3fType rtX, rtY, rtZ;
    nn::util::MatrixGetAxisX( &rtX, esetRt );
    nn::util::MatrixGetAxisY( &rtY, esetRt );
    nn::util::MatrixGetAxisZ( &rtZ, esetRt );

    if ( nn::util::VectorGetX( rtX ) != 0.f )
    {
        sx = nn::util::VectorGetX( srtX ) / nn::util::VectorGetX( rtX );
    }

    if ( nn::util::VectorGetY( rtY ) != 0.f )
    {
        sy = nn::util::VectorGetY( srtY ) / nn::util::VectorGetY( rtY );
    }

    if ( nn::util::VectorGetZ( rtZ ) != 0.f )
    {
        sz = nn::util::VectorGetZ( srtZ ) / nn::util::VectorGetZ( rtZ );
    }

    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( sx, sy, sz );

    nn::util::Matrix4x3fType scaleMatrix;
    nn::util::MatrixIdentity( &scaleMatrix );
    nn::util::MatrixSetScale( &scaleMatrix, scale );

    nn::util::Matrix4x3fType invEsetRt;
    nn::util::MatrixInverse( &invEsetRt, esetRt );
    nn::util::Matrix4x3fType invEsetSrt;
    nn::util::MatrixInverse( &invEsetSrt, esetSrt );

    {
        nn::util::Matrix4x3fType temp;
        nn::util::MatrixMultiply( &temp, pEmitter->GetMatrixRt(), invEsetRt );
        nn::util::MatrixMultiply( &temp, temp, YAxisMatrix );
        pEmitter->SetMatrixRt( temp );
    }
    {
        nn::util::Matrix4x3fType temp;
        nn::util::MatrixMultiply( &temp, pEmitter->GetMatrixSrt(), invEsetSrt );
        nn::util::MatrixMultiply( &temp, temp, scaleMatrix );
        nn::util::MatrixMultiply( &temp, temp, YAxisMatrix );
        pEmitter->SetMatrixSrt( temp );
    }
}

//------------------------------------------------------------------------------
// シェーダ取得
//------------------------------------------------------------------------------
detail::Shader* RenderStateSetArg::GetShader() NN_NOEXCEPT
{
    return pEmitter->GetShader( shaderType );
}

//------------------------------------------------------------------------------
// システム取得
//------------------------------------------------------------------------------
System* RenderStateSetArg::GetSystem() NN_NOEXCEPT
{
    return pEmitter->GetEmitterSet()->GetSystem();
}

System* EmitterDrawArg::GetSystem() NN_NOEXCEPT
{
    return pEmitter->GetEmitterSet()->GetSystem();
}

//------------------------------------------------------------------------------
// エンディアン反転 コールバック
//------------------------------------------------------------------------------
bool EndianFlipCallbackImpl( nn::vfx::EndianFlipArg& arg ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    if( arg.pCustomShaderParam && arg.customShaderParamSize >= 4 )
    {
        NN_SDK_ASSERT( arg.customShaderParamSize % 4 == 0 );
        int loop     = static_cast< int >( arg.customShaderParamSize / 4 );
        float* param = reinterpret_cast< float* >( arg.pCustomShaderParam );

        for( int i = 0; i < loop; i++ )
        {
            nn::vfx::detail::EndianUtil::Flip( reinterpret_cast< float* >( &param[ i ] ) );
        }
    }
#endif
    return true;
}

//------------------------------------------------------------------------------
// 描画設定後コールバック:カスタムシェーダパラメータを自動的に設定する。
//------------------------------------------------------------------------------
bool BindReservedCustomShaderConstantBuffer( nn::vfx::RenderStateSetArg& arg ) NN_NOEXCEPT
{
    nn::gfx::CommandBuffer* pCommandBuffer = arg.pCommandBuffer;
    Emitter*                pEmitter = arg.pEmitter;
    detail::Shader*         pShader = arg.GetShader();
    const EmitterResource*  pEmitterResource = pEmitter->GetEmitterResource();

    if( !pEmitterResource->m_pCustomShaderParam ) return false;

    int reservedConstantBufferSlotV = pShader->GetVertexConstantBufferSlot( detail::Shader::ConstantBufferType_ReservedParam );
    int reservedConstantBufferSlotP = pShader->GetPixelConstantBufferSlot( detail::Shader::ConstantBufferType_ReservedParam );

    nn::gfx::GpuAddress address;
    pEmitterResource->m_pResourceConstantBuffer->GetGpuAddress( &address );
    address.Offset( pEmitterResource->m_CustomShaderConstantBufferOffset );

    if( reservedConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( reservedConstantBufferSlotV, nn::gfx::ShaderStage_Vertex, address, pEmitterResource->m_CustomShaderParamSize );
    }
    if( reservedConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( reservedConstantBufferSlotP, nn::gfx::ShaderStage_Pixel, address, pEmitterResource->m_CustomShaderParamSize );
    }

    return true;
}

//------------------------------------------------------------------------------
// 描画設定後コールバック:エミッタプラグインパラメータを自動的に設定する。
//------------------------------------------------------------------------------
bool BindEmitterPluginConstantBufferAuto( nn::vfx::RenderStateSetArg& arg ) NN_NOEXCEPT
{
    nn::gfx::CommandBuffer* pCommandBuffer = arg.pCommandBuffer;
    Emitter*                pEmitter = arg.pEmitter;
    detail::Shader*         pShader = arg.GetShader();
    const EmitterResource*  pEmitterResource = pEmitter->GetEmitterResource();

    if( !pEmitterResource->m_pEmitterPluginData ) return false;

    int reservedConstantBufferSlotV = pShader->GetVertexConstantBufferSlot( detail::Shader::ConstantBufferType_ReservedParam );
    int reservedConstantBufferSlotP = pShader->GetPixelConstantBufferSlot( detail::Shader::ConstantBufferType_ReservedParam );

    nn::gfx::GpuAddress address;
    pEmitterResource->m_pResourceConstantBuffer->GetGpuAddress( &address );
    address.Offset( pEmitterResource->m_EmitterPluginConstantBufferOffset );

    if( reservedConstantBufferSlotV != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( reservedConstantBufferSlotV, nn::gfx::ShaderStage_Vertex, address, sizeof( nn::util::Float4 ) * 8 );
    }
    if( reservedConstantBufferSlotP != InvalidValueId_ConstantBufferSlotId )
    {
        pCommandBuffer->SetConstantBuffer( reservedConstantBufferSlotP, nn::gfx::ShaderStage_Pixel, address, sizeof( nn::util::Float4 ) * 8 );
    }

    return true;
}

//------------------------------------------------------------------------------
// 描画設定後のコールバック呼び出し
//------------------------------------------------------------------------------
bool InvokeRenderStateSetCallback( nn::vfx::RenderStateSetArg& arg ) NN_NOEXCEPT
{
    const CallbackSet* pCallbackSetEP = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_EmitterPlugin );
    const CallbackSet* pCallbackSetCA = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_CustomAction );
    const CallbackSet* pCallbackSetCS = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_CustomShader );

    if( pCallbackSetCA && pCallbackSetCA->renderStateSet )
    {
        bool ret = pCallbackSetCA->renderStateSet( arg );
        if( !ret )
        {
            return false;
        }
    }

    if( pCallbackSetCS && pCallbackSetCS->renderStateSet )
    {
        bool ret = pCallbackSetCS->renderStateSet( arg );
        if( !ret )
        {
            return false;
        }
    }
    else
    {
        if( arg.pEmitter->GetCustomShaderParameter() )
        {
            BindReservedCustomShaderConstantBuffer( arg );
        }
    }

    if( pCallbackSetEP && !pCallbackSetEP->renderStateSet )
    {
        BindEmitterPluginConstantBufferAuto( arg );
    }
    return true;
}

//------------------------------------------------------------------------------
// エミッタ初期化後のコールバック呼び出し
//------------------------------------------------------------------------------
bool InvokeEmitterInitializeCallback( nn::vfx::EmitterInitializeArg& arg ) NN_NOEXCEPT
{
    const CallbackSet* pCallbackSetEP = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_EmitterPlugin );
    const CallbackSet* pCallbackSetCA  = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_CustomAction );
    const CallbackSet* pCallbackSetCS  = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_CustomShader );

    bool epResult = true;
    bool caResult = true;
    bool csResult = true;

    if( pCallbackSetEP && pCallbackSetEP->emitterInitialize )
    {
        epResult = pCallbackSetEP->emitterInitialize( arg );
    }
    if( pCallbackSetCA && pCallbackSetCA->emitterInitialize )
    {
        caResult = pCallbackSetCA->emitterInitialize( arg );
    }
    if( pCallbackSetCS && pCallbackSetCS->emitterInitialize )
    {
        csResult = pCallbackSetCS->emitterInitialize( arg );
    }

    // 一つでも失敗したコールバックがあれば全体として false を返す。
    // その後にエミッタ破棄コールバックが呼ばれるので後処理は問題なし。
    return ( epResult && caResult && csResult );
}

//------------------------------------------------------------------------------
// 全パーティクル削除時のコールバック呼び出し
//------------------------------------------------------------------------------
void InvokeParticleRemoveCallbackAll( Emitter* pEmitter ) NN_NOEXCEPT
{
    ParticleRemoveCallback particleRemoveCallbackEP = NULL;
    ParticleRemoveCallback particleRemoveCallbackCA = NULL;
    ParticleRemoveCallback particleRemoveCallbackCS = NULL;

    const CallbackSet* pCallbackSetEP = pEmitter->GetCallbackSet( detail::CallbackSetType_EmitterPlugin );
    const CallbackSet* pCallbackSetCA = pEmitter->GetCallbackSet( detail::CallbackSetType_CustomAction );
    const CallbackSet* pCallbackSetCS = pEmitter->GetCallbackSet( detail::CallbackSetType_CustomShader );

    if( pCallbackSetEP && pCallbackSetEP->particleRemove )
    {
        particleRemoveCallbackEP = pCallbackSetEP->particleRemove;
    }
    if( pCallbackSetCA && pCallbackSetCA->particleRemove )
    {
        particleRemoveCallbackCA = pCallbackSetCA->particleRemove;
    }
    if( pCallbackSetCS && pCallbackSetCS->particleRemove )
    {
        particleRemoveCallbackCS = pCallbackSetCS->particleRemove;
    }

    if( particleRemoveCallbackEP || particleRemoveCallbackCA || particleRemoveCallbackCS )
    {
        for( int i = 0; i < pEmitter->GetEnableBufferCount(); i++ )
        {
            const float time = pEmitter->GetParticleTime( i );
            const float life = pEmitter->GetParticleLife( i );
            // MEMO: CalcParticle 同様、粒の生存判定をした方が今後良さそう
            InvokeParticleRemoveCallback( pEmitter, i, time, life, particleRemoveCallbackEP, particleRemoveCallbackCA, particleRemoveCallbackCS );
        }
    }
}

//------------------------------------------------------------------------------
// エミッタ終了時のコールバック呼び出し
//------------------------------------------------------------------------------
void InvokeEmitterFinalizeCallback( nn::vfx::EmitterFinalizeArg& arg ) NN_NOEXCEPT
{
    const CallbackSet* pCallbackSetEP = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_EmitterPlugin );
    const CallbackSet* pCallbackSetCA = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_CustomAction );
    const CallbackSet* pCallbackSetCS = arg.pEmitter->GetCallbackSet( detail::CallbackSetType_CustomShader );

    if( pCallbackSetEP && pCallbackSetEP->emitterFinalize )
    {
        pCallbackSetEP->emitterFinalize( arg );
    }
    if( pCallbackSetCA && pCallbackSetCA->emitterFinalize )
    {
        pCallbackSetCA->emitterFinalize( arg );
    }
    if( pCallbackSetCS && pCallbackSetCS->emitterFinalize )
    {
        pCallbackSetCS->emitterFinalize( arg );
    }
}

//---------------------------------------------------------------------------
//  履歴式ストライプ2 での履歴点の配列への参照を取得します。
//  この API は予告なく仕様が変更される可能性があります。
//---------------------------------------------------------------------------
void ParticleCalculateArgImpl::GetSuperStripeHistoryBuffer( detail::SuperStripeHistory** pOutHistoryHead, int* pOutHistoryNum ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutHistoryHead );
    NN_SDK_ASSERT_NOT_NULL( pOutHistoryNum );

    if( !( pEmitter->GetCalculationType() == detail::EmitterCalculationMode_Cpu &&
        pEmitter->GetEmitterResource()->m_EmitterPluginIndex == detail::SuperStripeSystem::PluginId ) )
    {
        // 履歴式ストライプ2 を持っていない場合は無効値を返す。
        *pOutHistoryHead = NULL;
        *pOutHistoryNum = 0;
        return;
    }

    detail::SuperStripeInstance* pStripe = reinterpret_cast< detail::SuperStripeInstance* >( pEmitterPluginData );
    if( pStripe == NULL )
    {
        // ストライプのデータが無い場合も無効値を返す。
        // ここに来る場合はエミッタが不正な状態。
        *pOutHistoryHead = NULL;
        *pOutHistoryNum = 0;
        return;
    }

    *pOutHistoryHead = pStripe->pHistoryHead->pNext;
    *pOutHistoryNum = pStripe->historyCount;
}

//---------------------------------------------------
// パーティクルの計算済みのSRT行列を取得します。
//---------------------------------------------------
void ParticleCalculateArgImpl::CalculateParticleMatrix( nn::util::Matrix4x3fType* pOutMatrix ) NN_NOEXCEPT
{
    nn::util::MatrixIdentity( pOutMatrix );

    nn::util::Vector3fType       rotation;
    CalculateCurrentRotateRadian( &rotation );

    nn::util::Matrix4x3f emitterMatrix;
    GetEmitterTransformMatrix( &emitterMatrix );

    // ローカルスケールを求める
    const nn::util::Matrix4x3fType& esetRt  = pEmitter->GetEmitterSet()->GetMatrixRt();
    const nn::util::Matrix4x3fType& esetSrt = pEmitter->GetEmitterSet()->GetMatrixSrt();

    float sx = 1.f, sy = 1.f, sz = 1.f;
    nn::util::Vector3fType srtX, srtY, srtZ;
    nn::util::MatrixGetAxisX( &srtX, esetSrt );
    nn::util::MatrixGetAxisY( &srtY, esetSrt );
    nn::util::MatrixGetAxisZ( &srtZ, esetSrt );

    nn::util::Vector3fType rtX, rtY, rtZ;
    nn::util::MatrixGetAxisX( &rtX, esetRt );
    nn::util::MatrixGetAxisY( &rtY, esetRt );
    nn::util::MatrixGetAxisZ( &rtZ, esetRt );

    if ( nn::util::VectorGetX( rtX ) != 0.f )
    {
        sx = nn::util::VectorGetX( srtX ) / nn::util::VectorGetX( rtX );
    }

    if ( nn::util::VectorGetY( rtY ) != 0.f )
    {
        sy = nn::util::VectorGetY( srtY ) / nn::util::VectorGetY( rtY );
    }

    if ( nn::util::VectorGetZ( rtZ ) != 0.f )
    {
        sz = nn::util::VectorGetZ( srtZ ) / nn::util::VectorGetZ( rtZ );
    }

    nn::util::Vector3fType scale_w;
    nn::util::Vector3fType scale_l;
    CalculateWorldScale( &scale_w );

    nn::util::VectorSetX( &scale_l, nn::util::VectorGetX( scale_w ) / sx );
    nn::util::VectorSetY( &scale_l, nn::util::VectorGetY( scale_w ) / sy );
    nn::util::VectorSetZ( &scale_l, nn::util::VectorGetZ( scale_w ) / sz );

    // モデルの向きのマトリクスを設定に合わせて生成
    if ( pEmitter->GetResEmitter()->ptcl.billboardType == nn::vfx::detail::ParticleBillboardType_VelocityDirectionalPolygon ||
         pEmitter->GetResEmitter()->ptcl.billboardType == nn::vfx::detail::ParticleBillboardType_VelocityDirectionalBillboard )
    {
        nn::util::Matrix4x3fType matVelLook;
        nn::util::Vector3fType upVector;
        nn::util::MatrixGetAxisY( &upVector, emitterMatrix );

        nn::util::Vector3fType basisX;
        nn::util::Vector3fType basisY;
        nn::util::Vector3fType basisZ;

        GetLocalDiff( &basisY );
        nn::util::VectorNormalize( &basisY, basisY );

        nn::util::VectorCross( &basisX, upVector, basisY );
        if ( nn::util::VectorGetX( basisX ) == 0.0f )
        {
            nn::util::Vector3fType upVectorDelta;
            nn::util::MatrixGetAxisZ( &upVectorDelta, emitterMatrix );
            nn::util::VectorMultiply( &upVectorDelta, upVectorDelta, 0.001f );
            nn::util::VectorAdd( &upVector, upVector, upVectorDelta );
            nn::util::VectorCross( &basisX, upVector, basisY );
        }

        nn::util::VectorNormalize( &basisX, basisX );
        nn::util::VectorCross( &basisZ, basisX, basisY );

        matVelLook = nn::util::Matrix4x3f(
            nn::util::VectorGetX( basisX ), nn::util::VectorGetY( basisX ), nn::util::VectorGetZ( basisX ),
            nn::util::VectorGetX( basisY ), nn::util::VectorGetY( basisY ), nn::util::VectorGetZ( basisY ),
            nn::util::VectorGetX( basisZ ), nn::util::VectorGetY( basisZ ), nn::util::VectorGetZ( basisZ ),
            0.0f, 0.0f, 0.0f );

        nn::util::Vector3fType pos_l;
        GetLocalPos( &pos_l );

        nn::util::Vector3fType rotate;
        CalculateCurrentRotateRadian( &rotate );

        nn::util::MatrixSetScaleRotateXyz( pOutMatrix, scale_l, rotate );
        nn::util::MatrixMultiply( pOutMatrix, *pOutMatrix, matVelLook );
        nn::util::MatrixSetAxisW( pOutMatrix, pos_l );
        nn::util::MatrixMultiply( pOutMatrix, *pOutMatrix, emitterMatrix );
    }
    else
    {
        nn::util::Vector3fType pos_l;
        GetLocalPos( &pos_l );

        nn::util::Vector3fType rotate;
        CalculateCurrentRotateRadian( &rotate );

        nn::util::MatrixSetScaleRotateXyz( pOutMatrix, scale_l, rotate );
        nn::util::MatrixSetAxisW( pOutMatrix, pos_l );
        nn::util::MatrixMultiply( pOutMatrix, *pOutMatrix, emitterMatrix );
    }
}
} // namespace vfx
} // namespace nn

