﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/vfx_BufferAllocator.h>
#include <nn/vfx/vfx_Misc.h>


namespace nn  {
namespace vfx {
namespace detail {

bool BufferAllocator::Initialize( InitializeArg& arg ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( arg.m_ManagementBufferSize >= InitializeArg::CalculateManagementBufferSize(1) );
    NN_SDK_ASSERT_NOT_NULL( arg.m_pMemoryPoolBuffer );
    NN_SDK_ASSERT_NOT_NULL( arg.m_pManagementBuffer );

    m_Start             = arg.m_pMemoryPoolBuffer;
    m_Size              = arg.m_MemoryPoolBufferSize;
    m_AllocatedCountMax     = static_cast<int>( ( arg.m_ManagementBufferSize - 32 ) / sizeof( Block ) );
    m_BlockArray        = new ( arg.m_pManagementBuffer ) Block[ m_AllocatedCountMax ];
    m_FreeBlock         = &m_BlockArray[0];
    m_BlockList.clear();

    nn::gfx::MemoryPool::InfoType mpInfo;
    mpInfo.SetDefault();
    mpInfo.SetMemoryPoolProperty( arg.m_MemoryPoolProperty );

    size_t poolAlign    = nn::gfx::MemoryPool::GetPoolMemoryAlignment( arg.m_pGfxDevice, mpInfo );
    NN_UNUSED( poolAlign );
    NN_SDK_ASSERT_ALIGNED( arg.m_pMemoryPoolBuffer, poolAlign );
    NN_SDK_ASSERT( arg.m_MemoryPoolBufferSize % nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( arg.m_pGfxDevice, mpInfo ) == 0 );
    NN_SDK_ASSERT( arg.m_MemoryPoolBufferSize >= nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( arg.m_pGfxDevice, mpInfo ) );

    mpInfo.SetPoolMemory( arg.m_pMemoryPoolBuffer, arg.m_MemoryPoolBufferSize );
    m_GfxMemoryPool.Initialize( arg.m_pGfxDevice, mpInfo );

    nn::gfx::Buffer::InfoType bfInfo;
    bfInfo.SetDefault();
    bfInfo.SetGpuAccessFlags( arg.m_GpuAccessFlag );
    bfInfo.SetSize( arg.m_MemoryPoolBufferSize );
    m_GfxBuffer.Initialize( arg.m_pGfxDevice, bfInfo, &m_GfxMemoryPool, 0, arg.m_MemoryPoolBufferSize );

    m_BufferAlign = nn::gfx::Buffer::GetBufferAlignment( arg.m_pGfxDevice, bfInfo );

    m_Start             = m_GfxBuffer.Map();
    m_AllocatedSize       = 0;
    m_AllocatedPtr          = 0;
#ifdef _VFX_NSDK01215_COMPATIBLE
    m_ReleaseCounter    = 1;
#else
    m_ReleaseCounter    = _RELEASE_COUNTER;
#endif

    GetGpuAddress( &m_GpuAddress );

    return true;
}


void BufferAllocator::Finalize( nn::gfx::Device* pGfxDevice ) NN_NOEXCEPT
{
    m_FreeBlockList.clear();
    m_GfxBuffer.Finalize( pGfxDevice );
    m_GfxMemoryPool.Finalize( pGfxDevice );
}


void* BufferAllocator::Alloc( size_t size, size_t alignment ) NN_NOEXCEPT
{
    size_t alloc_size = size;
    size_t alloc_alignment = alignment;

    // 管理領域を確保できるか確認
    if ( m_AllocatedCountMax <= m_AllocatedCount )
    {
        OutputWarning( "GpuBuffer management domain is lacking. \n" );
        return NULL;
    }

    // 空きブロックを調べる
    if ( !m_FreeBlock )
    {
        m_AllocatedPtr++;
        if ( m_AllocatedPtr == m_AllocatedCountMax )
        {
            m_AllocatedPtr = 0;
        }

        if ( m_BlockArray[m_AllocatedPtr].m_Addr == NULL )
        {
            m_FreeBlock = &m_BlockArray[m_AllocatedPtr];
        }
        else
        {
            for ( int i = 0; i < m_AllocatedCountMax; i++ )
            {
                if ( m_BlockArray[i].m_Addr == NULL )
                {
                    m_FreeBlock = &m_BlockArray[i];
                    m_AllocatedPtr = i;
                    break;
                }
            }
        }
    }

    Block* freeBlock = m_FreeBlock;

    // 空きメモリ領域を調べる
    BlockList::iterator begin_it    = m_BlockList.begin();
    BlockList::iterator end_it      = m_BlockList.end();
    void*               prev_end    = m_Start;

    for ( BlockList::iterator it = begin_it; it != end_it; ++it )
    {
        void* start = it->m_Addr;
        void* aligned_prev_end = nn::util::BytePtr( prev_end ).AlignUp( alloc_alignment ).Get();

        ptrdiff_t dist = nn::util::BytePtr( aligned_prev_end ).Distance( start );
        if ( dist >= static_cast<ptrdiff_t>(alloc_size) )
        {
            freeBlock->m_Addr = aligned_prev_end;
            freeBlock->m_Size = alloc_size;
            m_BlockList.insert( it, *freeBlock );
            m_FreeBlock = NULL;
            m_AllocatedCount++;
            m_AllocatedSize += freeBlock->m_Size;
            return freeBlock->m_Addr;
        }

        prev_end = nn::util::BytePtr( start ).Advance( it->m_Size ).Get();
    }

    // 最後のブロックから終わりまでの領域に入るかを調べる
    {
        void* aligned_prev_end = nn::util::BytePtr( prev_end ).AlignUp( alloc_alignment ).Get();
        ptrdiff_t dist = nn::util::BytePtr( aligned_prev_end ).Distance( nn::util::BytePtr( m_Start ).Advance( m_Size ).Get() );

        if ( dist >= static_cast<ptrdiff_t>( alloc_size ) )
        {
            freeBlock->m_Addr = aligned_prev_end;
            freeBlock->m_Size = alloc_size;
            m_BlockList.push_back( *freeBlock );
            m_FreeBlock = NULL;
            m_AllocatedCount++;
            m_AllocatedSize += freeBlock->m_Size;
            return freeBlock->m_Addr;
        }
    }

    Warning( nullptr, RuntimeWarningId_GpuBufferAllocationFailed );
    OutputWarning( "GpuBuffer Allocation Failed. Allocated size : %d Byte.\n", size );
    OutputWarning( "Current GpuBuffer Allocation : %d/%d.\n", m_AllocatedSize, m_Size );
    return NULL;
}

//------------------------------------------------------------------------------
void BufferAllocator::Free( void* ptr, bool immediate ) NN_NOEXCEPT
{
    if( ptr == NULL ) return;

    if ( immediate )
    {
        _Free( ptr );
    }
    else
    {
        BlockList::iterator step_it = m_BlockList.begin();
        BlockList::iterator end_it  = m_BlockList.end();

        for ( BlockList::iterator it = step_it; it != end_it; ++it )
        {
            if ( it->m_Addr == ptr )
            {
                it->m_ReleaseCounter = m_ReleaseCounter;
                m_FreeBlockList.push_back( *it );
                return;
            }
        }
    }
}

//---------------------------------------------------------------------------
//  返却リストブロックをヒープへ返却します。
//---------------------------------------------------------------------------
void BufferAllocator::FlushFreeList() NN_NOEXCEPT
{
    FreeBlockList::iterator step_it = m_FreeBlockList.begin();
    FreeBlockList::iterator end_it  = m_FreeBlockList.end();

    for ( ; step_it != end_it;)
    {
        FreeBlockList::iterator it = step_it;
        ++step_it;

        if ( it->m_ReleaseCounter == 0 )
        {
            m_FreeBlockList.erase( it );
            _Free( it->m_Addr );
        }
        else
        {
            it->m_ReleaseCounter--;
        }
    }
}

//---------------------------------------------------------------------------
//  メモリブロックをヒープへ返却します。
//---------------------------------------------------------------------------
void BufferAllocator::_Free( void* ptr ) NN_NOEXCEPT
{
    BlockList::iterator step_it = m_BlockList.begin();
    BlockList::iterator end_it  = m_BlockList.end();

    for (;step_it != end_it;)
    {
        BlockList::iterator it = step_it;
        ++step_it;

        if ( it->m_Addr == ptr )
        {
            it->m_Addr  = NULL;
            m_AllocatedSize -= it->m_Size;
            m_BlockList.erase( it );
            m_FreeBlock = &(*it);
            m_AllocatedCount--;
            return;
        }
    }

    //NN_SDK_LOG( "Invalid Pointer: %p" ptr );
}



} // namespace detail
} // namespace vfx
} // namespace nn
