﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d/ui2d_Window.h>

#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/ui2d/ui2d_Animation.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>

#include <nn/perf.h>

namespace nn
{
namespace ui2d
{

//----------------------------------------
Window::Frame::~Frame()
{
    if (pMaterial && !pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(pMaterial);
    }
    pMaterial = NULL;
}

//----------------------------------------
Window::Window(
    int   contentTexCount,
    int   frameTexCount
)
: m_FrameMode(WindowFrameMode_Around)
, m_WindowFlags(0)
, m_pUseLeftTopEmulationConstantBuffersOffset(NULL)
, m_UseLeftTopEmulationConstantBufferCount(0)
{
    NN_SDK_ASSERT(contentTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameTexCount <= TexMapMax, "name[%s]", GetName());

    Initialize();

    const int  frameNum = 1;
    int  frameTexCountSet[frameNum];
    frameTexCountSet[WindowFrame_LeftTop] = frameTexCount;

    InitializeTexCount(contentTexCount, frameTexCountSet, frameNum);
}

//----------------------------------------
Window::Window(
    int   contentTexCount,
    int   frameLTTexCount,
    int   frameRTTexCount,
    int   frameRBTexCount,
    int   frameLBTexCount
)
: m_FrameMode(WindowFrameMode_Around)
, m_WindowFlags(0)
, m_pUseLeftTopEmulationConstantBuffersOffset(NULL)
, m_UseLeftTopEmulationConstantBufferCount(0)
{
    NN_SDK_ASSERT(contentTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameLTTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameRTTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameRBTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameLBTexCount <= TexMapMax, "name[%s]", GetName());

    Initialize();

    const int  frameNum = 4;
    int  frameTexCount[frameNum];
    frameTexCount[WindowFrame_LeftTop] = frameLTTexCount;
    frameTexCount[WindowFrame_RightTop] = frameRTTexCount;
    frameTexCount[WindowFrame_RightBottom] = frameRBTexCount;
    frameTexCount[WindowFrame_LeftBottom] = frameLBTexCount;

    InitializeTexCount(contentTexCount, frameTexCount, frameNum);
}

//----------------------------------------
Window::Window(
    int   contentTexCount,
    int   cornerLTTexCount,
    int   cornerRTTexCount,
    int   cornerRBTexCount,
    int   cornerLBTexCount,
    int   frameLTexCount,
    int   frameTTexCount,
    int   frameRTexCount,
    int   frameBTexCount
)
: m_FrameMode(WindowFrameMode_Around)
, m_WindowFlags(0)
, m_pUseLeftTopEmulationConstantBuffersOffset(NULL)
, m_UseLeftTopEmulationConstantBufferCount(0)
{
    NN_SDK_ASSERT(contentTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(cornerLTTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(cornerRTTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(cornerRBTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(cornerLBTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameLTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameTTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameRTexCount <= TexMapMax, "name[%s]", GetName());
    NN_SDK_ASSERT(frameBTexCount <= TexMapMax, "name[%s]", GetName());

    Initialize();

    const int  frameNum = 8;
    int  frameTexCount[frameNum];
    frameTexCount[WindowFrame_LeftTop] = cornerLTTexCount;
    frameTexCount[WindowFrame_RightTop] = cornerRTTexCount;
    frameTexCount[WindowFrame_RightBottom] = cornerRBTexCount;
    frameTexCount[WindowFrame_LeftBottom] = cornerLBTexCount;
    frameTexCount[WindowFrame_Left ] = frameLTexCount;
    frameTexCount[WindowFrame_Top ] = frameTTexCount;
    frameTexCount[WindowFrame_Right ] = frameRTexCount;
    frameTexCount[WindowFrame_Bottom ] = frameBTexCount;

    InitializeTexCount(contentTexCount, frameTexCount, frameNum);
}

//----------------------------------------
Window::Window(
    BuildResultInformation* pOutBuildResultInformation,
    nn::gfx::Device* pDevice,
    const ResWindow* pBaseBlock,
    const ResWindow* pOverrideBlock,
    const BuildArgSet& buildArgSet
)
    : Base(pOutBuildResultInformation, pDevice, pBaseBlock, buildArgSet)
    , m_FrameMode(WindowFrameMode_Around)
    , m_WindowFlags(0)
    , m_pUseLeftTopEmulationConstantBuffersOffset(NULL)
    , m_UseLeftTopEmulationConstantBufferCount(0)
{
    Initialize();

    const ResWindow* pBlock;
    if (pOverrideBlock && buildArgSet.overrideMaterialUsageFlag == 0)
    {
        // 丸ごと上書きする場合
        pBlock = pOverrideBlock;
    }
    else
    {
        // 上書きがない、もしくは部分上書きの場合
        pBlock = pBaseBlock;
    }

    m_FrameMode = static_cast<WindowFrameMode>(
        (pBlock->windowFlags >> WindowFlag_WindowKind0) & ((1 << WindowFlag_WindowKindLength) - 1));

    if (pBlock->windowFlags & (1 << WindowFlag_UseVertexColorAll)) { m_WindowFlags |= WindowFlags_OverallVertexColorEnabled; }
    if (pBlock->windowFlags & (1 << WindowFlag_UseOneMaterialForAll)) { m_WindowFlags |= WindowFlags_OneMaterialForAllFrameEnabled; }
    if (pBlock->windowFlags & (1 << WindowFlag_NotDrawContent)) { m_WindowFlags |= WindowFlags_NotDrawContentEnabled; }

    const ResWindowContent *const pResContent = nn::util::ConstBytePtr(pBlock, pBlock->contentOffset).Get<ResWindowContent>();
    const uint8_t  texCoordNum = std::min(pResContent->texCoordCount, static_cast<uint8_t>(TexMapMax));

    InitializeContent(texCoordNum);

    // ウインドウサイズ関連パラメータを初期化
    InitializeSize(pBlock);

    // content
    // 頂点カラー
    for (int i = 0; i < VertexColor_MaxVertexColor; ++i)
    {
        m_Content.vtxColors[i] = pResContent->vtxCols[i];
    }

    // テクスチャ座標
    if (texCoordNum > 0)
    {
        if (!m_Content.texCoordArray.IsEmpty())
        {
            m_Content.texCoordArray.Copy(
                reinterpret_cast<const char*>(pResContent) + sizeof(*pResContent),
                texCoordNum);
        }
    }

    // インスタンスコピー時にキャプチャテクスチャの情報を正しくコピーするために拡張ユーザーデータに参照名を保存する必要がある。
    // 以下の処理でデータを保存する準備を行う。
    const size_t systemDataSize = sizeof(SystemDataCopyCaptureTextureInfo) + sizeof(CaptureTextureCopyInfo) * (pBlock->frameCount + 1);
    void* pSystemData = alloca(systemDataSize);
    NN_SDK_ASSERT_NOT_NULL(pSystemData);

    SystemDataCopyCaptureTextureInfo* pCopyInfoHeader = nn::util::BytePtr(pSystemData, 0).Get<SystemDataCopyCaptureTextureInfo>();
    pCopyInfoHeader->type = PaneSystemDataType_CaptureTextureRuntimeCopyInfo;
    pCopyInfoHeader->count = 0;

    CaptureTextureCopyInfo* pCopyInfo = nn::util::BytePtr(pSystemData, sizeof(SystemDataCopyCaptureTextureInfo)).Get<CaptureTextureCopyInfo>();
    CaptureTextureCopyInfo  localCopyInfo;

    // マテリアルの作成
    {
        const ResWindowContent *const pBaseResContent = nn::util::ConstBytePtr(pBaseBlock, pBaseBlock->contentOffset).Get<ResWindowContent>();
        const ResMaterial *const pResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseResContent->materialIdx);
        const ResMaterial *pOverrideResMaterial = NULL;
        if (pOverrideBlock) {
            const ResWindowContent *const pOverrideResContent = nn::util::ConstBytePtr(pOverrideBlock, pOverrideBlock->contentOffset).Get<ResWindowContent>();
            pOverrideResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideResContent->materialIdx);
        }

        // マテリアルを初期化してキャプチャテクスチャの情報を収集する。
        memset(&localCopyInfo, 0, sizeof(CaptureTextureCopyInfo));
        m_pMaterial = Layout::AllocateAndConstruct<Material, BuildResultInformation*, nn::gfx::Device*, const ResMaterial*, const ResMaterial*, const BuildArgSet&, CaptureTextureCopyInfo*>(pOutBuildResultInformation, pDevice, pResMaterial, pOverrideResMaterial, buildArgSet, &localCopyInfo);
        if (localCopyInfo.useFlags != 0)
        {
            localCopyInfo.targetId = CaptureTextureCopyTarget_Content;
            memcpy(&pCopyInfo[pCopyInfoHeader->count++], &localCopyInfo, sizeof(CaptureTextureCopyInfo));
        }
    }

    // Frame
    m_FrameCount = 0;
    m_pFrames = 0;
    if (pBlock->frameCount > 0)
    {
        InitializeFrame(pBlock->frameCount);

        const uint32_t *const pFrameOffsetTable = nn::util::ConstBytePtr(pBlock, pBlock->frameOffsetTableOffset).Get<uint32_t>();
        const uint32_t *const pBaseFrameOffsetTable = nn::util::ConstBytePtr(pBaseBlock, pBaseBlock->frameOffsetTableOffset).Get<uint32_t>();
        const uint32_t * pOverrideFrameOffsetTable = NULL;
        if (pOverrideBlock) {
            pOverrideFrameOffsetTable = nn::util::ConstBytePtr(pOverrideBlock, pOverrideBlock->frameOffsetTableOffset).Get<uint32_t>();
        }

        for (int i = 0; i < m_FrameCount; ++i)
        {
            const ResWindowFrame *const pResWindowFrame = nn::util::ConstBytePtr(pBlock, pFrameOffsetTable[i]).Get<ResWindowFrame>();
            m_pFrames[i].textureFlip = pResWindowFrame->textureFlip;

            // マテリアルの初期化
            const ResMaterial* pResMaterial = NULL;
            if(pBaseBlock->frameCount > i )
            {
                const ResWindowFrame* pBaseResWindowFrame = nn::util::ConstBytePtr(pBaseBlock, pBaseFrameOffsetTable[i]).Get<ResWindowFrame>();
                pResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pCurrentBuildResSet, pBaseResWindowFrame->materialIdx);
            }

            const ResMaterial *pOverrideResMaterial = NULL;
            if (pOverrideBlock) {
                const ResWindowFrame *const pOverrideResWindowFrame = nn::util::ConstBytePtr(pOverrideBlock, pOverrideFrameOffsetTable[i]).Get<ResWindowFrame>();
                pOverrideResMaterial = nn::ui2d::detail::GetResMaterial(buildArgSet.pOverrideBuildResSet, pOverrideResWindowFrame->materialIdx);

                // カラー補完を部分的に上書きする指定がされているが、（フレーム数の変更で）上書き前のマテリアルが見つからない場合は不正なので対応できません。(ツール上で設定できないようになっているはずです。)
                NN_SDK_ASSERT(
                    !detail::TestBit(buildArgSet.overrideMaterialUsageFlag, MaterialOverrideUsageFlag_InterpolateColorEnabled) || pResMaterial != NULL,
                    "Invalid condition for Window[%s]", GetName());
            }

            // マテリアルを初期化してキャプチャテクスチャの情報を収集する。
            memset(&localCopyInfo, 0, sizeof(CaptureTextureCopyInfo));
            m_pFrames[i].pMaterial = Layout::AllocateAndConstruct<Material, BuildResultInformation*, nn::gfx::Device*, const ResMaterial*, const ResMaterial*, const BuildArgSet&, CaptureTextureCopyInfo*>(pOutBuildResultInformation, pDevice, pResMaterial, pOverrideResMaterial, buildArgSet, &localCopyInfo);

            if (localCopyInfo.useFlags != 0)
            {
                localCopyInfo.targetId = static_cast<uint8_t>(CaptureTextureCopyTarget_Frame0 + i);
                memcpy(&pCopyInfo[pCopyInfoHeader->count++], &localCopyInfo, sizeof(CaptureTextureCopyInfo));
            }
        }

        // 「左上のみ」マテリアル機能エミュレーション実装。
        // nw 実装の時はライブラリ内部でプラットフォームのグラフィックス API を直接呼び出していたため
        // ウインドウひとつの描画に対して uniform の値を更新しながら逐次描画することが出来た。
        // しかし gfx 化されるにあたって CommandBuffer 形式に変更となり逐次描画が出来なくなた。
        // 機能を互換性を保つため、同等の機能を ConstantBuffer を複数持つことでエミュレーション実装する。
        if (pBlock->frameCount == 1)
        {
            InitializeUseLeftTopMaterialEmulation(pOutBuildResultInformation, pDevice);
        }
    }

    // 有効なデータが存在する場合のみコピー情報をシステム用ユーザー拡張データに追加する。
    if (pCopyInfoHeader->count > 0)
    {
        AddDynamicSystemExtUserData(PaneSystemDataType_CaptureTextureRuntimeCopyInfo, pCopyInfoHeader, sizeof(SystemDataCopyCaptureTextureInfo) + sizeof(CaptureTextureCopyInfo) * pCopyInfoHeader->count);
    }
}// NOLINT(impl/function_size)

//----------------------------------------
void Window::Initialize()
{
    m_WindowSize.left = 0;
    m_WindowSize.right = 0;
    m_WindowSize.top = 0;
    m_WindowSize.bottom = 0;
    m_WindowSize.frameSize.left = 0;
    m_WindowSize.frameSize.right = 0;
    m_WindowSize.frameSize.top = 0;
    m_WindowSize.frameSize.bottom = 0;
    for (int i = 0; i < VertexColor_MaxVertexColor; i++)
    {
        m_Content.texCoordArray.Initialize();
        nn::util::Unorm8x4 vertexColor = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
        m_Content.vtxColors[i] = vertexColor;
    }
    m_FrameMode = WindowFrameMode_Around;
    m_FrameCount = 0;
    m_WindowFlags = 0;
    m_pFrames = NULL;
    m_pMaterial = NULL;
    m_pUseLeftTopEmulationConstantBuffersOffset = NULL;
    m_UseLeftTopEmulationConstantBufferCount = 0;
}

//----------------------------------------
void Window::InitializeSize(const ResWindow* pBlock)
{
    m_WindowSize.left = pBlock->inflation.left;
    m_WindowSize.right = pBlock->inflation.right;
    m_WindowSize.top = pBlock->inflation.top;
    m_WindowSize.bottom = pBlock->inflation.bottom;
    if (m_WindowSize.left != 0 || m_WindowSize.right != 0 || m_WindowSize.top != 0 || m_WindowSize.bottom != 0)
    {
        m_WindowFlags |= WindowFlags_ContentInflationEnabled;
    }
    switch (m_FrameMode)
    {
    case WindowFrameMode_Horizontal:
        m_WindowSize.frameSize.left = pBlock->frameSize.left;
        m_WindowSize.frameSize.right = pBlock->frameSize.right;
        m_WindowSize.frameSize.top = pBlock->frameSize.top;
        m_WindowSize.frameSize.bottom = pBlock->frameSize.bottom;
        SetSize(Size::Create(GetSize().width, static_cast<float>(pBlock->frameSize.top)));
        break;
    case WindowFrameMode_HorizontalNoContent:
        m_WindowSize.frameSize.left = static_cast<uint16_t >(GetSize().width) - pBlock->frameSize.right;
        m_WindowSize.frameSize.right = pBlock->frameSize.right;
        m_WindowSize.frameSize.top = pBlock->frameSize.top;
        m_WindowSize.frameSize.bottom = pBlock->frameSize.bottom;
        SetSize(Size::Create(GetSize().width, static_cast<float>(pBlock->frameSize.top)));
        break;
    case WindowFrameMode_Around:
    default:
        m_WindowSize.frameSize.left = pBlock->frameSize.left;
        m_WindowSize.frameSize.right = pBlock->frameSize.right;
        m_WindowSize.frameSize.top = pBlock->frameSize.top;
        m_WindowSize.frameSize.bottom = pBlock->frameSize.bottom;
        break;
    }
}

//----------------------------------------
void Window::InitializeUseLeftTopMaterialEmulation(BuildResultInformation* pOutBuildResultInformation, nn::gfx::Device* pDevice)
{
    switch (m_FrameMode)
    {
    case WindowFrameMode_Horizontal:
    {
        // 右側のマテリアルをコピーするためコンスタントバッファは１つ必要。
        m_pUseLeftTopEmulationConstantBuffersOffset = static_cast<uint32_t*>(Layout::AllocateMemory(sizeof(uint32_t)));
        m_UseLeftTopEmulationConstantBufferCount = 1;
    }
    break;
    case WindowFrameMode_HorizontalNoContent:
    {
        // 右側のマテリアルをコピーするためコンスタントバッファは１つ必要。
        m_pUseLeftTopEmulationConstantBuffersOffset = static_cast<uint32_t*>(Layout::AllocateMemory(sizeof(uint32_t)));
        m_UseLeftTopEmulationConstantBufferCount = 1;
    }
    break;
    case WindowFrameMode_Around:
    default:
    {
        // オリジナル以外の３隅のマテリアルをコピーするためコンスタントバッファは３つ必要。
        m_pUseLeftTopEmulationConstantBuffersOffset = static_cast<uint32_t*>(Layout::AllocateMemory(sizeof(uint32_t) * 3));
        m_UseLeftTopEmulationConstantBufferCount = 3;
    }
    break;
    }

    if (pOutBuildResultInformation != NULL)
    {
        size_t constantBufferSize = GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, sizeof(Material::ConstantBufferForVertexShader));
        pOutBuildResultInformation->requiredUi2dConstantBufferSize += (constantBufferSize * m_UseLeftTopEmulationConstantBufferCount);
    }
}

//----------------------------------------
void Window::CopyImpl(const Window& window, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pNewRootName)
{
    m_WindowSize = window.m_WindowSize;
    m_FrameMode = window.m_FrameMode;
    m_FrameCount = window.m_FrameCount;
    m_WindowFlags = window.m_WindowFlags;
    m_UseLeftTopEmulationConstantBufferCount = 0;

    // content
    // 頂点カラー
    for (int i = 0; i < VertexColor_MaxVertexColor; ++i)
    {
        m_Content.texCoordArray.Initialize();
        m_Content.vtxColors[i] = window.m_Content.vtxColors[i];
    }

    // テクスチャ座標
    const int  texCoordNum = window.m_Content.texCoordArray.GetSize();
    if (texCoordNum > 0)
    {
        InitializeContent(texCoordNum);
        m_Content.texCoordArray.SetSize(texCoordNum);
        for (int  i = 0; i < texCoordNum; i++)
        {
            m_Content.texCoordArray.SetCoord(i, window.m_Content.texCoordArray.GetArray()[i]);
        }
    }

    bool captureTextureCopyInfoEnabled = detail::TestBit(window.GetSystemExtDataFlag(), PaneSystemDataType_CaptureTextureRuntimeCopyInfo);
    const void* pSystemData = window.FindSystemExtDataByType(PaneSystemDataType_CaptureTextureRuntimeCopyInfo);
    const SystemDataCopyCaptureTextureInfo* pCopyInfoHeader = NULL;
    const CaptureTextureCopyInfo* pCopyInfo[CaptureTextureCopyTarget_Max];

    // アクセスしやすいように CopyInfo の配列に変換する。
    if (captureTextureCopyInfoEnabled)
    {
        pCopyInfoHeader = nn::util::ConstBytePtr(pSystemData, 0).Get<SystemDataCopyCaptureTextureInfo>();
        memset(&pCopyInfo, 0, sizeof(CaptureTextureCopyInfo*) * CaptureTextureCopyTarget_Max);

        for (uint32_t i = 0; i < pCopyInfoHeader->count; ++i)
        {
            const CaptureTextureCopyInfo* pCopyInfoSrc = nn::util::ConstBytePtr(pSystemData, sizeof(SystemDataCopyCaptureTextureInfo) + sizeof(CaptureTextureCopyInfo) * i).Get<CaptureTextureCopyInfo>();
            pCopyInfo[pCopyInfoSrc->targetId] = pCopyInfoSrc;
        }
    }

    MaterialCopyContext copyContext;
    copyContext.pDevice = pDevice;
    copyContext.pResAccessor = pResAccessor;
    copyContext.pNewRootName = pNewRootName;

    // コンテンツのマテリアルの複製
    if (captureTextureCopyInfoEnabled &&
        pCopyInfo[CaptureTextureCopyTarget_Content] != NULL)
    {
        copyContext.pCaptureTextureCopyInfo = nn::util::ConstBytePtr(window.FindSystemExtDataByType(PaneSystemDataType_CaptureTextureRuntimeCopyInfo), sizeof(SystemDataCopyCaptureTextureInfo)).Get<CaptureTextureCopyInfo>();
        m_pMaterial = Layout::AllocateAndConstruct<Material, const Material&, MaterialCopyContext&>(*window.m_pMaterial, copyContext);
    }
    else
    {
        m_pMaterial = Layout::AllocateAndConstruct<Material, const Material&, nn::gfx::Device*>(*window.m_pMaterial, pDevice);
    }

    // フレームのマテリアルの複製
    InitializeFrame(m_FrameCount);
    for (int i = 0; i < m_FrameCount; ++i)
    {
        m_pFrames[i].textureFlip = window.m_pFrames[i].textureFlip;

        if (captureTextureCopyInfoEnabled &&
            pCopyInfo[CaptureTextureCopyTarget_Frame0 + i] != NULL)
        {
            copyContext.pCaptureTextureCopyInfo = nn::util::ConstBytePtr(window.FindSystemExtDataByType(PaneSystemDataType_CaptureTextureRuntimeCopyInfo), sizeof(SystemDataCopyCaptureTextureInfo) + sizeof(CaptureTextureCopyInfo) * (i + 1)).Get<CaptureTextureCopyInfo>();
            m_pFrames[i].pMaterial = Layout::AllocateAndConstruct<Material, const Material&, MaterialCopyContext&>(*window.m_pMaterial, copyContext);
        }
        else
        {
            m_pFrames[i].pMaterial = Layout::AllocateAndConstruct<Material, const Material&, nn::gfx::Device*>(*window.m_pFrames[i].pMaterial, pDevice);
        }
    }

    if (window.m_pUseLeftTopEmulationConstantBuffersOffset != NULL)
    {
        InitializeUseLeftTopMaterialEmulation(NULL, pDevice);
    }
    else
    {
        m_pUseLeftTopEmulationConstantBuffersOffset = NULL;
    }
}

//----------------------------------------
void
Window::InitializeTexCount(
    int   contentTexCount,
    int   frameTexCount[],
    int   frameNum
)
{
    InitializeContent(contentTexCount);

    m_WindowSize.left = 0;
    m_WindowSize.right = 0;
    m_WindowSize.top = 0;
    m_WindowSize.bottom = 0;
    m_WindowSize.frameSize.left = 0;
    m_WindowSize.frameSize.right = 0;
    m_WindowSize.frameSize.top = 0;
    m_WindowSize.frameSize.bottom = 0;

    // マテリアルの作成
    m_pMaterial = Layout::AllocateAndConstruct<Material>();
    if (m_pMaterial)
    {
        SetDefaultShaderId(m_pMaterial, contentTexCount);

        m_pMaterial->ReserveMem(contentTexCount, contentTexCount, contentTexCount);
    }

    // Frame
    InitializeFrame(frameNum);

    for (int i = 0; i < m_FrameCount; ++i)
    {
        // マテリアルの作成
        m_pFrames[i].pMaterial = Layout::AllocateAndConstruct<Material>();
        if (m_pFrames[i].pMaterial)
        {
            SetDefaultShaderId(m_pFrames[i].pMaterial, frameTexCount[i]);

            m_pFrames[i].pMaterial->ReserveMem(frameTexCount[i], frameTexCount[i], frameTexCount[i]);
        }
    }
}

//----------------------------------------
void
Window::InitializeContent(int  texNum)
{
    if (texNum > 0)
    {
        ReserveTexCoord(texNum);
    }
}

//----------------------------------------
void
Window::InitializeFrame(int  frameNum)
{
    m_FrameCount = 0;
    m_pFrames = Layout::NewArray<Frame>(frameNum);
    if (m_pFrames)
    {
        m_FrameCount = static_cast<uint8_t>(frameNum);
    }
}

//----------------------------------------
Window::~Window()
{
    NN_SDK_ASSERT(m_pMaterial == 0, "Window must be finalized bofore destruction, name[%s]", GetName());
}

//----------------------------------------
void
Window::Finalize(nn::gfx::Device* pDevice)
{
    Pane::Finalize(pDevice);

    if (m_pUseLeftTopEmulationConstantBuffersOffset != NULL)
    {
        Layout::FreeMemory(m_pUseLeftTopEmulationConstantBuffersOffset);
    }

    for (int i = 0; i < m_FrameCount; i++)
    {
        m_pFrames[i].pMaterial->Finalize(pDevice);
    }
    Layout::DeleteArray(m_pFrames, m_FrameCount);

    if (m_pMaterial && ! m_pMaterial->IsUserAllocated())
    {
        m_pMaterial->Finalize(pDevice);
        Layout::DeleteObj(m_pMaterial);
        m_pMaterial = 0;
    }

    m_Content.texCoordArray.Free();
}

//----------------------------------------
const nn::util::Unorm8x4
Window::GetVertexColor(int  idx) const
{
    NN_SDK_ASSERT(idx < VertexColor_MaxVertexColor, "out of bounds: idx[%u] < VertexColor_MaxVertexColor for Window[%s]", idx, GetName());

    return m_Content.vtxColors[idx];
}

//----------------------------------------
void
Window::SetVertexColor(
    int          idx,
    const nn::util::Unorm8x4& value
)
{
    NN_SDK_ASSERT(idx < VertexColor_MaxVertexColor, "out of bounds: idx[%u] < VertexColor_MaxVertexColor for Window[%s]", idx, GetName());

    m_Content.vtxColors[idx] = value;
}

//----------------------------------------
uint8_t
Window::GetVertexColorElement(int  idx) const
{
    return detail::GetVertexColorElement(m_Content.vtxColors, idx);
}

//----------------------------------------
void
Window::SetVertexColorElement(int  idx, uint8_t  value)
{
    detail::SetVertexColorElement(m_Content.vtxColors, idx, value);
}

//----------------------------------------
uint8_t
Window::GetMaterialCount() const
{
    return uint8_t (1 + m_FrameCount);
}

//----------------------------------------
Material*
Window::GetMaterial(int  idx) const
{
    NN_DETAIL_UI2D_WARNING(idx < GetMaterialCount(), "idx >= GetMaterialCount() : %d >= %d", idx, GetMaterialCount());

    return idx == 0 ? GetContentMaterial(): GetFrameMaterial(WindowFrame(idx - 1));
}

//----------------------------------------
Material* Window::FindMaterialByName(const char* pFindName, bool bRecursive)
{
    if (m_pMaterial)
    {
        if (detail::EqualsMaterialName(m_pMaterial->GetName(), pFindName))
        {
            return m_pMaterial;
        }
    }
    for (int i = 0; i < m_FrameCount; ++i)
    {
        if (detail::EqualsMaterialName(m_pFrames[i].pMaterial->GetName(), pFindName))
        {
            return m_pFrames[i].pMaterial;
        }
    }

    if (bRecursive)
    {
        // 子供に一致するものが無いか検索
        PaneList::iterator endIter = GetChildList().end();
        for (PaneList::iterator iter = GetChildList().begin(); iter != endIter; ++iter)
        {
            if (Material* pMat = (*iter).FindMaterialByName(pFindName, bRecursive))
            {
                return pMat;
            }
        }
    }

    return 0;
}

//----------------------------------------
const Material* Window::FindMaterialByName(const char* pFindName, bool bRecursive) const
{
    return const_cast<Window*>(this)->FindMaterialByName(pFindName, bRecursive);
}


//----------------------------------------
void
Window::SetFrameMaterial(WindowFrame frameIdx, Material* pMaterial)
{
    NN_SDK_ASSERT(frameIdx < WindowFrame_MaxWindowFrame, "out of bounds: frameIdx[%u] < WindowFrame_MaxWindowFrame for Window[%s]", frameIdx, GetName());

    if (m_pFrames[frameIdx].pMaterial == pMaterial)
    {
        return;
    }

    if (m_pFrames[frameIdx].pMaterial != NULL &&
        !m_pFrames[frameIdx].pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pFrames[frameIdx].pMaterial);
    }

    m_pFrames[frameIdx].pMaterial = pMaterial;
}

//----------------------------------------
void
Window::SetContentMaterial(Material* pMaterial)
{
    if (m_pMaterial == pMaterial)
    {
        return;
    }

    if (m_pMaterial != NULL && !m_pMaterial->IsUserAllocated())
    {
        Layout::DeleteObj(m_pMaterial);
    }

    m_pMaterial = pMaterial;
}

//----------------------------------------
void
Window::Calculate(DrawInfo& drawInfo, Pane::CalculateContext& context, bool isDirtyParentMtx)
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Window::CalculateMtx");

    Pane::Calculate(drawInfo, context, isDirtyParentMtx);

    LoadMtx(drawInfo);

#if defined(NN_UI2D_FIX_INFLUENCE_ALPHA_BUG)
    if (CheckInvisibleAndUpdateConstantBufferReady())
    {
        return;
    }
#else
    // 非表示、もしくはアルファ値が 0 になって見えない状態のペインは
    // コンスタントバッファの確保と更新処理は不要なので行わない。
    if (!IsVisible() || GetGlobalAlpha() == 0)
    {
        if (context.isInfluenceAlpha)
        {
            SetConstantBufferReady(false);
        }
        else
        {
            SetConstantBufferReadySelf(false);
        }
        return;
    }
#endif

    m_pMaterial->AllocateConstantBuffer(drawInfo);
    m_pMaterial->SetupBlendState(&drawInfo);
    for (int i = 0; i < m_FrameCount; i++)
    {
        m_pFrames[i].pMaterial->AllocateConstantBuffer(drawInfo);
        m_pFrames[i].pMaterial->SetupBlendState(&drawInfo);
    }

    // 使用メモリ量計測モード時に GpuBuffer からメモリが確保できない状態になることがある。
    // メモリが確保できない場合、ここ以下の処理は継続できないため途中で抜けます。
    // なお、計測モード時は実際の GpuBuffer::Allocate 呼び出しによって使用量を計測しているため
    // m_pMaterial->AllocateConstantBuffer 呼び出しの後で抜ける必要があります。
    if (m_pMaterial->GetConstantBufferForVertexShader(drawInfo) == NULL ||
        m_pMaterial->GetConstantBufferForPixelShader(drawInfo) == NULL)
    {
        // CalculateFrame 内にメモリが確保できる状態かどうかの分岐を入れるとコードが複雑になるため
        // CalculateFrame 内で確保される予定の GpuBuffer メモリをエミュレーションします。
        if (drawInfo.GetUi2dConstantBuffer()->IsAnalyseMode())
        {
            int allocateCount = 0;
            if (m_FrameCount == 1)
            {
                switch (m_FrameMode)
                {
                case WindowFrameMode_Around:
                    allocateCount = 3;
                    break;
                case WindowFrameMode_Horizontal:
                    allocateCount = 1;
                    break;
                case WindowFrameMode_HorizontalNoContent:
                    allocateCount = 1;
                    break;
                default:
                    NN_SDK_ASSERT(false, "Unknow frame mode error! for Window[%s]", GetName());
                    break;
                }
            }

            size_t constantBufferAlignment = drawInfo.GetGraphicsResource()->GetConstantBufferAlignment();
            for (int i = 0; i < allocateCount;i++)
            {
                drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(Material::ConstantBufferForVertexShader), constantBufferAlignment));
            }
        }
        return;
    }

    switch (m_FrameMode)
    {
    case WindowFrameMode_Around:
        CalculateAroundFrameWindow(drawInfo);
        break;
    case WindowFrameMode_Horizontal:
        CalculateHorizontalFrameWindow(drawInfo);
        break;
    case WindowFrameMode_HorizontalNoContent:
        CalculateHorizontalFrameNocontentWindow(drawInfo);
        break;
    default:
        NN_SDK_ASSERT(false, "Unknow frame mode error! for Window[%s]", GetName());
        break;
    }

    // ペインエフェクト機能が有効の場合はキャプチャに備えてマテリアルのコンスタントバッファ内の行列を書き換えます。
    // ペインエフェクト機能は Pane 側の機能ですが、派生先で自分自身を描画する際のコンスタントバッファの作成には直接関与することができないため
    // キャプチャ用の ModelView 行列や Projection 行列の設定は派生クラス側で行う必要があります。
    // また、コンスタントバッファにせていされる前の行列を操作すると階層ツリーの姿勢に影響してしまうためここで直接書き換えています。
    if (IsPaneEffectEnabled())
    {
        // 自分自身のコンスタントバッファを書き換え
        UpdateMaterialConstantBufferForEffectCapture(drawInfo);

        nn::util::MatrixT4x4fType   projMtx;
        CalculateCaptureProjectionMatrix(projMtx);

        nn::util::MatrixT4x3fType glbMtx;
        nn::util::MatrixIdentity(&glbMtx);

        // フレーム用のコンスタントバッファを書き換え
        for (int i = 0; i < m_FrameCount; i++)
        {
            Material* pMat = m_pFrames[i].pMaterial;
            Material::ConstantBufferForVertexShader* pConstantBuffer = pMat->GetConstantBufferForVertexShader(drawInfo);

            nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(pConstantBuffer->projection), projMtx);
            nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pConstantBuffer->modelView), glbMtx);
        }

        // 「左上だけ指定」時のコピーマテリアルを書き換え
        for (uint32_t i = 0; i < m_UseLeftTopEmulationConstantBufferCount; ++i)
        {
            nn::util::BytePtr   pUseLeftTopEmulationConstantBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
            Material::ConstantBufferForVertexShader* pConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pUseLeftTopEmulationConstantBuffer.Advance(m_pUseLeftTopEmulationConstantBuffersOffset[i]).Get());

            nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(pConstantBuffer->projection), projMtx);
            nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pConstantBuffer->modelView), glbMtx);
        }
    }

    // ユーザーシェーダー更新コールバックが設定されていたら呼び出す。
    UpdateUserShaderCallback pCallback = drawInfo.GetUpdateUserShaderCallback();
    void*   pUserData = drawInfo.GetUpdateUserShaderCallbackUserData();

    if (pCallback != NULL)
    {
        (*pCallback)(drawInfo, this, pUserData);
    }
}// NOLINT(impl/function_size)

void
Window::SetupPaneEffectSourceImageRenderState(nn::gfx::CommandBuffer& commandBuffer) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pMaterial);

    m_pMaterial->SetCommandBufferOnlyBlend(commandBuffer);
}


//----------------------------------------
void
Window::DrawSharedMaterialImpl(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    // フレームの描画
    // テクスチャが貼られていない場合、全方位ウィンドウでは、フレームは描画せずコンテンツだけ描画をおこなう。
    // (水平ウィンドウでは、なにも描画しない)
    Material* pTevMaterial = m_pFrames[0].pMaterial;
    if (pTevMaterial->GetTexMapCount() > 0)
    {
        // テクスチャコンバイナ（シェーダー・ピクセルシェーダー定数）の設定は、左上マテリアルを参照する。
        // 一回だけ設定コマンドを発行する。
        {
            if (drawInfo.RecordCurrentShader(pTevMaterial->GetShaderInfo(), pTevMaterial->GetShaderVariation()))
            {
                pTevMaterial->SetShader(commandBuffer);
            }
            drawInfo.SetupProgram(&commandBuffer);

            // ピクセルシェーダーコンスタントバッファとブレンドの設定
            pTevMaterial->ApplyPixelShaderConstantBuffer(commandBuffer, drawInfo);
            pTevMaterial->SetCommandBufferOnlyBlend(commandBuffer);

            // ペインエフェクトが有効な場合はキャプチャ用に描画設定を変更する。
            if (IsPaneEffectEnabled())
            {
                UpdateRenderStateForPaneEffectCapture(commandBuffer, drawInfo);
            }
        }
        // エミュレーション描画が有効かどうか
        if(m_UseLeftTopEmulationConstantBufferCount > 0)
        {
            NN_SDK_ASSERT(m_FrameCount == 1, "Window[%s]", GetName());
            NN_SDK_ASSERT(m_pUseLeftTopEmulationConstantBuffersOffset != NULL, "Window[%s]", GetName());

            {
                pTevMaterial->SetupSubmaterialOf_Texture(drawInfo, commandBuffer);
                pTevMaterial->ApplyVertexShaderConstantBuffer(commandBuffer, drawInfo);
                pTevMaterial->ApplyGeometryShaderConstantBuffer(commandBuffer, drawInfo);

                detail::DrawQuad(commandBuffer, drawInfo);
            }

            {
                const nn::gfx::GpuAddress* pGpuAddrForIndexBuffer;
                pGpuAddrForIndexBuffer = drawInfo.GetGraphicsResource()->GetIndexBufferGpuAddress();

                // エミュレーション用データの数だけ頂点バッファを入れ替えつつ描画する。
                for (uint32_t i = 0; i < m_UseLeftTopEmulationConstantBufferCount;i++ )
                {
                    nn::gfx::GpuAddress gpuAddr;

                    gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
                    gpuAddr.Offset(m_pUseLeftTopEmulationConstantBuffersOffset[i]);

                    commandBuffer.SetConstantBuffer(pTevMaterial->GetConstantBufferSlotForVertexShader(), nn::gfx::ShaderStage_Vertex, gpuAddr, sizeof(Material::ConstantBufferForVertexShader));
                    commandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint16, *pGpuAddrForIndexBuffer, static_cast<int>(GraphicsResource::VertexBufferIndexCount), 0);
                }
            }

        }else{

            for (int i = 0; i < m_FrameCount; ++i)
            {
                const Frame& frame = m_pFrames[i];
                if (frame.pMaterial->GetTexMapCount() <= 0)
                {
                    continue;
                }

                frame.pMaterial->SetupSubmaterialOf_Texture(drawInfo, commandBuffer);
                frame.pMaterial->ApplyVertexShaderConstantBuffer(commandBuffer, drawInfo);
                frame.pMaterial->ApplyGeometryShaderConstantBuffer(commandBuffer, drawInfo);

                detail::DrawQuad(commandBuffer, drawInfo);
            }
        }
    }

    // コンテントの描画
    if(!(m_WindowFlags & WindowFlags_NotDrawContentEnabled))
    {
        if( m_FrameMode == WindowFrameMode_Around)
        {
            // シェーダーも含め DrawQuad 内で再設定します。
            m_pMaterial->SetupSubmaterialOf_Texture(drawInfo, commandBuffer);

            detail::SetupMaterialRenderState(commandBuffer, drawInfo, *m_pMaterial);

            // ペインエフェクトが有効な場合はキャプチャ用に描画設定を変更する。
            if (IsPaneEffectEnabled())
            {
                UpdateRenderStateForPaneEffectCapture(commandBuffer, drawInfo);
            }
            detail::DrawQuad(commandBuffer, drawInfo);

        }else if(m_FrameMode == WindowFrameMode_Horizontal){

            // テクスチャ、頂点属性は自分の設定を使う
            m_pMaterial->SetupSubmaterialOf_Texture(drawInfo, commandBuffer);
            m_pMaterial->ApplyVertexShaderConstantBuffer(commandBuffer, drawInfo);
            m_pMaterial->ApplyGeometryShaderConstantBuffer(commandBuffer, drawInfo);

            detail::DrawQuad(commandBuffer, drawInfo);
        }else if(m_FrameMode != WindowFrameMode_HorizontalNoContent){
            // なにもしません。
        }
    }
}

//----------------------------------------
void
Window::DrawNormalImpl(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    // フレームの描画
    for (int i = 0; i < m_FrameCount; ++i)
    {
        const Frame& frame = m_pFrames[i];
        if (frame.pMaterial->GetTexMapCount() > 0)
        {
            Material* pTevMaterial = frame.pMaterial;

            frame.pMaterial->SetupSubmaterialOf_Texture(drawInfo, commandBuffer);

            // テクスチャコンバイナの設定は、左上マテリアルを参照する
            if (drawInfo.RecordCurrentShader(pTevMaterial->GetShaderInfo(), pTevMaterial->GetShaderVariation()))
            {
                pTevMaterial->SetShader(commandBuffer);
            }
            drawInfo.SetupProgram(&commandBuffer);
            frame.pMaterial->SetCommandBuffer(commandBuffer, drawInfo);

            if (IsPaneEffectEnabled())
            {
                commandBuffer.SetBlendState(drawInfo.GetGraphicsResource()->GetPresetBlendState(nn::ui2d::PresetBlendStateId_OpaqueOrAlphaTest));
            }

            detail::DrawQuad(commandBuffer, drawInfo);
        }
    }

    // コンテンツの描画
    if (!(m_WindowFlags & WindowFlags_NotDrawContentEnabled))
    {
        // テクスチャは自分の設定を使う
        m_pMaterial->SetupSubmaterialOf_Texture(drawInfo, commandBuffer);

        detail::SetupMaterialRenderState(commandBuffer, drawInfo, *m_pMaterial);

        if (IsPaneEffectEnabled())
        {
            UpdateRenderStateForPaneEffectCapture(commandBuffer, drawInfo);
        }

        detail::DrawQuad(commandBuffer, drawInfo);
    }
}

//----------------------------------------
void
Window::DrawSelf(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    NN_SDK_ASSERT(drawInfo.GetUi2dConstantBuffer() != NULL, "ConstantBuffer must not be null for Window[%s]", GetName());
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::ui2d::Window::DrawSelf");

    // 使用メモリ計測モード時は描画は行わない。
    if (drawInfo.GetUi2dConstantBuffer()->IsAnalyseMode())
    {
        return;
    }

    // マテリアル共有モードの描画かどうか
    const bool IsOneMaterialForAllFrameEnabled = (m_WindowFlags & WindowFlags_OneMaterialForAllFrameEnabled) != 0;
    if(IsOneMaterialForAllFrameEnabled || m_UseLeftTopEmulationConstantBufferCount > 0)
    {
        DrawSharedMaterialImpl(drawInfo, commandBuffer);
    }else{

        NN_SDK_ASSERT(m_FrameMode == WindowFrameMode_Around, "Window[%s]", GetName());

        DrawNormalImpl(drawInfo, commandBuffer);
    }
}


//----------------------------------------
bool
Window::CompareCopiedInstanceTest(const Window& target) const
{
    // 下記構造体はパディングが入らない前提で memcmp で比較している。
    if (memcmp(&m_WindowSize, &target.m_WindowSize, sizeof(WindowSize)) != 0 ||
        memcmp(m_Content.vtxColors, target.m_Content.vtxColors, sizeof(nn::util::Unorm8x4) * VertexColor_MaxVertexColor) != 0)
    {
        return false;
    }

    if (m_Content.texCoordArray.CompareCopiedInstanceTest(target.m_Content.texCoordArray) == false)
    {
        return false;
    }

    if (m_FrameMode != target.m_FrameMode ||
        m_FrameCount != target.m_FrameCount ||
        m_WindowFlags != target.m_WindowFlags ||
        m_UseLeftTopEmulationConstantBufferCount != target.m_UseLeftTopEmulationConstantBufferCount)
    {
        return false;
    }

    for (int i = 0; i < m_FrameCount; ++i)
    {
        if (m_pFrames[i].textureFlip != target.m_pFrames[i].textureFlip)
        {
            return false;
        }
        if (target.m_pFrames[i].pMaterial != NULL)
        {
            if (m_pFrames[i].pMaterial == NULL)
            {
                return false;
            }

            if (m_pFrames[i].pMaterial->CompareCopiedInstanceTest(*target.m_pFrames[i].pMaterial) == false)
            {
                return false;
            }
        }
    }

    if (target.m_pMaterial != NULL)
    {
        if (m_pMaterial == NULL)
        {
            return false;
        }

        if (m_pMaterial->CompareCopiedInstanceTest(*target.m_pMaterial) == false)
        {
            return false;
        }
    }

    if (m_pUseLeftTopEmulationConstantBuffersOffset == NULL &&
        target.m_pUseLeftTopEmulationConstantBuffersOffset != NULL)
    {
        return false;
    }

    if (m_pUseLeftTopEmulationConstantBuffersOffset != NULL &&
        target.m_pUseLeftTopEmulationConstantBuffersOffset == NULL)
    {
        return false;
    }

    return true;
}

//----------------------------------------
void
Window::CalculateAroundFrameWindow(DrawInfo& drawInfo)
{
    WindowFrameSize frameSize;
    frameSize.left = m_WindowSize.frameSize.left;
    frameSize.right = m_WindowSize.frameSize.right;
    frameSize.top = m_WindowSize.frameSize.top;
    frameSize.bottom = m_WindowSize.frameSize.bottom;

    const nn::util::Float2& basePt = GetVertexPos();

    // frameの描画
    switch (m_FrameCount)
    {
    case 1:
        CalculateFrame(drawInfo, basePt, m_pFrames[WindowFrame_LeftTop], frameSize, GetGlobalAlpha());
        break;
    case 4:
        CalculateFrame4(drawInfo, basePt, m_pFrames, frameSize, GetGlobalAlpha());
        break;
    case 8:
        CalculateFrame8(drawInfo, basePt, m_pFrames, frameSize, GetGlobalAlpha());
        break;
    default:
        NN_SDK_ASSERT(false, "Wrong frame number error for Window[%s].", GetName());
        break;
    }

    // contentの描画
    if (! (m_WindowFlags & WindowFlags_NotDrawContentEnabled))
    {
        bool vertexColorEnabled = detail::TestVertexColorEnabled(m_Content.vtxColors);
        if (vertexColorEnabled || GetGlobalAlpha() != 255)
        {
            m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), ShaderVariation_Standard, true, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());
            if (m_WindowFlags & WindowFlags_OverallVertexColorEnabled)
            {
                detail::SetupFrameSize(drawInfo, *m_pMaterial, GetSize(), frameSize);
                detail::SetupVertexColor(drawInfo, *m_pMaterial, m_Content.vtxColors);
                m_pMaterial->GetConstantBufferForVertexShader(drawInfo)->frameSpec = detail::FrameSpecFlag_Content;
            }
            else
            {
                detail::SetupVertexColor(drawInfo, *m_pMaterial, m_Content.vtxColors);
                m_pMaterial->GetConstantBufferForVertexShader(drawInfo)->frameSpec = detail::FrameSpecFlag_Normal;
            }
        }
        else
        {
            m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), ShaderVariation_WithoutVertexColor, true, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());
            m_pMaterial->GetConstantBufferForVertexShader(drawInfo)->frameSpec = detail::FrameSpecFlag_Normal;
        }
        CalculateContent(drawInfo, basePt, frameSize, GetGlobalAlpha());
    }
}

//----------------------------------------
void
Window::CalculateHorizontalFrameWindow(DrawInfo& drawInfo)
{
    WindowFrameSize frameSize;
    frameSize.left = m_WindowSize.frameSize.left;
    frameSize.right = m_WindowSize.frameSize.right;
    frameSize.top = 0.0f;
    frameSize.bottom = 0.0f;

    SetSize(Size::Create(GetSize().width, m_WindowSize.frameSize.top));

    const nn::util::Float2& basePt = GetVertexPos();

    // frameの描画
    switch (m_FrameCount)
    {
    case 1:
        CalculateHorizontalFrame(drawInfo, basePt, m_pFrames[WindowFrame_LeftTop], frameSize, GetGlobalAlpha());
        break;
    case 2:
        CalculateHorizontalFrame2(drawInfo, basePt, m_pFrames, frameSize, GetGlobalAlpha());
        break;
    default:
        NN_SDK_ASSERT(false, "Wrong frame number error for Window[%s].", GetName());
        break;
    }

    // contentの描画
    if (! (m_WindowFlags & WindowFlags_NotDrawContentEnabled))
    {
        // SetupGraphics とシェーダー関連パラメーターの初期化が行われていなかったので修正しています。
        const bool isVerTexColorEnabled = detail::TestVertexColorEnabled(m_Content.vtxColors) || GetGlobalAlpha() != 255;
        const ShaderVariation shaderVariation = isVerTexColorEnabled ? ShaderVariation_Standard : ShaderVariation_WithoutVertexColor;

        m_pMaterial->SetupGraphics(drawInfo, GetGlobalAlpha(), shaderVariation, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        if(isVerTexColorEnabled)
        {
            detail::SetupFrameSize(drawInfo, *m_pMaterial, GetSize(), frameSize);
            detail::SetupVertexColor(drawInfo, *m_pMaterial, m_Content.vtxColors);
        }

        m_pMaterial->GetConstantBufferForVertexShader(drawInfo)->frameSpec = (m_WindowFlags & WindowFlags_OverallVertexColorEnabled) ? detail::FrameSpecFlag_Content : detail::FrameSpecFlag_Normal;

        CalculateContent(drawInfo, basePt, frameSize, GetGlobalAlpha());
    }
}

//----------------------------------------
void
Window::CalculateHorizontalFrameNocontentWindow(DrawInfo& drawInfo)
{
    WindowFrameSize frameSize;
    frameSize.right = m_WindowSize.frameSize.right;
    frameSize.left = GetSize().width - frameSize.right;
    frameSize.top = 0.0f;
    frameSize.bottom = 0.0f;

    SetSize(Size::Create(GetSize().width, m_WindowSize.frameSize.top));


    const nn::util::Float2& basePt = GetVertexPos();

    // frameの描画
    switch (m_FrameCount)
    {
    case 1:
        CalculateHorizontalNocontentFrame(drawInfo, basePt, m_pFrames[WindowFrame_LeftTop], frameSize, GetGlobalAlpha());
        break;
    case 2:
        CalculateHorizontalNocontentFrame2(drawInfo, basePt, m_pFrames, frameSize, GetGlobalAlpha());
        break;
    default:
        NN_SDK_ASSERT(false, "Wrong frame number error for Window[%s].", GetName());
        break;
    }
}

//----------------------------------------
void
Window::CalculateContent(
    DrawInfo& drawInfo,
    const nn::util::Float2& basePt,
    const WindowFrameSize& frameSize,
    uint8_t  alpha
)
{
    NN_UNUSED(alpha);

    nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(
        basePt.v[0] + frameSize.left,
        basePt.v[1] - frameSize.top);
    Size size = Size::Create(
        GetSize().width - frameSize.left - frameSize.right,
        GetSize().height - frameSize.top - frameSize.bottom);

    if (m_WindowFlags & WindowFlags_ContentInflationEnabled)
    {
        const float WINDOWINFL_FP_RECIP_SCALING_FACTOR = 1.0f / static_cast<float>(nn::ui2d::WindowInflationFixedPoint_ScalingFactor);

        float winSizeL = static_cast<float>(m_WindowSize.left) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;
        float winSizeR = static_cast<float>(m_WindowSize.right) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;
        float winSizeT = static_cast<float>(m_WindowSize.top) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;
        float winSizeB = static_cast<float>(m_WindowSize.bottom) * WINDOWINFL_FP_RECIP_SCALING_FACTOR;

        pos.v[0] -= winSizeL;
        pos.v[1] += winSizeT;
        size.width += winSizeL + winSizeR;
        size.height += winSizeT + winSizeB;
    }

    detail::CalculateQuadWithTexCoords(
        drawInfo,
        m_pMaterial->GetConstantBufferForVertexShader(drawInfo),
        pos,
        size,
        m_Content.texCoordArray.GetSize(),
        m_Content.texCoordArray.GetArray());
}

//----------------------------------------
void
Window::CalculateFrame(
    DrawInfo& drawInfo,
    const nn::util::Float2& basePt,
    const Frame& frame,
    const WindowFrameSize& frameSize,
    uint8_t  alpha
)
{
    NN_SDK_ASSERT(drawInfo.GetUi2dConstantBuffer() != NULL,
        "ConstantBuffer must not be null for Window[%s]", GetName());
    NN_SDK_ASSERT(m_pUseLeftTopEmulationConstantBuffersOffset != NULL,
        "m_pUseLeftTopEmulationConstantBuffersOffset must not be null for Window[%s]", GetName());

    // フレームのテクスチャがない場合は何もしない
    if (frame.pMaterial->GetTexMapCount() == 0)
    {
        return;
    }

    bool vertexColorEnabled = detail::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_Standard, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        if (m_WindowFlags & WindowFlags_OverallVertexColorEnabled)
        {
            detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
            detail::SetupVertexColor(drawInfo, *frame.pMaterial, m_Content.vtxColors);
        }
    }
    else
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_WithoutVertexColor, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        if (m_WindowFlags & WindowFlags_OverallVertexColorEnabled)
        {
            detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
        }
    }

    nn::util::Float2 polPt;
    Size polSize;

    detail::GetLTFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    detail::SetupFrame4Transform(drawInfo, *frame.pMaterial, WindowFrame_LeftTop, TextureFlip_None, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
    detail::CalculateQuad(drawInfo, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), polPt, polSize);

    Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader = NULL;
    size_t constantBufferAlignment = drawInfo.GetGraphicsResource()->GetConstantBufferAlignment();

    {
        m_pUseLeftTopEmulationConstantBuffersOffset[0] = static_cast<uint32_t>(drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(Material::ConstantBufferForVertexShader), constantBufferAlignment)));

        nn::util::BytePtr   pUseLeftTopEmulationConstantBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        if (pUseLeftTopEmulationConstantBuffer.Get() != NULL)
        {
            pConstantBufferForVertexShader = static_cast<Material::ConstantBufferForVertexShader*>(pUseLeftTopEmulationConstantBuffer.Advance(m_pUseLeftTopEmulationConstantBuffersOffset[0]).Get());

            if (pConstantBufferForVertexShader != NULL)
            {
                memcpy(pConstantBufferForVertexShader, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
                pConstantBufferForVertexShader->frameSpec = detail::DecideFrame4Transform(WindowFrame_RightTop, TextureFlip_FlipU, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
                detail::GetRTFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
                detail::CalculateQuad(drawInfo, pConstantBufferForVertexShader, polPt, polSize);
            }
        }
    }

    {
        m_pUseLeftTopEmulationConstantBuffersOffset[1] = static_cast<uint32_t>(drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(Material::ConstantBufferForVertexShader), constantBufferAlignment)));

        nn::util::BytePtr   pUseLeftTopEmulationConstantBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        if (pUseLeftTopEmulationConstantBuffer.Get() != NULL)
        {
            pConstantBufferForVertexShader = static_cast<Material::ConstantBufferForVertexShader*>(pUseLeftTopEmulationConstantBuffer.Advance(m_pUseLeftTopEmulationConstantBuffersOffset[1]).Get());
            if (pConstantBufferForVertexShader != NULL)
            {
                memcpy(pConstantBufferForVertexShader, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
                pConstantBufferForVertexShader->frameSpec = detail::DecideFrame4Transform(WindowFrame_RightBottom, TextureFlip_Rotate180, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
                detail::GetRBFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
                detail::CalculateQuad(drawInfo, pConstantBufferForVertexShader, polPt, polSize);
            }
        }
    }

    {
        m_pUseLeftTopEmulationConstantBuffersOffset[2] = static_cast<uint32_t>(drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(Material::ConstantBufferForVertexShader), constantBufferAlignment)));

        nn::util::BytePtr   pUseLeftTopEmulationConstantBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        if (pUseLeftTopEmulationConstantBuffer.Get() != NULL)
        {
            pConstantBufferForVertexShader = static_cast<Material::ConstantBufferForVertexShader*>(pUseLeftTopEmulationConstantBuffer.Advance(m_pUseLeftTopEmulationConstantBuffersOffset[2]).Get());
            if (pConstantBufferForVertexShader != NULL)
            {
                memcpy(pConstantBufferForVertexShader, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
                pConstantBufferForVertexShader->frameSpec = detail::DecideFrame4Transform(WindowFrame_LeftBottom, TextureFlip_FlipV, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
                detail::GetLBFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
                detail::CalculateQuad(drawInfo, pConstantBufferForVertexShader, polPt, polSize);
            }
        }
    }
}

//----------------------------------------
void
Window::CalculateHorizontalFrame(
    DrawInfo& drawInfo,
    const nn::util::Float2& basePt,
    const Frame& frame,
    const WindowFrameSize& frameSize,
    uint8_t  alpha
)
{
    NN_SDK_ASSERT(drawInfo.GetUi2dConstantBuffer() != NULL,
        "ConstantBuffer must not be null for Window[%s]", GetName());
    NN_SDK_ASSERT(m_pUseLeftTopEmulationConstantBuffersOffset != NULL,
        "m_pUseLeftTopEmulationConstantBuffersOffset must not be null for Window[%s]", GetName());

    // フレームのテクスチャがない場合は何もしない
    if (frame.pMaterial->GetTexMapCount() == 0)
    {
        return;
    }

    bool vertexColorEnabled = detail::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_Standard, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        detail::SetupVertexColor(drawInfo, *frame.pMaterial, m_Content.vtxColors);
        if (m_WindowFlags & WindowFlags_OverallVertexColorEnabled)
        {
            detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
        }
    }
    else
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_WithoutVertexColor, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        if (m_WindowFlags & WindowFlags_OverallVertexColorEnabled)
        {
            detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
        }
    }

    nn::util::Float2 polPt;
    Size polSize;

    //  左
    detail::SetupHorizontalFrameTransform(drawInfo, *frame.pMaterial, WindowFrame_LeftTop, TextureFlip_None, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
    detail::GetHorizontalLeftFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    detail::CalculateQuad(drawInfo, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), polPt, polSize);

    //  右
    m_pUseLeftTopEmulationConstantBuffersOffset[0] = static_cast<uint32_t>(drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(Material::ConstantBufferForVertexShader), drawInfo.GetGraphicsResource()->GetConstantBufferAlignment())));

    nn::util::BytePtr   pUseLeftTopEmulationConstantBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
    Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader = static_cast<Material::ConstantBufferForVertexShader*>(pUseLeftTopEmulationConstantBuffer.Advance(m_pUseLeftTopEmulationConstantBuffersOffset[0]).Get());
    if (pConstantBufferForVertexShader != NULL)
    {
        memcpy(pConstantBufferForVertexShader, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
        pConstantBufferForVertexShader->frameSpec = detail::DecideHorizontalFrameTransform(WindowFrame_RightTop, TextureFlip_FlipU, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
        detail::GetHorizontalRightFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
        detail::CalculateQuad(drawInfo, pConstantBufferForVertexShader, polPt, polSize);
    }
}

//----------------------------------------
void
Window::CalculateHorizontalNocontentFrame(
    DrawInfo& drawInfo,
    const nn::util::Float2& basePt,
    const Frame& frame,
    const WindowFrameSize& frameSize,
    uint8_t  alpha
)
{
    NN_SDK_ASSERT(drawInfo.GetUi2dConstantBuffer() != NULL,
        "ConstantBuffer must not be null for Window[%s]", GetName());
    NN_SDK_ASSERT(m_pUseLeftTopEmulationConstantBuffersOffset != NULL,
        "m_pUseLeftTopEmulationConstantBuffersOffset must not be null for Window[%s]", GetName());

    // フレームのテクスチャがない場合は何もしない
    if (frame.pMaterial->GetTexMapCount() == 0)
    {
        return;
    }

    bool vertexColorEnabled = detail::TestVertexColorEnabled(m_Content.vtxColors);
    if (vertexColorEnabled || GetGlobalAlpha() != 255)
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_Standard, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        detail::SetupVertexColor(drawInfo, *frame.pMaterial, m_Content.vtxColors);
        if (m_WindowFlags & WindowFlags_OverallVertexColorEnabled)
        {
            detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
        }
    }
    else
    {
        frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_WithoutVertexColor, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

        if (m_WindowFlags & WindowFlags_OverallVertexColorEnabled)
        {
            detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
        }
    }

    nn::util::Float2 polPt;
    Size polSize;

    //  左
    detail::SetupHorizontalNocontextFrameTransform(drawInfo, *frame.pMaterial, WindowFrame_LeftTop, TextureFlip_None, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
    detail::GetHorizontalLeftFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
    detail::CalculateQuad(drawInfo, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), polPt, polSize);

    //  右
    m_pUseLeftTopEmulationConstantBuffersOffset[0] = static_cast<uint32_t>(drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(Material::ConstantBufferForVertexShader), drawInfo.GetGraphicsResource()->GetConstantBufferAlignment())));

    nn::util::BytePtr   pUseLeftTopEmulationConstantBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
    if (pUseLeftTopEmulationConstantBuffer.Get() != NULL)
    {
        Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader = static_cast<Material::ConstantBufferForVertexShader*>(pUseLeftTopEmulationConstantBuffer.Advance(m_pUseLeftTopEmulationConstantBuffersOffset[0]).Get());
        if (pConstantBufferForVertexShader != NULL)
        {
            memcpy(pConstantBufferForVertexShader, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
            pConstantBufferForVertexShader->frameSpec = detail::DecideHorizontalNocontextFrameTransform(WindowFrame_RightTop, TextureFlip_FlipU, m_WindowFlags & WindowFlags_OverallVertexColorEnabled);
            detail::GetHorizontalRightFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
            detail::CalculateQuad(drawInfo, pConstantBufferForVertexShader, polPt, polSize);
        }
    }
}

//----------------------------------------
void
Window::CalculateHorizontalFrame2(
    DrawInfo& drawInfo,
    const nn::util::Float2& basePt,
    const Frame* frames,
    const WindowFrameSize& frameSize,
    uint8_t  alpha
)
{
    const bool vertexColorEnabled = (detail::TestVertexColorEnabled(m_Content.vtxColors) || GetGlobalAlpha() != 255);
    const bool overallVertexColorEnabled = m_WindowFlags & WindowFlags_OverallVertexColorEnabled;
    for (int i = 0; i < 2; i++)
    {
        const Frame* pFrame = &frames[i];

        if (vertexColorEnabled)
        {
            pFrame->pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_Standard, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

            if (overallVertexColorEnabled)
            {
                detail::SetupFrameSize(drawInfo, *pFrame->pMaterial, GetSize(), frameSize);
                detail::SetupVertexColor(drawInfo, *pFrame->pMaterial, m_Content.vtxColors);
            }
        }
        else
        {
            pFrame->pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_WithoutVertexColor, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

            if (overallVertexColorEnabled)
            {
                detail::SetupFrameSize(drawInfo, *pFrame->pMaterial, GetSize(), frameSize);
            }
        }

        nn::util::Float2 polPt;
        Size polSize;

        if(i == WindowFrame_LeftTop)
        {
            detail::GetHorizontalLeftFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
        }else{
            detail::GetHorizontalRightFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
        }

        // 「左上マテリアル使用」のエミュレーションデータ作成処理。
        // 左上マテリアルのコンスタントバッファをコピーして必要な部分だけ再設定している。
        if ((i > 0) &&
            (m_WindowFlags & WindowFlags_OneMaterialForAllFrameEnabled))
        {
            Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader = pFrame->pMaterial->GetConstantBufferForVertexShader(drawInfo);

            memcpy(pConstantBufferForVertexShader, frames[0].pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
            pConstantBufferForVertexShader->frameSpec = detail::DecideHorizontalFrameTransform(WindowFrame(i), pFrame->GetTextureFlip(), overallVertexColorEnabled);
            detail::CalculateQuad(drawInfo, pConstantBufferForVertexShader, polPt, polSize);
        }
        else
        {
            detail::SetupHorizontalFrameTransform(drawInfo, *pFrame->pMaterial, WindowFrame(i), pFrame->GetTextureFlip(), overallVertexColorEnabled);
            detail::CalculateQuad(drawInfo, pFrame->pMaterial->GetConstantBufferForVertexShader(drawInfo), polPt, polSize);
        }

        pFrame->pMaterial->SetAllowableTextureQuantity(drawInfo);
    }
}

//----------------------------------------
void
Window::CalculateHorizontalNocontentFrame2(
    DrawInfo&         drawInfo,
    const nn::util::Float2&       basePt,
    const Frame*            frames,
    const WindowFrameSize&  frameSize,
    uint8_t                       alpha
)
{
    const bool vertexColorEnabled = (detail::TestVertexColorEnabled(m_Content.vtxColors) || GetGlobalAlpha() != 255);
    const bool overallVertexColorEnabled = m_WindowFlags & WindowFlags_OverallVertexColorEnabled;
    for (int i = 0; i < 2; i++)
    {
        const Frame& frame = frames[i];
        if (vertexColorEnabled)
        {
            frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_Standard, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

            if (overallVertexColorEnabled)
            {
                detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
                detail::SetupVertexColor(drawInfo, *frame.pMaterial, m_Content.vtxColors);
            }
        }
        else
        {
            frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_WithoutVertexColor, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());

            if (overallVertexColorEnabled)
            {
                detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
            }
        }

        nn::util::Float2 polPt;
        Size polSize;

        if(i == WindowFrame_LeftTop)
        {
            detail::GetHorizontalLeftFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
        }else{
            detail::GetHorizontalRightFrameSize(&polPt, &polSize, basePt, GetSize(), frameSize);
        }

        // 「左上マテリアル使用」のエミュレーションデータ作成処理。
        // 左上マテリアルのコンスタントバッファをコピーして必要な部分だけ再設定している。
        if ((i > 0) &&
            (m_WindowFlags & WindowFlags_OneMaterialForAllFrameEnabled))
        {
            Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader = frame.pMaterial->GetConstantBufferForVertexShader(drawInfo);

            memcpy(pConstantBufferForVertexShader, frames[0].pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
            pConstantBufferForVertexShader->frameSpec = detail::DecideHorizontalNocontextFrameTransform(WindowFrame(i), frame.GetTextureFlip(), overallVertexColorEnabled);
            detail::CalculateQuad(drawInfo, pConstantBufferForVertexShader, polPt, polSize);
        }
        else
        {
            detail::SetupHorizontalNocontextFrameTransform(drawInfo, *frames[i].pMaterial, WindowFrame(i), frames[i].GetTextureFlip(), overallVertexColorEnabled);
            detail::CalculateQuad(drawInfo, frames[i].pMaterial->GetConstantBufferForVertexShader(drawInfo), polPt, polSize);
        }

        frames[i].pMaterial->SetAllowableTextureQuantity(drawInfo);
    }
}

//----------------------------------------
void
Window::CalculateFrame4(
    DrawInfo&               drawInfo,
    const nn::util::Float2& basePt,
    const Frame*            frames,
    const WindowFrameSize&  frameSize,
    uint8_t                 alpha
)
{
    nn::util::Float2 polPt[WindowFrame_RightBottom + 1];
    Size polSize[WindowFrame_RightBottom + 1];

    detail::GetLTFrameSize(&polPt[WindowFrame_LeftTop], &polSize[WindowFrame_LeftTop], basePt, GetSize(), frameSize);
    detail::GetRTFrameSize(&polPt[WindowFrame_RightTop], &polSize[WindowFrame_RightTop], basePt, GetSize(), frameSize);
    detail::GetLBFrameSize(&polPt[WindowFrame_LeftBottom], &polSize[WindowFrame_LeftBottom], basePt, GetSize(), frameSize);
    detail::GetRBFrameSize(&polPt[WindowFrame_RightBottom], &polSize[WindowFrame_RightBottom], basePt, GetSize(), frameSize);

    const bool OneMaterialForAllFrameEnabled = (m_WindowFlags & WindowFlags_OneMaterialForAllFrameEnabled) != 0;
    const bool OverallVertexColorEnabled = m_WindowFlags & WindowFlags_OverallVertexColorEnabled;
    const bool vertexColorEnabled = detail::TestVertexColorEnabled(m_Content.vtxColors) || GetGlobalAlpha() != 255;

    for (int i = WindowFrame_LeftTop; i <= WindowFrame_RightBottom; ++i)
    {
        const Frame& frame = frames[i];

        if (frame.pMaterial && frame.pMaterial->GetTexMapCount() > 0)
        {
            if (vertexColorEnabled)
            {
                frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_Standard, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());
                if (OverallVertexColorEnabled)
                {
                    detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
                    detail::SetupVertexColor(drawInfo, *frame.pMaterial, m_Content.vtxColors);
                }
            }
            else
            {
                frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_WithoutVertexColor, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());
                if (OverallVertexColorEnabled)
                {
                    detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
                }
            }

            // 「左上マテリアル使用」のエミュレーションデータ作成処理。
            // 左上マテリアルのコンスタントバッファをコピーして必要な部分だけ再設定している。
            if ((i > 0) && OneMaterialForAllFrameEnabled)
            {
                Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader = frame.pMaterial->GetConstantBufferForVertexShader(drawInfo);

                // テクスチャ情報一旦退避しておいて書き戻します。
                float tempcpTexSize0[4];
                memcpy(tempcpTexSize0, pConstantBufferForVertexShader->rcpTexSize0, sizeof(tempcpTexSize0));
                memcpy(pConstantBufferForVertexShader, frames[0].pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
                memcpy(pConstantBufferForVertexShader->rcpTexSize0, tempcpTexSize0, sizeof(tempcpTexSize0));

                detail::SetupFrame4Transform(drawInfo, *frame.pMaterial, WindowFrame(i), frame.GetTextureFlip(), OverallVertexColorEnabled);
            }
            else
            {
                // 通常のフレーム処理。
                detail::SetupFrame4Transform(drawInfo, *frame.pMaterial, WindowFrame(i), frame.GetTextureFlip(), OverallVertexColorEnabled);
            }

            detail::CalculateQuad(drawInfo, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), polPt[i], polSize[i]);
        }
    }
}

//----------------------------------------
void
Window::CalculateFrame8(
    DrawInfo&               drawInfo,
    const nn::util::Float2& basePt,
    const Frame*            frames,
    const WindowFrameSize&  frameSize,
    uint8_t                 alpha
)
{
    const float x0 = basePt.v[0];
    const float x1 = basePt.v[0] + frameSize.left;
    const float x2 = basePt.v[0] + this->GetSize().width - frameSize.right;

    const float y0 = basePt.v[1];
    const float y1 = basePt.v[1] - frameSize.top;
    const float y2 = basePt.v[1] - this->GetSize().height + frameSize.bottom;

    const float w0 = frameSize.left;
    const float w1 = this->GetSize().width - frameSize.left - frameSize.right;
    const float w2 = frameSize.right;

    const float h0 = frameSize.top;
    const float h1 = this->GetSize().height - frameSize.top - frameSize.bottom;
    const float h2 = frameSize.bottom;

    const detail::Rect frameRect[WindowFrame_MaxWindowFrame] =
    {
        { x0, y0, w0, h0 }, // LT
        { x2, y0, w2, h0 }, // RT
        { x0, y2, w0, h2 }, // LB
        { x2, y2, w2, h2 }, // RB
        { x0, y1, w0, h1 }, // L
        { x2, y1, w2, h1 }, // R
        { x1, y0, w1, h0 }, // T
        { x1, y2, w1, h2 }, // B
    };

    const bool OneMaterialForAllFrameEnabled = (m_WindowFlags & WindowFlags_OneMaterialForAllFrameEnabled) != 0;
    const bool OverallVertexColorEnabled = m_WindowFlags & WindowFlags_OverallVertexColorEnabled;
    const bool vertexColorEnabled = detail::TestVertexColorEnabled(m_Content.vtxColors) || GetGlobalAlpha() != 255;

    for (int i = 0; i < WindowFrame_MaxWindowFrame; ++i)
    {
        const Frame& frame = frames[i];

        if (frame.pMaterial->GetTexMapCount() > 0)
        {

            if (vertexColorEnabled)
            {
                frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_Standard, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());
                if (OverallVertexColorEnabled)
                {
                    detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
                    detail::SetupVertexColor(drawInfo, *frame.pMaterial, m_Content.vtxColors);
                }
            }
            else
            {
                frame.pMaterial->SetupGraphics(drawInfo, alpha, ShaderVariation_WithoutVertexColor, false, this->GetGlobalMtx(), &this->GetSize(), GetExtUserDataArray(), GetExtUserDataCount());
                if (OverallVertexColorEnabled)
                {
                    detail::SetupFrameSize(drawInfo, *frame.pMaterial, GetSize(), frameSize);
                }
            }

            // 「左上マテリアル使用」のエミュレーションデータ作成処理。
            // 左上マテリアルのコンスタントバッファをコピーして必要な部分だけ再設定している。
            if ((i > 0) && OneMaterialForAllFrameEnabled)
            {
                Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader = frame.pMaterial->GetConstantBufferForVertexShader(drawInfo);

                // テクスチャ情報一旦退避しておいて書き戻します。
                float tempcpTexSize0[4];
                memcpy(tempcpTexSize0, pConstantBufferForVertexShader->rcpTexSize0, sizeof(tempcpTexSize0));
                memcpy(pConstantBufferForVertexShader, frames[0].pMaterial->GetConstantBufferForVertexShader(drawInfo), sizeof(Material::ConstantBufferForVertexShader));
                memcpy(pConstantBufferForVertexShader->rcpTexSize0, tempcpTexSize0, sizeof(tempcpTexSize0));

                pConstantBufferForVertexShader->frameSpec = detail::DecideFrameTransform(WindowFrame(i), frame.GetTextureFlip(), OverallVertexColorEnabled);
            }
            else
            {
                detail::SetupFrameTransform(drawInfo, *frame.pMaterial, WindowFrame(i), frame.GetTextureFlip(), OverallVertexColorEnabled);
            }

            detail::CalculateQuad(drawInfo, frame.pMaterial->GetConstantBufferForVertexShader(drawInfo), frameRect[i].Position(), frameRect[i].Size());
        }
    }
}

} // namespace ui2d
} // namespace nn
