﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>

#include <nn/ui2d/ui2d_ShaderContainer.h>
#include <nn/ui2d/ui2d_Util.h>
#include <nn/ui2d/ui2d_Layout.h>
#include <nn/ui2d/ui2d_ShaderInfo.h>

namespace nn
{
namespace ui2d
{

//----------------------------------------
ShaderRefLink::ShaderRefLink(bool isOwned)
    : m_IsOwned(isOwned)
{
    m_Name[0] = '\0';
}

//----------------------------------------
ShaderRefLink::~ShaderRefLink()
{
}

//----------------------------------------
void ShaderRefLink::Finalize(nn::gfx::Device* pDevice)
{
    m_ShaderInfo.Finalize(pDevice, !m_IsOwned);
}

//----------------------------------------
void
ShaderRefLink::SetName(
    const char* pResourceName)
{
    NN_SDK_ASSERT_NOT_NULL(pResourceName);

    NN_SDK_ASSERT(std::strlen(pResourceName) < ShaderNameMax, "Invalid shader name[%s]", pResourceName);
    nn::util::Strlcpy(m_Name, pResourceName, sizeof(m_Name));
}

//----------------------------------------
ShaderContainer::~ShaderContainer()
{
    NN_SDK_ASSERT(this->empty(), "ShaderContainer must be empty before destruction.");
}

//----------------------------------------
void
ShaderContainer::Finalize(nn::gfx::Device* pDevice)
{
    while (!this->empty())
    {
        ShaderRefLink* pLink = &this->front();
        pLink->Finalize(pDevice);
        this->erase(this->iterator_to(*pLink));
        Layout::DeleteObj(pLink);
    }
}

//----------------------------------------
ShaderInfo*
ShaderContainer::RegisterShader(const char* pName, bool isOwned)
{
    NN_SDK_ASSERT(pName);

    ShaderRefLink* pLink = Layout::AllocateAndConstruct<ShaderRefLink>(isOwned);

    if (pLink == NULL)
    {
        return NULL;
    }

    pLink->SetName(pName);

    this->push_back(*pLink);

    return &pLink->GetShaderInfo();
}

//----------------------------------------
void
ShaderContainer::UnregisterShader(ShaderInfo* pShaderInfo)
{
    NN_SDK_ASSERT_NOT_NULL(pShaderInfo);

    for (iterator iter = this->begin(); iter != this->end(); ++iter)
    {
        if (&iter->GetShaderInfo() == pShaderInfo)
        {
            this->erase(iter);
            Layout::DeleteObj(&(*iter));
            break;
        }
    }
}

//----------------------------------------
const ShaderInfo*
ShaderContainer::FindShaderByName(const char* pName) const
{
    const_iterator endIter = this->end();
    for (const_iterator iter = this->begin(); iter != endIter; ++iter)
    {
        if (detail::EqualsShaderName(pName, iter->GetResourceName()))
        {
            return &iter->GetShaderInfo();
        }
    }

    return NULL;
}

} // namespace nn::ui2d
} // namespace nn
