﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_PaneEffect.h>
#include <nn/ui2d/ui2d_ResourceAccessor.h>
#include <nn/ui2d/ui2d_MaterialHelper.h>

namespace nn
{
namespace ui2d
{
namespace detail
{

// デフォルトの UV 設定
static const nn::util::Float2    DefaultTexCoords[4] =
{
    NN_UTIL_FLOAT_2_INITIALIZER(0.0f, 0.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(1.0f, 0.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(0.0f, 1.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(1.0f, 1.0f),
};

//-----------------------------------------------------------------------------------
PaneEffect::PaneEffect()
    : m_pTargetPane(NULL)
{
    memset(&m_MaskInfo, 0, sizeof(MaskInfo));
    memset(&m_DropShadowInfo, 0, sizeof(DropShadowInfo));
}

//-----------------------------------------------------------------------------------
void PaneEffect::Initialize(nn::gfx::Device* pDevice, const Pane* pTargetPane, const BuildArgSet& buildArgSet)
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    NN_SDK_ASSERT_NOT_NULL(pTargetPane);

    m_pTargetPane = pTargetPane;

    memset(&m_MaskInfo, 0, sizeof(MaskInfo));
    memset(&m_DropShadowInfo, 0, sizeof(DropShadowInfo));

    const SystemDataMaskTexture* pMaskData = m_pTargetPane->GetMaskData();
    const SystemDataDropShadow* pDropShadowData = m_pTargetPane->GetDropShadowData();

    // マスク機能を初期化
    if (pMaskData != NULL)
    {
        InitializeMaskFunction(pDevice, pMaskData, buildArgSet);
    }

    // ドロップシャドウ機能を初期化
    if (pDropShadowData)
    {
        InitializeDropShadowFunction(pDevice, pDropShadowData, buildArgSet);
    }
}


//-----------------------------------------------------------------------------------
PaneEffect::PaneEffect(const PaneEffect& src, const Pane* pTargetPane, nn::gfx::Device* pDevice, ResourceAccessor* pResAccessor, const char* pNewRootName, const Layout* pLayout)
    : m_pTargetPane(pTargetPane)
{
    NN_SDK_ASSERT_NOT_NULL(pTargetPane);
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    NN_SDK_ASSERT_NOT_NULL(pResAccessor);
    NN_SDK_ASSERT_NOT_NULL(pNewRootName);

    memset(&m_MaskInfo, 0, sizeof(MaskInfo));
    memset(&m_DropShadowInfo, 0, sizeof(DropShadowInfo));

    if (src.IsMaskFunctionEnabled())
    {
        // コピーして必要な要素を上書きする。
        memcpy(&m_MaskInfo, &src.m_MaskInfo, sizeof(MaskInfo));

        m_MaskInfo.pMaskData = m_pTargetPane->GetMaskData();

        if (detail::TestBit(m_MaskInfo.pMaskData->resMaskCaptureTexMapAdditionalInfo, ResTexMapAdditionalInfo::InfoType::InfoType_CaptureTextureEnabled))
        {
            m_MaskInfo.maskCaptureTexMap.Set(AcquireCaptureTexture(NULL, 0, pDevice, pResAccessor, pNewRootName, src.m_MaskInfo.maskCaptureTextureName));
        }

        if (detail::TestBit(m_MaskInfo.pMaskData->resMaskTexMapAdditionalInfo, ResTexMapAdditionalInfo::InfoType::InfoType_CaptureTextureEnabled))
        {
            m_MaskInfo.maskTexMap.Set(AcquireCaptureTexture(NULL, 0, pDevice, pResAccessor, pNewRootName, src.m_MaskInfo.maskTargetCaptureTextureName));
        }
    }

    if (src.IsDropShadowFunctionEnabled())
    {
        // コピーして必要な要素を上書きする。
        memcpy(&m_DropShadowInfo, &src.m_DropShadowInfo, sizeof(DropShadowInfo));

        m_DropShadowInfo.pDropShadowData = m_pTargetPane->GetDropShadowData();

        m_DropShadowInfo.captureTexMap.Set(AcquireCaptureTexture(NULL, 0, pDevice, pResAccessor, pNewRootName, src.m_DropShadowInfo.captureTextureName));

        // 描画結果キャッシュ済みフラグはリセットしておく。
        // コピー後にユーザーが任意の場所で作成するか、初回更新時に作成されることを期待。
        m_DropShadowInfo.staticRenderingCached = false;

        // 描画に使用する一時的なレンダーターゲットテクスチャを作成します。
        CreateDropShadowTempTextures(pDevice, pLayout, m_DropShadowInfo.pDropShadowData);
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::Finalize(nn::gfx::Device* pDevice)
{
    if (IsDropShadowFunctionEnabled())
    {
        for (int i = 0; i < SystemDataDropShadow::DropShadowType_Max; ++i)
        {
            if (m_DropShadowInfo.pStaticRenderingTextures[i] != NULL)
            {
                if (m_DropShadowInfo.pStaticRenderingTextures[i]->IsValid())
                {
                    m_DropShadowInfo.pStaticRenderingTextures[i]->Finalize(pDevice);
                }
                Layout::FreeMemory(m_DropShadowInfo.pStaticRenderingTextures[i]);
                m_DropShadowInfo.pStaticRenderingTextures[i] = NULL;
            }
        }

        if (m_DropShadowInfo.pBlurRenderTarget != NULL)
        {
            if (m_DropShadowInfo.pBlurRenderTarget->IsValid())
            {
                m_DropShadowInfo.pBlurRenderTarget->Finalize(pDevice);
            }
            Layout::FreeMemory(m_DropShadowInfo.pBlurRenderTarget);
            m_DropShadowInfo.pBlurRenderTarget = NULL;
        }

        if (m_DropShadowInfo.pMaskCopyRenderTarget != NULL)
        {
            if (m_DropShadowInfo.pMaskCopyRenderTarget->IsValid())
            {
                m_DropShadowInfo.pMaskCopyRenderTarget->Finalize(pDevice);
            }
            Layout::FreeMemory(m_DropShadowInfo.pMaskCopyRenderTarget);
            m_DropShadowInfo.pMaskCopyRenderTarget = NULL;
        }
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::Calculate(DrawInfo& drawInfo)
{
    const nn::util::MatrixT4x3fType modelView = m_pTargetPane->GetGlobalMtx();

    if (IsMaskFunctionEnabled())
    {
        drawInfo.SetModelViewMtx(modelView);
        CalculateMaskConstantBuffer(drawInfo);
    }

    if (IsDropShadowFunctionEnabled())
    {
        drawInfo.SetModelViewMtx(modelView);
        CalculateDropShadowConstantBuffer(drawInfo);
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::Draw(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    // ドロップシャドウが有効な場合はマスクの描画結果もキャプチャされているのでドロップシャドウの最終描画を優先する
    if (IsDropShadowFunctionEnabled())
    {
        // 内部で IsDropShadowStaticRenderingReady() を呼び出してキャッシュ有効時には不要な描画を行わないようにしています。
        // キャッシュを作成した同一フレームでキャッシュ結果を描画するために
        DrawDropShadow(drawInfo, commandBuffer);

        // キャッシュを使用した静的レンダリング。
        if (IsDropShadowStaticRenderingReady())
        {
            DrawStaticCachedShadow(drawInfo, commandBuffer);
        }
    }

    // ペイン本体の描画
    if (IsMaskFunctionEnabled())
    {
        if (!IsDropShadowFunctionEnabled() || !IsDropShadowEffectOnlyDrawEnabled())
        {
            DrawMaskedImage(drawInfo, commandBuffer);
        }
    }
    else if (IsDropShadowFunctionEnabled())
    {
        if (!IsDropShadowEffectOnlyDrawEnabled())
        {
            // オリジナルペインのコピー画像をレンダリングする
            DrawDropShadowOriginalImage(drawInfo, commandBuffer);
        }
    }
}

//-----------------------------------------------------------------------------------
size_t PaneEffect::GetRequiredConstantBufferSize(nn::gfx::Device* pDevice) const
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);

    size_t requiredConstantBufferSize = 0;

    if (m_MaskInfo.pMaskData != NULL)
    {
        requiredConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, sizeof(Material::ConstantBufferForVertexShader));
    }

    if (m_DropShadowInfo.pDropShadowData != NULL)
    {
        // マスクとドロップシャドウが併用されている場合はマスクの描画結果をキャプチャするための描画を行う
        if (m_MaskInfo.pMaskData != NULL)
        {
            requiredConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, sizeof(Material::ConstantBufferForVertexShader));
        }

        // 各エフェクトごとに横ブラーと影描画用
        for (int i = 0; i < SystemDataDropShadow::DropShadowType_Max; ++i)
        {
            if ((m_DropShadowInfo.pDropShadowData->flags & (1 << i)) != 0)
            {
                // 縦ブラーと横ブラー用の 2 つ必要。
                int vertexShaderBufferCount = 2;

                // キャッシュを使用する場合は同一フレームでキャプチャと描画を行う可能性があるため追加でもう一つ必要
                if (IsDropShadowStaticRenderingEnabled())
                {
                    ++vertexShaderBufferCount;
                }

                // VertexShader
                requiredConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, sizeof(Material::ConstantBufferForVertexShader)) * vertexShaderBufferCount;
                // PixelShader
                requiredConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, sizeof(detail::DropShadowBlurConstantBuffer));
            }
        }

        // 最終描画
        requiredConstantBufferSize += GetAlignedBufferSize(pDevice, nn::gfx::GpuAccess_ConstantBuffer, sizeof(Material::ConstantBufferForVertexShader));
    }

    return requiredConstantBufferSize;
}

//-----------------------------------------------------------------------------------
size_t PaneEffect::SetupEffectDrawVertexShaderConstantBuffer(DrawInfo& drawInfo)
{
    NN_SDK_ASSERT_NOT_NULL(drawInfo.GetGraphicsResource());
    size_t  constantBufferAlignment = drawInfo.GetGraphicsResource()->GetConstantBufferAlignment();

    size_t  offset = drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(Material::ConstantBufferForVertexShader), constantBufferAlignment));
    NN_SDK_ASSERT(offset <= size_t(0xFFFFFFFF), "offset is over the limit for Pane[%s] pane effect function.", m_pTargetPane->GetName());

    // ピクセルシェーダーのコンスタントバッファは使用しないため確保しない。

    // コンスタントバッファを作成する
    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL)
    {
        nn::util::BytePtr   pConstantBufferForVertexShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        Material::ConstantBufferForVertexShader* pVertexShaderConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pConstantBufferForVertexShader.Advance(offset).Get());

        memset(pVertexShaderConstantBuffer, 0, sizeof(Material::ConstantBufferForVertexShader));

        for (int i = 0; i < 4; ++i)
        {
            for (int j = 0; j < 4; ++j)
            {
                pVertexShaderConstantBuffer->vertexColor[i][j] = 255.0f;
            }
        }

        nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pVertexShaderConstantBuffer->modelView), m_pTargetPane->GetGlobalMtx());

        // CalculateTextureMtx で取得した変換なしの行列
        float texMtx[2][3] =
        {
            { 1.0f, 0.0f, 0.0f },
            { 0.0f, -1.0f, 1.0f }
        };

        LoadTexCoordMatrix(pVertexShaderConstantBuffer, 0, texMtx);
        LoadTexCoordMatrix(pVertexShaderConstantBuffer, 1, texMtx);
        LoadTexCoordMatrix(pVertexShaderConstantBuffer, 2, texMtx);

        drawInfo.SetTexCoordSrc(0, 0);
        drawInfo.SetTexCoordSrc(1, 0);
        drawInfo.SetTexCoordSrc(2, 0);

        for (int i = 0; i < 4; ++i)
        {
            pVertexShaderConstantBuffer->color[i] = 1.0f / 255.0f;
        }

        drawInfo.LoadProjectionMtx(pVertexShaderConstantBuffer->projection);
    }

    return offset;
}

//-----------------------------------------------------------------------------------
void PaneEffect::InitializeMaskFunction(
    nn::gfx::Device* pDevice,
    const SystemDataMaskTexture* pMaskData,
    const BuildArgSet& buildArgSet)
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    NN_SDK_ASSERT_NOT_NULL(pMaskData);

    const BuildResSet* pBuildResSet = buildArgSet.pCurrentBuildResSet;

    NN_SDK_ASSERT_NOT_NULL(pBuildResSet);

    const ResTexture *const pResTextures = nn::util::ConstBytePtr(pBuildResSet->pTextureList, sizeof(*buildArgSet.pCurrentBuildResSet->pTextureList)).Get<ResTexture>();

    // 内部利用でメモリの寿命をコントロールする前提でポインタコピーして使用する。
    m_MaskInfo.pMaskData = pMaskData;

    // リソースからキャプチャテクスチャの TexMap を作成する
    const char* pFileName = nn::util::ConstBytePtr(pResTextures, pResTextures[pMaskData->resMaskCaptureTexMap.texIdx].nameStrOffset).Get<char>();
    if (detail::TestBit(pMaskData->resMaskCaptureTexMapAdditionalInfo, ResTexMapAdditionalInfo::InfoType::InfoType_CaptureTextureEnabled))
    {
        // 自動生成されるキャプチャテクスチャは上書きされることがないため必ずローカルのキャプチャテクスチャで解決する。
        bool overrided = false;
        m_MaskInfo.maskCaptureTexMap.Set(AcquireCaptureTextureWithResolvePrefix(m_MaskInfo.maskCaptureTextureName, CaptureTexturePathMax, buildArgSet, overrided, pDevice, pBuildResSet->pResAccessor, pFileName));
    }
    else
    {
        m_MaskInfo.maskCaptureTexMap.Set(pBuildResSet->pResAccessor->AcquireTexture(pDevice, pFileName));
    }
    m_MaskInfo.maskCaptureTexMap.SetWrapMode(pMaskData->resMaskCaptureTexMap.GetWarpModeS(), pMaskData->resMaskCaptureTexMap.GetWarpModeT());
    m_MaskInfo.maskCaptureTexMap.SetFilter(pMaskData->resMaskCaptureTexMap.GetMinFilter(), pMaskData->resMaskCaptureTexMap.GetMagFilter());

    // リソースからマスクテクスチャの TexMap を作成する
    pFileName = nn::util::ConstBytePtr(pResTextures, pResTextures[pMaskData->resMaskTexMap.texIdx].nameStrOffset).Get<char>();
    if (detail::TestBit(pMaskData->resMaskTexMapAdditionalInfo, ResTexMapAdditionalInfo::InfoType::InfoType_CaptureTextureEnabled))
    {
        bool overrided = false;
        m_MaskInfo.maskTexMap.Set(AcquireCaptureTextureWithResolvePrefix(m_MaskInfo.maskTargetCaptureTextureName, CaptureTexturePathMax, buildArgSet, overrided, pDevice, pBuildResSet->pResAccessor, pFileName));
    }
    else
    {
        m_MaskInfo.maskTexMap.Set(pBuildResSet->pResAccessor->AcquireTexture(pDevice, pFileName));
    }
    m_MaskInfo.maskTexMap.SetWrapMode(pMaskData->resMaskTexMap.GetWarpModeS(), pMaskData->resMaskTexMap.GetWarpModeT());
    m_MaskInfo.maskTexMap.SetFilter(pMaskData->resMaskTexMap.GetMinFilter(), pMaskData->resMaskTexMap.GetMagFilter());

    // マスク機能用のアーカイブシェーダーを取得します。
    char archiveShaderName[8];
    ConvertBlendsToArchiveShaderName(archiveShaderName, IndividualArchiveShaderVariationTableFirstBlendWithMaskShader, 0);
    m_MaskInfo.pShaderInfo = buildArgSet.pLayout->AcquireArchiveShader(pDevice, archiveShaderName);
    NN_SDK_ASSERT(m_MaskInfo.pShaderInfo != NULL, "ShaderInfo[%s] not found for InitializeMaskFunction[%s].", archiveShaderName, m_pTargetPane->GetName());

    const void* pVariationTable = m_MaskInfo.pShaderInfo->GetVariationTable();
    NN_SDK_ASSERT(pVariationTable != NULL, "VariationTable not found for PaneEffect::InitializeMaskFunction[%s].", m_pTargetPane->GetName());

    // バリエーションを検索して設定
    m_MaskInfo.shaderVariationIndex = SearchShaderVariationIndexFromTable(pVariationTable, IndividualArchiveShaderVariationTableFirstBlendWithMaskShader, 0);
}

//-----------------------------------------------------------------------------------
void PaneEffect::ApplyMaskParameters(Material::ConstantBufferForVertexShader* pConstantBuffer)
{
    float texMtx[2][3];

    Material::CalculateTextureMtx(texMtx, m_MaskInfo.pMaskData->maskTexSrt, m_MaskInfo.maskTexMap);

    if (m_MaskInfo.pMaskData->maskWithPane)
    {
        LoadTexCoordMatrix(pConstantBuffer, 0, texMtx);
    }
    else
    {
        LoadTexCoordMatrix(pConstantBuffer, 1, texMtx);
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::CalculateMaskConstantBuffer(DrawInfo& drawInfo)
{
    m_MaskInfo.vertexShaderConstantBufferOffset = static_cast<uint32_t>(SetupEffectDrawVertexShaderConstantBuffer(drawInfo));

    // コンスタントバッファを作成する
    if (drawInfo.GetUi2dConstantBuffer()->GetMappedPointer() != NULL)
    {
        nn::util::BytePtr   pConstantBufferForVertexShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        Material::ConstantBufferForVertexShader* pVertexShaderConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pConstantBufferForVertexShader.Advance(m_MaskInfo.vertexShaderConstantBufferOffset).Get());

        // マスクパラメータをコンスタントバッファへ反映する
        ApplyMaskParameters(pVertexShaderConstantBuffer);

        detail::CalculateQuadWithTexCoords(
            drawInfo,
            pVertexShaderConstantBuffer,
            m_pTargetPane->GetVertexPos(),
            m_pTargetPane->GetSize(),
            1,
            &DefaultTexCoords);
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::SetupMaskShader(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    SetupShader(drawInfo, commandBuffer, m_MaskInfo.pShaderInfo, m_MaskInfo.shaderVariationIndex);

    nn::gfx::DescriptorSlot maskTexSampler =
        drawInfo.GetGraphicsResource()->GetSamplerDescriptorSlot(
            m_MaskInfo.maskTexMap.GetWrapModeS(),
            m_MaskInfo.maskTexMap.GetWrapModeT(),
            m_MaskInfo.maskTexMap.GetMinFilter(),
            m_MaskInfo.maskTexMap.GetMagFilter());

    nn::gfx::DescriptorSlot maskCaptureTexSampler =
        drawInfo.GetGraphicsResource()->GetSamplerDescriptorSlot(
            m_MaskInfo.maskCaptureTexMap.GetWrapModeS(),
            m_MaskInfo.maskCaptureTexMap.GetWrapModeT(),
            m_MaskInfo.maskCaptureTexMap.GetMinFilter(),
            m_MaskInfo.maskCaptureTexMap.GetMagFilter());

    // マスクシェーダー
    //  vec4 color = texture(uTexture0, vUv0);
    //  color.a *= texture(uTexture1, vUv1).a;
    if (m_MaskInfo.pMaskData->maskWithPane)
    {
        //  ペインの描画結果の形状でマスクに設定したテクスチャをマスクする
        commandBuffer.SetTextureAndSampler(m_MaskInfo.pShaderInfo->GetTextureSlot(m_MaskInfo.shaderVariationIndex, 0), nn::gfx::ShaderStage_Pixel, *m_MaskInfo.maskTexMap.GetTextureInfo()->GetTextureDescriptorSlot(), maskTexSampler);
        commandBuffer.SetTextureAndSampler(m_MaskInfo.pShaderInfo->GetTextureSlot(m_MaskInfo.shaderVariationIndex, 1), nn::gfx::ShaderStage_Pixel, *m_MaskInfo.maskCaptureTexMap.GetTextureInfo()->GetTextureDescriptorSlot(), maskCaptureTexSampler);
    }
    else
    {
        //  マスクに設定したテクスチャの形状でマスクする
        commandBuffer.SetTextureAndSampler(m_MaskInfo.pShaderInfo->GetTextureSlot(m_MaskInfo.shaderVariationIndex, 0), nn::gfx::ShaderStage_Pixel, *m_MaskInfo.maskCaptureTexMap.GetTextureInfo()->GetTextureDescriptorSlot(), maskCaptureTexSampler);
        commandBuffer.SetTextureAndSampler(m_MaskInfo.pShaderInfo->GetTextureSlot(m_MaskInfo.shaderVariationIndex, 1), nn::gfx::ShaderStage_Pixel, *m_MaskInfo.maskTexMap.GetTextureInfo()->GetTextureDescriptorSlot(), maskTexSampler);
    }

    commandBuffer.SetBlendState(drawInfo.GetGraphicsResource()->GetPresetBlendState(PresetBlendStateId_OpaqueOrAlphaTest));
}

//-----------------------------------------------------------------------------------
void PaneEffect::DrawMaskedImage(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    SetupMaskShader(drawInfo, commandBuffer);
    m_pTargetPane->SetupPaneEffectSourceImageRenderState(commandBuffer);
    DrawSpriteCommonImpl(drawInfo, commandBuffer, m_MaskInfo.pShaderInfo, 0, m_MaskInfo.vertexShaderConstantBufferOffset);
}


//-----------------------------------------------------------------------------------
void PaneEffect::InitializeDropShadowFunction(
    nn::gfx::Device* pDevice,
    const SystemDataDropShadow* pDropShadowData,
    const BuildArgSet& buildArgSet)
{
    // 内部利用でメモリの寿命をコントロールする前提でポインタコピーして使用する。
    m_DropShadowInfo.pDropShadowData = pDropShadowData;

    const BuildResSet* pBuildResSet = buildArgSet.pCurrentBuildResSet;
    const ResTexture *const pResTextures = nn::util::ConstBytePtr(pBuildResSet->pTextureList, sizeof(*buildArgSet.pCurrentBuildResSet->pTextureList)).Get<ResTexture>();

    const char* pFileName = nn::util::ConstBytePtr(pResTextures, pResTextures[pDropShadowData->resSourceCapturedImage.texIdx].nameStrOffset).Get<char>();
    // 自動生成されるキャプチャテクスチャは上書きされることがないため必ずローカルのキャプチャテクスチャで解決する。
    bool overrided = false;
    m_DropShadowInfo.captureTexMap.Set(AcquireCaptureTextureWithResolvePrefix(m_DropShadowInfo.captureTextureName, CaptureTexturePathMax, buildArgSet, overrided, pDevice, pBuildResSet->pResAccessor, pFileName));
    m_DropShadowInfo.captureTexMap.SetWrapMode(pDropShadowData->resSourceCapturedImage.GetWarpModeS(), pDropShadowData->resSourceCapturedImage.GetWarpModeT());
    m_DropShadowInfo.captureTexMap.SetFilter(pDropShadowData->resSourceCapturedImage.GetMinFilter(), pDropShadowData->resSourceCapturedImage.GetMagFilter());

    // ドロップシャドウ用のアーカイブシェーダーを取得します。
    char archiveShaderName[8];
    ConvertBlendsToArchiveShaderName(archiveShaderName, IndividualArchiveShaderVariationTableFirstBlendWithDropShadow, 0);
    m_DropShadowInfo.pShaderInfo = buildArgSet.pLayout->AcquireArchiveShader(pDevice, archiveShaderName);

    // 描画に使用する一時的なレンダーターゲットテクスチャを作成します。
    CreateDropShadowTempTextures(pDevice, buildArgSet.pLayout, pDropShadowData);

    const void* pVariationTable = m_DropShadowInfo.pShaderInfo->GetVariationTable();
    NN_SDK_ASSERT(pVariationTable != NULL, "VariationTable not found for PaneEffect::InitializeDropShadowFunction[%s].", m_pTargetPane->GetName());

    // ドロップシャドウ用のシェーダー ID
    const int dropShadowSecondKeys[detail::DropShadowShaderVariation_Max] =
    {
        0, 1, 2, 3,         // ドロップシャドウ横ブラーシェーダー
        4, 5, 6, 7,         // ドロップシャドウ縦ブラーシェーダー
        14, 15, 16, 17,     // ドロップシャドウ縦ブラー＆アルファ付き乗算ブレンドシェーダー
        104, 105, 106, 107, // ドロップシャドウ縦ブラー＆ノックアウトシェーダー
        114, 115, 116, 117, // ドロップシャドウ縦ブラー＆ノックアウト＆アルファ付き乗算ブレンドシェーダー
        8, 18               // ペイン本体の最終描画用コピーシェーダー
    };

    // バリエーションを検索して設定
    for (int i = 0; i < detail::DropShadowShaderVariation_Max; ++i)
    {
        m_DropShadowInfo.dropShadowShaderVariationIndex[i] = SearchShaderVariationIndexFromTable(pVariationTable, IndividualArchiveShaderVariationTableFirstBlendWithDropShadow, dropShadowSecondKeys[i]);
        m_DropShadowInfo.dropShadowBlurPixelShaderCB[i] = m_DropShadowInfo.pShaderInfo->GetPixelShader(m_DropShadowInfo.dropShadowShaderVariationIndex[i])->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "uDropShadowBlur");
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::CreateDropShadowTempTextures(nn::gfx::Device* pDevice, const Layout* pLayout, const SystemDataDropShadow* pDropShadowData)
{
    if (IsMaskFunctionEnabled())
    {
        m_DropShadowInfo.pMaskCopyRenderTarget = Layout::AllocateAndConstruct<RenderTargetTextureInfo>();

        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(static_cast<int>(m_pTargetPane->GetSize().width));
        info.SetHeight(static_cast<int>(m_pTargetPane->GetSize().height));
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetMipCount(1);

        m_DropShadowInfo.pMaskCopyRenderTarget->Initialize(pDevice, pLayout, info, RenderTargetTextureLifetime_Layout);
    }
    else
    {
        m_DropShadowInfo.pMaskCopyRenderTarget = NULL;
    }

    {
        m_DropShadowInfo.pBlurRenderTarget = Layout::AllocateAndConstruct<RenderTargetTextureInfo>();

        int blurExtendSize = pDropShadowData->blurSizeMax * 2;

        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(static_cast<int>(m_pTargetPane->GetSize().width + blurExtendSize));
        info.SetHeight(static_cast<int>(m_pTargetPane->GetSize().height + blurExtendSize));
        info.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetMipCount(1);

        m_DropShadowInfo.pBlurRenderTarget->Initialize(pDevice, pLayout, info, RenderTargetTextureLifetime_Layout);
    }

    if (IsDropShadowStaticRenderingEnabled())
    {
        for (int i = 0; i < SystemDataDropShadow::DropShadowType_Max; ++i)
        {
            if (!detail::TestBit(m_DropShadowInfo.pDropShadowData->flags, i))
            {
                m_DropShadowInfo.pStaticRenderingTextures[i] = NULL;
                continue;
            }

            m_DropShadowInfo.pStaticRenderingTextures[i] = Layout::AllocateAndConstruct<RenderTargetTextureInfo>();

            int blurExtendSize = pDropShadowData->blurSizeMax * 2;

            nn::gfx::Texture::InfoType info;
            info.SetDefault();
            info.SetWidth(static_cast<int>(m_pTargetPane->GetSize().width + blurExtendSize));
            info.SetHeight(static_cast<int>(m_pTargetPane->GetSize().height + blurExtendSize));
            info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
            info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            info.SetMipCount(1);

            m_DropShadowInfo.pStaticRenderingTextures[i]->Initialize(pDevice, pLayout, info, RenderTargetTextureLifetime_Layout);
        }
    }
}

//-----------------------------------------------------------------------------------
int PaneEffect::MakeBlurPixelShaderConstantBuffer(DrawInfo& drawInfo, int cbOffset, const detail::BlurParams& blurParams) const
{
    nn::util::BytePtr   pBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
    detail::DropShadowBlurConstantBuffer* pCB = static_cast<detail::DropShadowBlurConstantBuffer *>(pBuffer.Advance(cbOffset).Get());

    memset(pCB, 0, sizeof(detail::DropShadowBlurConstantBuffer));

    // スプレッドの逆数が最大アルファ値になるように係数を設定
    float spread = (1.0f - (blurParams.spread / 100.0f));
    // ゼロ除算を避けるため極小の値を設定しておく
    if (spread == 0.0f)
    {
        spread = 0.0039f;
    }
    pCB->spread = 1.0f / spread;

    // カラーバッファを埋める色を設定する
    pCB->color[0] = blurParams.color[0] / 255.0f;
    pCB->color[1] = blurParams.color[1] / 255.0f;
    pCB->color[2] = blurParams.color[2] / 255.0f;
    pCB->color[3] = blurParams.color[3] / 255.0f;

    // ブラーのためのウェイトを計算する
    // 最大 63 タップだが、必要に応じて前半から値を埋めている
    if (blurParams.size != 0.0f)
    {
        float   sum = 0.0f;
        float   weights[MaxDropShadowBlurRadius];

        memset(weights, 0, sizeof(weights));

        for (int i = 0; i < MaxDropShadowBlurRadius; ++i)
        {
            // ガウス関数でウエイトテーブルを計算する
            // ガウス関数の基準の形をサイズに応じて適用範囲を変化させてぼかし幅を調整します。
            // 2.8 は最大で標準偏差何個分の領域まで使用するかを調整するパラメータで、描画結果を見て調整しています。
            float length = static_cast<float>(i) / blurParams.size * 2.8f;
            weights[i] = expf(-0.5f * length*length);

            if (i == 0)
            {
                sum += weights[i];
            }
            else
            {
                // 中心以外は反転した位置も含めて 2 回サンプリングされるためウエイトへの影響を倍にする。
                sum += weights[i] * 2.0f;
            }
        }

        // 正規化
        for (int i = 0; i < MaxDropShadowBlurRadius; ++i)
        {
            weights[i] /= sum;
        }

        // 正規化したテーブルをウエイトを考慮したサンプラ補間用に変換する
        // 隣同士のピクセルのウエイトの割合に応じてサンプリングする UV をずらして 1 回のフェッチで 2タップ処理する。

        // 横方向はオリジナルのキャプチャテクスチャからサンプリングする
        float texelStepU = 1.0f / static_cast<float>(m_pTargetPane->GetSize().width);
        // 縦方向はブラー領域込みのテクスチャからサンプリングする
        float texelStepV = 1.0f / static_cast<float>(m_DropShadowInfo.pBlurRenderTarget->GetSize().height);

        // 中心ピクセルのウエイト
        pCB->centerWeight = weights[0];

        for (int i = 0; i < MaxDropShadowBlurRadius / 2; ++i)
        {
            int left = 1 + i * 2;
            if (left + 1 < MaxDropShadowBlurRadius)
            {
                // 隣り合ったテクセルのウエイトからサンプリングする UV をずらす
                float total = weights[left] + weights[left + 1];
                float texelOffset = 0.0f;
                if (total > 0.0f)
                {
                    texelOffset = static_cast<float>(left) + (weights[left + 1] / total);
                }
                pCB->offsetsH[i] = texelOffset * texelStepU;
                pCB->offsetsV[i] = texelOffset * texelStepV;
                pCB->weights[i] = total;
            }
            else
            {
                // 右側にピクセルがないため左側だけサンプリングする設定にする
                pCB->offsetsH[i] = static_cast<float>(left) * texelStepU;
                pCB->offsetsV[i] = static_cast<float>(left) * texelStepV;
                pCB->weights[i] = weights[left];
            }
        }
    }
    else
    {
        memset(pCB->weights, 0, sizeof(pCB->weights));
        pCB->weights[0] = 1.0f;
    }

    float globalAlpha = static_cast<float>(m_pTargetPane->GetGlobalAlpha());

    // 影響のない値をデフォルトで設定しておく。
    pCB->paneTransparency = 1.0f;
    pCB->inversePaneTransparency = 1.0f;

    if (globalAlpha > 0.0f &&
        !IsDropShadowStaticRenderingEnabled())
    {
        pCB->paneTransparency = globalAlpha / 255.0f;
        pCB->inversePaneTransparency = 255.0f / globalAlpha;
    }

    // タップサイズを計算して返す。
    int tapSize = 0;
    for (int i = 0; i < MaxDropShadowBlurRadius / 2; ++i)
    {
        const float thresholdValue = (1.0f / 256.0f) / (static_cast<float>(MaxDropShadowBlurRadius) * 2.0f);
        if (pCB->weights[i] < thresholdValue)
        {
            break;
        }
        ++tapSize;
    }

    return tapSize;
}


//-----------------------------------------------------------------------------------
void PaneEffect::MakeHorizontalBlurConstantBuffer(DrawInfo& drawInfo, int cbOffset, const nn::util::Float2& blurTextureCenter, const Size& blurTextureSize, const nn::util::Float2& marginOffset) const
{
    // ブラー用テクスチャのサイズでレンダリングされます。
    nn::util::BytePtr   pBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
    Material::ConstantBufferForVertexShader* pConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pBuffer.Advance(cbOffset).Get());

    // ブラー用の拡張領域のサイズでレンダリングして中心にオリジナルサイズのキャプチャ結果が描画されるように UV を調整する。
    nn::util::Float2    arrangedTexCoords[4];
    memset(arrangedTexCoords, 0, sizeof(nn::util::Float2) * 4);

    arrangedTexCoords[0].x = 0.0f - marginOffset.x;
    arrangedTexCoords[1].x = 1.0f + marginOffset.x;
    arrangedTexCoords[2].x = 0.0f - marginOffset.x;
    arrangedTexCoords[3].x = 1.0f + marginOffset.x;
    arrangedTexCoords[0].y = 0.0f - marginOffset.y;
    arrangedTexCoords[1].y = 0.0f - marginOffset.y;
    arrangedTexCoords[2].y = 1.0f + marginOffset.y;
    arrangedTexCoords[3].y = 1.0f + marginOffset.y;

    detail::CalculateQuadWithTexCoords(drawInfo, pConstantBuffer, blurTextureCenter, blurTextureSize, 1, &arrangedTexCoords);

    // CalculateQuadWithTexCoords の中で LoadMtxModelView が行われているため、呼び出し後に上書きする
    nn::util::MatrixT4x4fType   projMtx;
    nn::util::MatrixOrthographicOffCenterRightHanded(&projMtx, blurTextureCenter.x, blurTextureCenter.x + blurTextureSize.width, blurTextureCenter.y, blurTextureCenter.y - blurTextureSize.height, 0.0f, 500.0f);

    nn::util::MatrixT4x3fType glbMtx;

    detail::CalculateCaptureRootMatrix(glbMtx, drawInfo);

    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(pConstantBuffer->projection), projMtx);
    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pConstantBuffer->modelView), glbMtx);
}


//-----------------------------------------------------------------------------------
void PaneEffect::MakeVerticalBlurConstantBuffer(
    DrawInfo& drawInfo,
    int cbOffset,
    const BlurParams& blurParams,
    const nn::util::Float2& blurTextureCenter,
    const Size& blurTextureSize,
    const nn::util::Float2& blurTextureScale,
    const nn::util::Float2& marginOffset) const
{
    nn::util::BytePtr   pBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
    Material::ConstantBufferForVertexShader* pConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pBuffer.Advance(cbOffset).Get());

    nn::util::Float2    drawPos = m_pTargetPane->GetVertexPos();

    // 描画位置をブラーサイズ分大きくなるようにずらす
    drawPos.x -= static_cast<float>(m_DropShadowInfo.pDropShadowData->blurSizeMax);
    drawPos.y += static_cast<float>(m_DropShadowInfo.pDropShadowData->blurSizeMax);

    float offsetXByPixel = 0.0f;
    float offsetYByPixel = 0.0f;

    if (blurParams.distance > 0.0f)
    {
        float radAngle = nn::util::DegreeToRadian(blurParams.angle);
        offsetXByPixel = cosf(radAngle) * blurParams.distance;
        offsetYByPixel = sinf(radAngle) * blurParams.distance;
        drawPos.x -= offsetXByPixel;
        drawPos.y -= offsetYByPixel;
    }

    nn::util::Float2    arrangedTexCoords[2][4];
    memset(arrangedTexCoords, 0, sizeof(nn::util::Float2) * 2 * 4);
    float offsetXByTexCoord = offsetXByPixel / static_cast<float>(blurTextureSize.width) * blurTextureScale.x;
    float offsetYByTexCoord = offsetYByPixel / static_cast<float>(blurTextureSize.height) * blurTextureScale.y;

    memcpy(arrangedTexCoords, DefaultTexCoords, sizeof(nn::util::Float2) * 4);

    // TexCoord の 1 番にノックアウト用のテクスチャ用の UV を設定します。
    arrangedTexCoords[1][0].x = 0.0f - marginOffset.x - offsetXByTexCoord;
    arrangedTexCoords[1][1].x = 1.0f + marginOffset.x - offsetXByTexCoord;
    arrangedTexCoords[1][2].x = 0.0f - marginOffset.x - offsetXByTexCoord;
    arrangedTexCoords[1][3].x = 1.0f + marginOffset.x - offsetXByTexCoord;
    arrangedTexCoords[1][0].y = 0.0f - marginOffset.y + offsetYByTexCoord;
    arrangedTexCoords[1][1].y = 0.0f - marginOffset.y + offsetYByTexCoord;
    arrangedTexCoords[1][2].y = 1.0f + marginOffset.y + offsetYByTexCoord;
    arrangedTexCoords[1][3].y = 1.0f + marginOffset.y + offsetYByTexCoord;

    drawInfo.SetTexCoordSrc(0, 0);
    drawInfo.SetTexCoordSrc(1, 1);

    // 描画結果をキャプチャして使用する場合はキャプチャ用のバッファ全体に出力するため出力サイズを調整する。
    if (IsDropShadowStaticRenderingEnabled())
    {
        // ブラー幅を拡大したテクスチャの中心にレンダリングする。
        detail::CalculateQuadWithTexCoords(drawInfo, pConstantBuffer, blurTextureCenter, blurTextureSize, 2, arrangedTexCoords);

        // CalculateQuadWithTexCoords の中で LoadMtxModelView が行われているため、呼び出し後に上書きする
        nn::util::MatrixT4x4fType   projMtx;
        nn::util::MatrixOrthographicOffCenterRightHanded(&projMtx, blurTextureCenter.x, blurTextureCenter.x + blurTextureSize.width, blurTextureCenter.y, blurTextureCenter.y - blurTextureSize.height, 0.0f, 500.0f);

        nn::util::MatrixT4x3fType glbMtx;

        detail::CalculateCaptureRootMatrix(glbMtx, drawInfo);

        nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(pConstantBuffer->projection), projMtx);
        nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pConstantBuffer->modelView), glbMtx);
    }
    else
    {
        detail::CalculateQuadWithTexCoords(drawInfo, pConstantBuffer, drawPos, blurTextureSize, 2, arrangedTexCoords);
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::MakeStaticRenderingConstantBuffer(DrawInfo& drawInfo, int cbOffset, const BlurParams& blurParams, const Size& blurTextureSize) const
{
    nn::util::BytePtr   pBuffer(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
    Material::ConstantBufferForVertexShader* pConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pBuffer.Advance(cbOffset).Get());

    nn::util::Float2    drawPos = m_pTargetPane->GetVertexPos();

    if (blurParams.distance > 0.0f)
    {
        float radAngle = nn::util::DegreeToRadian(blurParams.angle);
        drawPos.x -= cosf(radAngle) * blurParams.distance;
        drawPos.y -= sinf(radAngle) * blurParams.distance;
    }

    // 描画位置をブラーサイズ分大きくなるようにずらす
    drawPos.x -= static_cast<float>(m_DropShadowInfo.pDropShadowData->blurSizeMax);
    drawPos.y += static_cast<float>(m_DropShadowInfo.pDropShadowData->blurSizeMax);

    // 静的レンダリング時に透明度アニメーションを反映させるためグローバルアルファを適用します。
    pConstantBuffer->color[3] = static_cast<float>(m_pTargetPane->GetGlobalAlpha()) / 255.0f / 255.0f;

    detail::CalculateQuadWithTexCoords(drawInfo, pConstantBuffer, drawPos, blurTextureSize, 1, &DefaultTexCoords);
}

//-----------------------------------------------------------------------------------
int PaneEffect::MakeDropShadowConstantBufferSet(DrawInfo& drawInfo, SystemDataDropShadow::DropShadowType type)
{
    detail::BlurParams  blurParams;

    // コンスタントバッファ用メモリを確保する
    AllocateDropShadowEffectConstantBufferSet(drawInfo, type);

    // ブラー用のパラメータを作成する
    MakeBlurParams(&blurParams, m_DropShadowInfo.pDropShadowData, type);

    int tapSize = 0;

    nn::ui2d::Size  blurTextureSize;
    blurTextureSize.width = m_DropShadowInfo.pBlurRenderTarget->GetSize().width;
    blurTextureSize.height = m_DropShadowInfo.pBlurRenderTarget->GetSize().height;

    // キャッシュを使用した静的レンダリングの時は作成する必要がない。
    if (!IsDropShadowStaticRenderingReady())
    {
        nn::util::Float2    blurTextureCenter;
        blurTextureCenter.x = -blurTextureSize.width * 0.5f;
        blurTextureCenter.y = blurTextureSize.height * 0.5f;

        // 元のキャプチャ画像を基準としたブラー用テクスチャの縦横のスケール
        nn::util::Float2    blurTextureScale;
        blurTextureScale.x = static_cast<float>(blurTextureSize.width) / static_cast<float>(m_pTargetPane->GetSize().width);
        blurTextureScale.y = static_cast<float>(blurTextureSize.height) / static_cast<float>(m_pTargetPane->GetSize().height);

        // ブラー用テクスチャサイズでオリジナルテクスチャのサイズを扱う際の UV オフセット
        nn::util::Float2    marginOffset;
        marginOffset.x = (blurTextureScale.x - 1.0f) * 0.5f;
        marginOffset.y = (blurTextureScale.y - 1.0f) * 0.5f;

        // 横ブラー用描画用の頂点シェーダーコンスタントバッファを作成する
        MakeHorizontalBlurConstantBuffer(drawInfo, m_DropShadowInfo.horizontalBlurVertexShaderCB[type], blurTextureCenter, blurTextureSize, marginOffset);

        // 縦ブラー&影最終描画用の頂点シェーダーコンスタントバッファを作成する
        MakeVerticalBlurConstantBuffer(drawInfo, m_DropShadowInfo.verticalBlurAndShadowVertexShaderCB[type], blurParams, blurTextureCenter, blurTextureSize, blurTextureScale, marginOffset);

        // ドロップシャドウのブラー用ピクセルシェーダーコンスタントバッファを作成する
        // シェーダーバリエーションをウエイトテーブルから決定する。
        // シェーダーは 4 サンプルごとにバリエーションが分けられているので、ウエイトの有効な数値が入ったところまでフェッチする用にバリエーションを設定する
        tapSize = MakeBlurPixelShaderConstantBuffer(drawInfo, m_DropShadowInfo.blurPixelShaderCB[type], blurParams);
    }

    // 静的レンダリングが有効ならば専用のコンスタントバッファを作成する
    if (IsDropShadowStaticRenderingEnabled())
    {
        MakeStaticRenderingConstantBuffer(drawInfo, m_DropShadowInfo.staticRenderingVertexShaderCB[type], blurParams, blurTextureSize);
    }

    if (tapSize > 0)
    {
        return (tapSize - 1) / 4;
    }
    else
    {
        return 0;
    }
}


//-----------------------------------------------------------------------------------
void PaneEffect::AllocateDropShadowEffectConstantBufferSet(DrawInfo& drawInfo, SystemDataDropShadow::DropShadowType type)
{
    size_t  constantBufferAlignment = drawInfo.GetGraphicsResource()->GetConstantBufferAlignment();

    // 描画結果キャッシュ済みの場合はもう必要ないため確保しない
    if (!m_DropShadowInfo.staticRenderingCached)
    {
        // 横ブラー頂点シェーダー
        m_DropShadowInfo.horizontalBlurVertexShaderCB[type] = static_cast<uint32_t>(SetupEffectDrawVertexShaderConstantBuffer(drawInfo));
        // 縦ブラー＆エフェクト最終描画頂点シェーダー
        m_DropShadowInfo.verticalBlurAndShadowVertexShaderCB[type] = static_cast<uint32_t>(SetupEffectDrawVertexShaderConstantBuffer(drawInfo));
        // ブラーピクセルシェーダー
        m_DropShadowInfo.blurPixelShaderCB[type] = static_cast<uint32_t>(drawInfo.GetUi2dConstantBuffer()->Allocate(nn::util::align_up(sizeof(detail::DropShadowBlurConstantBuffer), constantBufferAlignment)));
    }

    // キャッシュ描画頂点シェーダー
    if (IsDropShadowStaticRenderingEnabled())
    {
        m_DropShadowInfo.staticRenderingVertexShaderCB[type] = static_cast<uint32_t>(SetupEffectDrawVertexShaderConstantBuffer(drawInfo));
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::CalculateDropShadowMaskCaptureDrawConstantBuffer(DrawInfo& drawInfo, int captureMaskDrawCBOffset)
{
    nn::ui2d::Size  blurTextureSize;
    blurTextureSize.width = m_DropShadowInfo.pBlurRenderTarget->GetSize().width;
    blurTextureSize.height = m_DropShadowInfo.pBlurRenderTarget->GetSize().height;

    nn::util::Float2    blurTextureCenter;
    blurTextureCenter.x = -blurTextureSize.width * 0.5f;
    blurTextureCenter.y = blurTextureSize.height * 0.5f;

    nn::util::BytePtr   pConstantBufferForVertexShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
    Material::ConstantBufferForVertexShader* pVertexShaderConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pConstantBufferForVertexShader.Advance(captureMaskDrawCBOffset).Get());

    // マスクパラメータをコンスタントバッファへ反映する
    ApplyMaskParameters(pVertexShaderConstantBuffer);

    detail::CalculateQuadWithTexCoords(drawInfo, pVertexShaderConstantBuffer, blurTextureCenter, blurTextureSize, 1, &DefaultTexCoords);

    // CalculateQuadWithTexCoords の中で LoadMtxModelView が行われているため、呼び出し後に上書きする
    nn::util::MatrixT4x4fType   projMtx;
    nn::util::MatrixOrthographicOffCenterRightHanded(&projMtx, blurTextureCenter.x, blurTextureCenter.x + blurTextureSize.width, blurTextureCenter.y, blurTextureCenter.y - blurTextureSize.height, 0.0f, 500.0f);

    nn::util::MatrixT4x3fType glbMtx;
    detail::CalculateCaptureRootMatrix(glbMtx, drawInfo);

    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(pVertexShaderConstantBuffer->projection), projMtx);
    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x3*>(pVertexShaderConstantBuffer->modelView), glbMtx);
}

//-----------------------------------------------------------------------------------
void PaneEffect::MakeBlurParams(BlurParams* pBlurParams, const SystemDataDropShadow* pDropShadowSystemData, SystemDataDropShadow::DropShadowType type) const
{
    NN_SDK_ASSERT_NOT_NULL(pBlurParams);
    NN_SDK_ASSERT_NOT_NULL(pDropShadowSystemData);

    // ドロップシャドウ用の各種パラメータからドロップシャドウ用の共通シェーダーで扱うためのデータに変換します。
    switch (type)
    {
    case SystemDataDropShadow::DropShadowType_Stroke:
        // 境界線は spread が 100% のドロップシャドウとして扱います。
        memcpy(pBlurParams->color, pDropShadowSystemData->strokeColor, sizeof(float) * 4);
        pBlurParams->size = pDropShadowSystemData->strokeSize;
        pBlurParams->spread = 100.0f;
        pBlurParams->angle = 0.0f;
        pBlurParams->distance = 0.0f;
        break;
    case SystemDataDropShadow::DropShadowType_OuterGlow:
        // 光彩は位置ずらしのないドロップシャドウとして扱います。
        memcpy(pBlurParams->color, pDropShadowSystemData->outerGlowColor, sizeof(float) * 4);
        pBlurParams->size = pDropShadowSystemData->outerGlowSize;
        pBlurParams->spread = pDropShadowSystemData->outerGlowSpread;
        pBlurParams->angle = 0.0f;
        pBlurParams->distance = 0.0f;
        break;
    case SystemDataDropShadow::DropShadowType_DropShadow:
        // ドロップシャドウはすべてのパラメータを使用します。
        memcpy(pBlurParams->color, pDropShadowSystemData->dropShadowColor, sizeof(float) * 4);
        pBlurParams->size = pDropShadowSystemData->dropShadowSize;
        pBlurParams->spread = pDropShadowSystemData->dropShadowSpread;
        pBlurParams->angle = pDropShadowSystemData->dropShadowAngle;
        pBlurParams->distance = pDropShadowSystemData->dropShadowDistance;
        break;
    default:
        NN_SDK_ASSERT(false, "");
        break;
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::CalculateDropShadowConstantBuffer(DrawInfo& drawInfo)
{
    NN_SDK_ASSERT_NOT_NULL(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());

    // マスクが有効な場合はマスクの描画結果をキャプチャするための描画を行う準備をする
    if (IsMaskFunctionEnabled())
    {
        m_DropShadowInfo.captureMaskDrawCB = static_cast<uint32_t>(SetupEffectDrawVertexShaderConstantBuffer(drawInfo));
        CalculateDropShadowMaskCaptureDrawConstantBuffer(drawInfo, m_DropShadowInfo.captureMaskDrawCB);
    }

    // ペイン本体の描画用のコンスタントバッファを確保する
    m_DropShadowInfo.constantBufferForFinalDraw = static_cast<uint32_t>(SetupEffectDrawVertexShaderConstantBuffer(drawInfo));

    // コンスタントバッファ未作成のフラグとしてシェーダーバリエーションオフセットに -1 を入れておく。
    // 各バッファのオフセットだと uint32_t で未作成状態を表すことができない。
    for (int i = 0; i < SystemDataDropShadow::DropShadowType_Max; ++i)
    {
        m_DropShadowInfo.blurShaderVariationOffset[i] = -1;
    }

    // コンスタントバッファを作成する
    // 境界線
    if (IsDropShadowStrokeEnabled() &&
        m_DropShadowInfo.pDropShadowData->strokeColor[3] > 0.0f)
    {
        SystemDataDropShadow::DropShadowType    type = SystemDataDropShadow::DropShadowType_Stroke;
        m_DropShadowInfo.blurShaderVariationOffset[type] = MakeDropShadowConstantBufferSet(drawInfo, type);
    }

    // 光彩
    if (IsDropShadowOuterGlowEnabled() &&
        m_DropShadowInfo.pDropShadowData->outerGlowColor[3] > 0.0f)
    {
        SystemDataDropShadow::DropShadowType    type = SystemDataDropShadow::DropShadowType_OuterGlow;
        m_DropShadowInfo.blurShaderVariationOffset[type] = MakeDropShadowConstantBufferSet(drawInfo, type);
    }

    // ドロップシャドウ
    if (IsDropShadowEnabled() &&
        m_DropShadowInfo.pDropShadowData->dropShadowColor[3] > 0.0f)
    {
        SystemDataDropShadow::DropShadowType    type = SystemDataDropShadow::DropShadowType_DropShadow;
        m_DropShadowInfo.blurShaderVariationOffset[type] = MakeDropShadowConstantBufferSet(drawInfo, type);
    }

    // 最終的なペイン本体の描画用のコンスタントバッファを作成する
    {
        nn::util::BytePtr   pConstantBufferForVertexShader(drawInfo.GetUi2dConstantBuffer()->GetMappedPointer());
        Material::ConstantBufferForVertexShader* pVertexShaderConstantBuffer = static_cast<Material::ConstantBufferForVertexShader*>(pConstantBufferForVertexShader.Advance(m_DropShadowInfo.constantBufferForFinalDraw).Get());

        if (IsDropShadowStaticRenderingReady())
        {
            // 静的レンダリング時に透明度アニメーションを反映させるためグローバルアルファを適用します。
            pVertexShaderConstantBuffer->color[3] = static_cast<float>(m_pTargetPane->GetGlobalAlpha()) / 255.0f / 255.0f;
        }

        detail::CalculateQuadWithTexCoords(drawInfo, pVertexShaderConstantBuffer, m_pTargetPane->GetVertexPos(), m_pTargetPane->GetSize(), 1, &DefaultTexCoords);
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::DrawSpriteCommonImpl(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer, const ShaderInfo* pShaderInfo, int variationIndex, int vertexShaderCBOffset)
{
    size_t vertexShaderConstantBufferSize = sizeof(Material::ConstantBufferForVertexShader);
    nn::gfx::GpuAddress gpuAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
    gpuAddr.Offset(vertexShaderCBOffset);

    const nn::gfx::GpuAddress* pGpuAddrForIndexBuffer;
    pGpuAddrForIndexBuffer = drawInfo.GetGraphicsResource()->GetIndexBufferGpuAddress();

    NN_SDK_ASSERT(pShaderInfo->GetVertexShaderSlot(variationIndex) >= 0, "An invalid shader-slot(for DrawSpriteCommonImpl)was detected at [variationIndex=%d]! Are you sure that you intend to use vertex-shader ? ", variationIndex);
    commandBuffer.SetConstantBuffer(pShaderInfo->GetVertexShaderSlot(variationIndex), nn::gfx::ShaderStage_Vertex, gpuAddr, vertexShaderConstantBufferSize);

    commandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint16, *pGpuAddrForIndexBuffer, static_cast<int>(GraphicsResource::VertexBufferIndexCount), 0);
}

//-----------------------------------------------------------------------------------
void PaneEffect::CaptureMaskedImage(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer, nn::gfx::ColorTargetView* pTarget)
{
    nn::gfx::ClearColorValue    clearColor;

    clearColor.valueUint[0] = 0;
    clearColor.valueUint[1] = 0;
    clearColor.valueUint[2] = 0;
    clearColor.valueUint[3] = 0;

    commandBuffer.ClearColorTarget(pTarget, clearColor, NULL);
    commandBuffer.SetRenderTargets(1, &pTarget, NULL);
    commandBuffer.SetViewportScissorState(m_DropShadowInfo.pMaskCopyRenderTarget->GetViewportScissorState());

    SetupMaskShader(drawInfo, commandBuffer);
    DrawSpriteCommonImpl(drawInfo, commandBuffer, m_MaskInfo.pShaderInfo, 0, m_DropShadowInfo.captureMaskDrawCB);

    commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
    commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);

    drawInfo.ResetRenderTarget(commandBuffer);
}

//-----------------------------------------------------------------------------------
int PaneEffect::CalculateVerticalBlurShaderId(PresetBlendStateId blendId, bool knockout) const
{
    int blurShaderId = 0;

    // 乗算の場合はシェーダーでカラーを調整する必要があるため別バリエーションになる
    if (blendId == PresetBlendStateId_Multiplication)
    {
        if (knockout)
        {
            blurShaderId = detail::DropShadowShaderVariation_VerticalBlur1WithKnockoutColorMul;
        }
        else
        {
            blurShaderId = detail::DropShadowShaderVariation_VerticalBlur1WithColorMul;
        }
    }
    else
    {
        if (knockout)
        {
            blurShaderId = detail::DropShadowShaderVariation_VerticalBlur1WithKnockout;
        }
        else
        {
            blurShaderId = detail::DropShadowShaderVariation_VerticalBlur1;
        }
    }

    return blurShaderId;
}

//-----------------------------------------------------------------------------------
void PaneEffect::DrawBluredShadow(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer, SystemDataDropShadow::DropShadowType type, PresetBlendStateId blendId, bool knockout)
{
    int blurVariationOffset = m_DropShadowInfo.blurShaderVariationOffset[type];
    int hBlurCBOffset = m_DropShadowInfo.horizontalBlurVertexShaderCB[type];
    int vBlurCBOffset = m_DropShadowInfo.verticalBlurAndShadowVertexShaderCB[type];
    int blurCBOffset = m_DropShadowInfo.blurPixelShaderCB[type];

    const TextureInfo*  pBlurSrcTexture = m_DropShadowInfo.captureTexMap.GetTextureInfo();
    nn::gfx::ClearColorValue    clearColor;

    // マスクが使用されている場合はブラーソーステクスチャをキャプチャしたものに切り替える
    if (IsMaskFunctionEnabled())
    {
        pBlurSrcTexture = m_DropShadowInfo.pMaskCopyRenderTarget;
    }

    const ShaderInfo* pShaderInfo = m_DropShadowInfo.pShaderInfo;
    nn::gfx::ColorTargetView* pBlurBuffer = m_DropShadowInfo.pBlurRenderTarget->GetColorTargetView();

    clearColor.valueUint[0] = 0;
    clearColor.valueUint[1] = 0;
    clearColor.valueUint[2] = 0;
    clearColor.valueUint[3] = 0;

    commandBuffer.ClearColorTarget(pBlurBuffer, clearColor, NULL);
    commandBuffer.SetRenderTargets(1, &pBlurBuffer, NULL);
    commandBuffer.SetViewportScissorState(m_DropShadowInfo.pBlurRenderTarget->GetViewportScissorState());

    // バリエーションを検索して設定
    // 横ブラーシェーダー
    int blurShaderId = detail::DropShadowShaderVariation_HorizontalBlur1 + blurVariationOffset;
    int variationIndex = m_DropShadowInfo.dropShadowShaderVariationIndex[blurShaderId];

    SetupShader(drawInfo, commandBuffer, pShaderInfo, variationIndex);

    // オリジナルサイズでキャプチャした画像の UV を調整してブラーサイズ込みのテクスチャの中央に描画する。
    // 範囲外は透明になるように透明なボーダーカラーを設定したサンプラーを設定する。
    nn::gfx::DescriptorSlot captureTextureSampler = drawInfo.GetGraphicsResource()->GetSamplerDescriptorSlot(PresetSamplerId_ClampToTransparentBorderColor);

    commandBuffer.SetTextureAndSampler(pShaderInfo->GetTextureSlot(static_cast<uint8_t>(variationIndex), 0), nn::gfx::ShaderStage_Pixel, *pBlurSrcTexture->GetTextureDescriptorSlot(), captureTextureSampler);

    NN_SDK_ASSERT(pShaderInfo->GetVertexShaderSlot(variationIndex) >= 0, "An invalid shader-slot(for DropShadowConstantBufferForVertexShader)was detected at [variationIndex=%d]! Are you sure that you intend to use vertex-shader ? ", variationIndex);

    // ブラー用ピクセルシェーダーのコンスタントバッファを設定
    nn::gfx::GpuAddress fsConstantBufferAddr = *drawInfo.GetUi2dConstantBuffer()->GetGpuAddress();
    fsConstantBufferAddr.Offset(blurCBOffset);
    size_t blurConstantBufferSize = sizeof(detail::DropShadowBlurConstantBuffer);
    if (m_DropShadowInfo.dropShadowBlurPixelShaderCB[blurShaderId] >= 0)
    {
        commandBuffer.SetConstantBuffer(m_DropShadowInfo.dropShadowBlurPixelShaderCB[blurShaderId], nn::gfx::ShaderStage_Pixel, fsConstantBufferAddr, blurConstantBufferSize);
    }

    commandBuffer.SetBlendState(drawInfo.GetGraphicsResource()->GetPresetBlendState(PresetBlendStateId_OpaqueOrAlphaTest));

    DrawSpriteCommonImpl(drawInfo, commandBuffer, pShaderInfo, variationIndex, hBlurCBOffset);

    // 同期してレンダーターゲット切り替え
    commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
    commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);

    NN_SDK_ASSERT(drawInfo.GetDefaultColorTargetView() != NULL, "You must set default color target view information by DrawInfo::SetDefaultColorTargetView() to use the drop shadow function.");

    drawInfo.ResetRenderTarget(commandBuffer);

    // 縦ブラー＆ボケ画像最終描画
    // 乗算の場合はシェーダーでカラーを調整する必要があるため別バリエーションになる
    blurShaderId = CalculateVerticalBlurShaderId(blendId, knockout) + blurVariationOffset;
    variationIndex = m_DropShadowInfo.dropShadowShaderVariationIndex[blurShaderId];

    SetupShader(drawInfo, commandBuffer, pShaderInfo, variationIndex);

    commandBuffer.SetTextureAndSampler(pShaderInfo->GetTextureSlot(static_cast<uint8_t>(variationIndex), 0), nn::gfx::ShaderStage_Pixel, *m_DropShadowInfo.pBlurRenderTarget->GetTextureDescriptorSlot(), captureTextureSampler);
    if (IsDropShadowKnockoutEnabled())
    {
        if (IsMaskFunctionEnabled())
        {
            commandBuffer.SetTextureAndSampler(pShaderInfo->GetTextureSlot(static_cast<uint8_t>(variationIndex), 1), nn::gfx::ShaderStage_Pixel, *m_DropShadowInfo.pMaskCopyRenderTarget->GetTextureDescriptorSlot(), captureTextureSampler);
        }
        else
        {
            commandBuffer.SetTextureAndSampler(pShaderInfo->GetTextureSlot(static_cast<uint8_t>(variationIndex), 1), nn::gfx::ShaderStage_Pixel, *m_DropShadowInfo.captureTexMap.GetTextureInfo()->GetTextureDescriptorSlot(), captureTextureSampler);
        }
    }

    NN_SDK_ASSERT(pShaderInfo->GetVertexShaderSlot(variationIndex) >= 0, "An invalid shader-slot(for DropShadowConstantBufferForVertexShader)was detected at [variationIndex=%d]! Are you sure that you intend to use vertex-shader ? ", variationIndex);

    if (m_DropShadowInfo.dropShadowBlurPixelShaderCB[blurShaderId] >= 0)
    {
        commandBuffer.SetConstantBuffer(m_DropShadowInfo.dropShadowBlurPixelShaderCB[blurShaderId], nn::gfx::ShaderStage_Pixel, fsConstantBufferAddr, blurConstantBufferSize);
    }

    commandBuffer.SetBlendState(drawInfo.GetGraphicsResource()->GetPresetBlendState(blendId));

    DrawSpriteCommonImpl(drawInfo, commandBuffer, pShaderInfo, variationIndex, vBlurCBOffset);
}

//-----------------------------------------------------------------------------------
void PaneEffect::DrawStaticCachedShadow(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    for (int i = SystemDataDropShadow::DropShadowType_Max - 1; i >= 0; --i)
    {
        if (!detail::TestBit(m_DropShadowInfo.pDropShadowData->flags, i))
        {
            continue;
        }

        PresetBlendStateId blendId = ConvertBlendType(static_cast<detail::DropShadowBlendMode>(m_DropShadowInfo.pDropShadowData->blendType[i]));

        int variationIndex = m_DropShadowInfo.dropShadowShaderVariationIndex[detail::DropShadowShaderVariation_Copy];

        // アルファ付き乗算のために専用シェーダーを使用する
        if (blendId == PresetBlendStateId_Multiplication)
        {
            variationIndex = m_DropShadowInfo.dropShadowShaderVariationIndex[detail::DropShadowShaderVariation_CopyColorMul];
        }
        const ShaderInfo* pShaderInfo = m_DropShadowInfo.pShaderInfo;

        SetupShader(drawInfo, commandBuffer, pShaderInfo, variationIndex);

        nn::gfx::DescriptorSlot samplerDescriptor =
            drawInfo.GetGraphicsResource()->GetSamplerDescriptorSlot(TexWrap_Clamp, TexWrap_Clamp, TexFilter_Linear, TexFilter_Linear);
        commandBuffer.SetTextureAndSampler(pShaderInfo->GetTextureSlot(static_cast<uint8_t>(variationIndex), 0), nn::gfx::ShaderStage_Pixel, *m_DropShadowInfo.pStaticRenderingTextures[i]->GetTextureDescriptorSlot(), samplerDescriptor);
        commandBuffer.SetBlendState(drawInfo.GetGraphicsResource()->GetPresetBlendState(blendId));

        DrawSpriteCommonImpl(drawInfo, commandBuffer, pShaderInfo, variationIndex, m_DropShadowInfo.staticRenderingVertexShaderCB[i]);
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::SetupViewportScissorStateInfo(nn::gfx::ViewportStateInfo* pViewportStateInfo, nn::gfx::ScissorStateInfo* pScissorStateInfo, const TexSize& texSize)
{
    NN_SDK_ASSERT_NOT_NULL(pViewportStateInfo);
    NN_SDK_ASSERT_NOT_NULL(pScissorStateInfo);

    pScissorStateInfo->SetOriginX(0);
    pScissorStateInfo->SetOriginY(0);
    pScissorStateInfo->SetWidth(static_cast<int>(texSize.width));
    pScissorStateInfo->SetHeight(static_cast<int>(texSize.height));

    pViewportStateInfo->SetOriginX(0.0f);
    pViewportStateInfo->SetOriginY(0.0f);
    pViewportStateInfo->SetWidth(static_cast<float>(texSize.width));
    pViewportStateInfo->SetHeight(static_cast<float>(texSize.height));
}

//-----------------------------------------------------------------------------------
void PaneEffect::DrawDropShadow(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    // マスクの描画結果キャプチャ
    if (IsMaskFunctionEnabled())
    {
        CaptureMaskedImage(drawInfo, commandBuffer, m_DropShadowInfo.pMaskCopyRenderTarget->GetColorTargetView());
    }

    // 静的レンダリングが無効、もしくはキャッシュ作成が必要な場合に通常描画を行います。
    if (!IsDropShadowStaticRenderingReady())
    {
        nn::gfx::ColorTargetView*   pPrevColorTargetView = drawInfo.GetDefaultColorTargetView();
        nn::gfx::ScissorStateInfo   prevScissorStateInfo = drawInfo.GetDefaultScissorInfo();
        nn::gfx::ViewportStateInfo  prevViewportStateInfo = drawInfo.GetDefaultViewportInfo();

        // ドロップシャドウが一番下に来るように描画順を調整
        for (int i = SystemDataDropShadow::DropShadowType_Max - 1; i >= 0; --i)
        {
            if (!detail::TestBit(m_DropShadowInfo.pDropShadowData->flags, i))
            {
                continue;
            }

            // アルファが 0 などで描画されない場合はコンスタントバッファやシェーダーのセットアップがキャンセルされるため描画も行わない。
            if (m_DropShadowInfo.blurShaderVariationOffset[i] < 0)
            {
                continue;
            }

            PresetBlendStateId blendId = PresetBlendStateId_Default;

            nn::gfx::ScissorStateInfo   newScissorStateInfo;
            nn::gfx::ViewportStateInfo  newViewportStateInfo;

            if (!IsDropShadowStaticRenderingEnabled())
            {
                blendId = ConvertBlendType(static_cast<detail::DropShadowBlendMode>(m_DropShadowInfo.pDropShadowData->blendType[i]));
            }
            else
            {
                // キャッシュ作成時の処理
                // ブラーテクスチャの描画先をキャッシュ用のテクスチャに切り替えてレンダリングする。
                nn::gfx::ColorTargetView*   pCacheTexture = m_DropShadowInfo.pStaticRenderingTextures[i]->GetColorTargetView();
                nn::gfx::ClearColorValue    clearColor;

                clearColor.valueUint[0] = 255;
                clearColor.valueUint[1] = 255;
                clearColor.valueUint[2] = 255;
                clearColor.valueUint[3] = 255;

                commandBuffer.ClearColorTarget(pCacheTexture, clearColor, NULL);

                // 新しいレンダーターゲットのビューポートシザーステートを設定する
                SetupViewportScissorStateInfo(&newViewportStateInfo, &newScissorStateInfo, m_DropShadowInfo.pStaticRenderingTextures[i]->GetSize());

                // 描画先をキャッシュテクスチャに切り替える
                drawInfo.SetDefaultColorTargetView(pCacheTexture);
                drawInfo.SetDefaultViewportScissorInfo(newViewportStateInfo, newScissorStateInfo);

                blendId = PresetBlendStateId_OpaqueOrAlphaTest;
            }

            // 現在設定されている DefaultColorTargetView へぼかした影をレンダリングします
            DrawBluredShadow(drawInfo, commandBuffer, static_cast<SystemDataDropShadow::DropShadowType>(i), blendId, IsDropShadowKnockoutEnabled());
        }

        // 静的レンダリングが有効な場合で同一フレームでキャプチャを行った場合に必要な処理です。
        if (IsDropShadowStaticRenderingEnabled())
        {
            // すぐにテクスチャとして使用するためキャッシュを更新
            commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
            commandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Texture);
            // 静的レンダリングを有効にします。
            m_DropShadowInfo.staticRenderingCached = true;

            // レンダーターゲットをキャプチャする前のものに戻します。
            drawInfo.SetDefaultColorTargetView(pPrevColorTargetView);
            drawInfo.SetDefaultViewportScissorInfo(prevViewportStateInfo, prevScissorStateInfo);
            drawInfo.ResetRenderTarget(commandBuffer);
        }
    }
}

//-----------------------------------------------------------------------------------
void PaneEffect::DrawDropShadowOriginalImage(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer)
{
    // ペイン本体の最終描画
    int variationIndex = m_DropShadowInfo.dropShadowShaderVariationIndex[detail::DropShadowShaderVariation_Copy];

    const ShaderInfo* pShaderInfo = m_DropShadowInfo.pShaderInfo;
    SetupShader(drawInfo, commandBuffer, pShaderInfo, variationIndex);

    nn::gfx::DescriptorSlot samplerDescriptor =
        drawInfo.GetGraphicsResource()->GetSamplerDescriptorSlot(TexWrap_Clamp, TexWrap_Clamp, TexFilter_Linear, TexFilter_Linear);
    commandBuffer.SetTextureAndSampler(pShaderInfo->GetTextureSlot(static_cast<uint8_t>(variationIndex), 0), nn::gfx::ShaderStage_Pixel, *m_DropShadowInfo.captureTexMap.GetTextureInfo()->GetTextureDescriptorSlot(), samplerDescriptor);
    m_pTargetPane->SetupPaneEffectSourceImageRenderState(commandBuffer);

    DrawSpriteCommonImpl(drawInfo, commandBuffer, pShaderInfo, variationIndex, m_DropShadowInfo.constantBufferForFinalDraw);
}

//-----------------------------------------------------------------------------------
PresetBlendStateId PaneEffect::ConvertBlendType(detail::DropShadowBlendMode mode) const
{
    PresetBlendStateId blendId = PresetBlendStateId_Default;

    switch(mode)
    {
    case    detail::DropShadowBlendMode_Mul:
        blendId = PresetBlendStateId_Multiplication;
        break;
    case    detail::DropShadowBlendMode_Add:
        blendId = PresetBlendStateId_Addition;
        break;
    case    detail::DropShadowBlendMode_Sub:
        blendId = PresetBlendStateId_Subtraction;
        break;
    default:
        break;
    }

    return blendId;
}

//-----------------------------------------------------------------------------------
void PaneEffect::SetupShader(DrawInfo& drawInfo, nn::gfx::CommandBuffer& commandBuffer, const ShaderInfo* pShaderInfo, int variationIndex)
{
    if (drawInfo.RecordCurrentShader(pShaderInfo, static_cast<uint8_t>(variationIndex)))
    {
        pShaderInfo->SetShader(commandBuffer, variationIndex);
    }
    drawInfo.SetupProgram(&commandBuffer);
}


} // namespace nn::ui2d::detail

} // namespace nn::ui2d
} // namespace nn
