﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>

#include <nn/ui2d/ui2d_Common.h>
#include <nn/ui2d/ui2d_DrawInfo.h>
#include <nn/ui2d/ui2d_GraphicsResource.h>
#include <nn/ui2d/ui2d_Material.h>
#include <nn/ui2d/ui2d_Pane.h>
#include <nn/ui2d/ui2d_Layout.h>

namespace nn
{
namespace ui2d
{

void UserShaderInformation::SetDefault()
{
    std::memset(userShaderName, 0, ArchiveShaderNameMax);
    vertexShaderConstantBufferExtendSize = 0;
    geometryShaderConstantBufferExtendSize = 0;
    pixelShaderConstantBufferExtendSize = 0;
}

bool UserShaderInformation::SetShaderName(const char* pShaderName)
{
    size_t userShaderNameLength = strlen(pShaderName);
    NN_SDK_ASSERT(userShaderNameLength < nn::ui2d::ArchiveShaderNameMax, "shaderName[%s]", pShaderName);
    nn::util::Strlcpy(userShaderName, pShaderName, nn::ui2d::ArchiveShaderNameMax);

    return userShaderNameLength > 0 && userShaderNameLength < nn::ui2d::ArchiveShaderNameMax;
}

void BuildResultInformation::SetDefault() NN_NOEXCEPT
{
    this->requiredUi2dConstantBufferSize = 0;
    this->requiredFontConstantBufferSize = 0;
}

namespace detail
{

void TexCoordArray::Initialize()
{
    m_Cap = 0;
    m_Count = 0;
    m_pData = 0;
}

void
TexCoordArray::Free()
{
    if (m_pData)
    {
        Layout::FreeMemory(m_pData);
        m_pData = 0;

        m_Cap = 0;
        m_Count = 0;
    }
}

void
TexCoordArray::Reserve(int  num)
{
    NN_SDK_ASSERT(num <= TexMapMax, "out of bounds: num[%d] <= TexMapMax", num);

    if (m_Cap < num)
    {
        Free();

        // テクスチャ座標
        const uint32_t  coordNum = num;
        nn::util::Float2* pVecArray = static_cast<nn::util::Float2*>(Layout::AllocateMemory(sizeof(nn::util::Float2) * PaneVertex_MaxPaneVertex * coordNum));
        pVecArray->v[0] = 0.0f;
        pVecArray->v[1] = 0.0f;
        m_pData = reinterpret_cast<TexCoordQuad *>(pVecArray);
        if (m_pData)
        {
            m_Cap = static_cast<int8_t>(num);
        }
    }
}

void
TexCoordArray::SetSize(int  num)
{
    if (m_pData && num <= m_Cap)
    {
        static const nn::util::Float2 texCoords[PaneVertex_MaxPaneVertex] =
        {
            NN_UTIL_FLOAT_2_INITIALIZER(0.f, 0.f),
            NN_UTIL_FLOAT_2_INITIALIZER(1.f, 0.f),
            NN_UTIL_FLOAT_2_INITIALIZER(0.f, 1.f),
            NN_UTIL_FLOAT_2_INITIALIZER(1.f, 1.f)
        };

        for (int j = m_Count; j < num; ++j)
        {
            for (int i = 0; i < PaneVertex_MaxPaneVertex; ++i)
            {
                m_pData[j][i] = texCoords[i];
            }
        }
        m_Count = static_cast<int8_t>(num);
    }
}

void
TexCoordArray::GetCoord(
    TexCoordQuad coord,
    int  idx
) const
{
    NN_SDK_ASSERT(idx < m_Count, "out of bounds: idx[%d] < m_Count[%d]", idx, m_Count);

    for (int i = 0; i < PaneVertex_MaxPaneVertex; ++i)
    {
        coord[i] = m_pData[idx][i];
    }
}

void
TexCoordArray::SetCoord(
    int  idx,
    const TexCoordQuad coord
)
{
    NN_SDK_ASSERT(idx < m_Count, "out of bounds: idx[%d] < m_Count[%d]", idx, m_Count);

    for (int i = 0; i < PaneVertex_MaxPaneVertex; ++i)
    {
        m_pData[idx][i] = coord[i];
    }
}

void
TexCoordArray::Copy(
    const void* pResTexCoord,
    int  texCoordNum
)
{
    NN_SDK_ASSERT(texCoordNum <= m_Cap, "out of bounds: texCoordNum[%d] <= m_Cap[%d]", texCoordNum, m_Cap);

    m_Count = std::max(m_Count, static_cast<uint8_t>(texCoordNum));
    const ResVec2 (*src)[PaneVertex_MaxPaneVertex] = static_cast<const ResVec2 (*)[PaneVertex_MaxPaneVertex]>(pResTexCoord);
    for (int j = 0; j < texCoordNum; ++j)
    {
        for (int i = 0; i < PaneVertex_MaxPaneVertex; ++i)
        {
            m_pData[j][i] = src[j][i].operator const nn::util::Float2();
        }
    }
}

bool
TexCoordArray::CompareCopiedInstanceTest(const TexCoordArray& target) const
{
    if (m_Cap != target.m_Cap ||
        m_Count != target.m_Count)
    {
        return false;
    }

    for (int j = 0; j < m_Count; ++j)
    {
        for (int i = 0; i < PaneVertex_MaxPaneVertex; ++i)
        {
            if (memcmp(m_pData[j][i].v, target.m_pData[j][i].v, sizeof(nn::util::Float2)) != 0)
            {
                return false;
            }
        }
    }

    return true;
}

//----------------------------------------
const ResMaterial*
GetResMaterial(
    const BuildResSet* pBuildResSet,
    uint16_t  materialIdx)
{
    NN_SDK_ASSERT_NOT_NULL(pBuildResSet->pMaterialList);
    const size_t sizeMatOffs = sizeof(*pBuildResSet->pMaterialList);
    const uint32_t *const pMatOffsets = nn::util::ConstBytePtr(pBuildResSet->pMaterialList, sizeMatOffs).Get<uint32_t>();
    return nn::util::ConstBytePtr(pBuildResSet->pMaterialList, pMatOffsets[materialIdx]).Get<ResMaterial>();
}

//----------------------------------------
const ResShapeInfo*
GetResShapeInfo(
    const BuildResSet* pBuildResSet,
    uint32_t  shapeInfoIdx)
{
    NN_SDK_ASSERT_NOT_NULL(pBuildResSet->pShapeInfoList);
    const size_t sizeShapeInfoOffs = sizeof(*pBuildResSet->pShapeInfoList);
    const uint32_t *const pShapeInfoOffsets = nn::util::ConstBytePtr(pBuildResSet->pShapeInfoList, sizeShapeInfoOffs).Get<uint32_t>();
    return nn::util::ConstBytePtr(pBuildResSet->pShapeInfoList, pShapeInfoOffsets[shapeInfoIdx]).Get<ResShapeInfo>();
}

//----------------------------------------
void
CalculateQuad(
    DrawInfo& drawInfo,
    Material::ConstantBufferForVertexShader* pConstantBufferForVertexShader,
    const nn::util::Float2&         basePt,
    const Size&         size
)
{
    // ローカル座標変換設定
    pConstantBufferForVertexShader->transform[0] = size.width;
    pConstantBufferForVertexShader->transform[1] = size.height;
    pConstantBufferForVertexShader->transform[2] = basePt.v[0];
    pConstantBufferForVertexShader->transform[3] = basePt.v[1];

    drawInfo.LoadMtxModelView(pConstantBufferForVertexShader->modelView);
}

//----------------------------------------
void
CalculateQuadWithTexCoords(
    DrawInfo& drawInfo,
    Material::ConstantBufferForVertexShader* pVertexShaderConstantBuffer,
    const nn::util::Float2& basePt,
    const Size& size,
    int  texCoordNum,
    const nn::util::Float2 (*texCoords)[PaneVertex_MaxPaneVertex]
)
{
    // テクスチャ座標設定
    if (0 < texCoordNum && texCoords != NULL)
    {
        for (int i = 0; i < TexMapMax; ++i)
        {
            int src = drawInfo.GetTexCoordSrc(i);
            if (src < 0)
            {
                break;
            }

            if (texCoords != NULL && src < texCoordNum)
            {
                float (*vertexTexCoord)[4] = NULL;
                switch (i)
                {
                    case 0:
                        vertexTexCoord = pVertexShaderConstantBuffer->vertexTexCoord0;
                        break;
                    case 1:
                        vertexTexCoord = pVertexShaderConstantBuffer->vertexTexCoord1;
                        break;
                    case 2:
                        vertexTexCoord = pVertexShaderConstantBuffer->vertexTexCoord2;
                        break;
                    default: NN_UNEXPECTED_DEFAULT;
                }
                vertexTexCoord[0][0] = texCoords[src][0].v[0];
                vertexTexCoord[0][1] = texCoords[src][0].v[1];
                vertexTexCoord[0][2] = 0.0f;
                vertexTexCoord[0][3] = 1.0f;
                vertexTexCoord[1][0] = texCoords[src][1].v[0];
                vertexTexCoord[1][1] = texCoords[src][1].v[1];
                vertexTexCoord[1][2] = 0.0f;
                vertexTexCoord[1][3] = 1.0f;
                vertexTexCoord[2][0] = texCoords[src][2].v[0];
                vertexTexCoord[2][1] = texCoords[src][2].v[1];
                vertexTexCoord[2][2] = 0.0f;
                vertexTexCoord[2][3] = 1.0f;
                vertexTexCoord[3][0] = texCoords[src][3].v[0];
                vertexTexCoord[3][1] = texCoords[src][3].v[1];
                vertexTexCoord[3][2] = 0.0f;
                vertexTexCoord[3][3] = 1.0f;
            }
        }
    }

    // ローカル座標変換設定
    pVertexShaderConstantBuffer->transform[0] = size.width;
    pVertexShaderConstantBuffer->transform[1] = size.height;
    pVertexShaderConstantBuffer->transform[2] = basePt.v[0];
    pVertexShaderConstantBuffer->transform[3] = basePt.v[1];

    drawInfo.LoadMtxModelView(pVertexShaderConstantBuffer->modelView);
}

//----------------------------------------
void
SetupMaterialRenderState(
    nn::gfx::CommandBuffer& commandBuffer,
    DrawInfo&     drawInfo,
    Material&     material)
{
    // シェーダの設定
    if (drawInfo.RecordCurrentShader(material.GetShaderInfo(), material.GetShaderVariation()))
    {
        material.SetShader(commandBuffer);
    }

    // プログラムが設定されてから一度だけ行う初期化。
    drawInfo.SetupProgram(&commandBuffer);

    material.SetCommandBuffer(commandBuffer, drawInfo);
}



//----------------------------------------
void
DrawQuad(
    nn::gfx::CommandBuffer& commandBuffer,
    DrawInfo&     drawInfo)
{
    const nn::gfx::GpuAddress* pGpuAddrForIndexBuffer;
    pGpuAddrForIndexBuffer = drawInfo.GetGraphicsResource()->GetIndexBufferGpuAddress();
    commandBuffer.DrawIndexed(nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint16, *pGpuAddrForIndexBuffer, static_cast<int>(GraphicsResource::VertexBufferIndexCount), 0);
}

//----------------------------------------
void
DrawBox(
    nn::gfx::CommandBuffer& commandBuffer,
    DrawInfo&           drawInfo,
    Material&           material,
    const nn::util::Float2&   pos,
    const Size&         size,
    const nn::util::Unorm8x4&          color)
{
    NN_UNUSED(commandBuffer);
    NN_UNUSED(drawInfo);
    NN_UNUSED(material);
    NN_UNUSED(pos);
    NN_UNUSED(size);
    NN_UNUSED(color);

    // 現在一時的に利用できなくなっています。
    /*
    GraphicsResource* pGraphicsResource = drawInfo.GetGraphicsResource();

    // テクスチャなし、頂点カラーなしのシェーダを設定
    drawInfo.SetShader(NULL_TEXTURE_SHADER + SHADER_ID_QUANTITY);

    commandBuffer.SetVertexBuffer(0, &pGraphicsResource->vertexBufferView, sizeof(float) * 2);
    commandBuffer.SetIndexBuffer(&pGraphicsResource->indexBufferView, nn::gfx::IndexFormat_Uint32);

    ResMaterial::ConstantBufferForVertexShader* pVertexShaderConstantBuffer = material.GetConstantBufferForVertexShader();
    ResMaterial::ConstantBufferForPixelShader* psh = material.GetConstantBufferForPixelShader();

    {
        const float scale = 1.0f / 255.0f; // 頂点カラーを[0, 1]に正規化するためのスケールです。
        static const Color8 colors[VERTEXCOLOR_MAX] = {
            Color8(255, 255, 255, 255),
            Color8(255, 255, 255, 255),
            Color8(255, 255, 255, 255),
            Color8(255, 255, 255, 255)
        };
        detail::SetupVertexColor(material, colors);
        // 頂点カラーへ積算するカラーを設定します。
        pVertexShaderConstantBuffer->color[0] = scale * 1.0f;
        pVertexShaderConstantBuffer->color[1] = scale * 1.0f;
        pVertexShaderConstantBuffer->color[2] = scale * 1.0f;
        pVertexShaderConstantBuffer->color[3] = scale * scale * 255.f;
    }

    {
        FloatColor float_color(color);
        psh->interpolateWidth[0] = float_color.r;
        psh->interpolateWidth[1] = float_color.g;
        psh->interpolateWidth[2] = float_color.b;
        psh->interpolateWidth[3] = float_color.a;
        psh->interpolateOffset[0] = 0.0f;
        psh->interpolateOffset[1] = 0.0f;
        psh->interpolateOffset[2] = 0.0f;
        psh->interpolateOffset[3] = 0.0f;
    }

    {
        material.GetBlendStateInfo()->SetLogicOperationEnabled(false);

        nn::gfx::BlendTargetStateInfo* pBlendTargetStateInfo = material.GetBlendTargetStateInfo();
        pBlendTargetStateInfo->SetBlendEnabled(true);
        pBlendTargetStateInfo->SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        pBlendTargetStateInfo->SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        pBlendTargetStateInfo->SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        pBlendTargetStateInfo->SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        pBlendTargetStateInfo->SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        pBlendTargetStateInfo->SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
    }

    // フレームの設定を初期化
    pVertexShaderConstantBuffer->frameSpec = TEXTUREFLIP_NONE;

    // ローカル座標変換設定
    pVertexShaderConstantBuffer->transform[0] = size.width;
    pVertexShaderConstantBuffer->transform[1] = size.height;
    pVertexShaderConstantBuffer->transform[2] = pos.x;
    pVertexShaderConstantBuffer->transform[3] = pos.y;

    drawInfo.LoadMtxModelView(pVertexShaderConstantBuffer->modelView);
    drawInfo.LoadProjectionMtx(pVertexShaderConstantBuffer->projection);

    // テクスチャモードを0にする -> テクスチャをフェッチしない設定
    psh->textureMode = 0;

    material.SetCommandBuffer(commandBuffer, pGraphicsResource);
    commandBuffer.DrawIndexed(static_cast<int>(GraphicsResource::VertexBufferIndexCount), 0, 0);
    */
}

} // namespace nn::ui2d::detail

} // namespace nn::ui2d
} // namespace nn
