﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/os/os_Tick.h>
#include <nn/spy/detail/fnd/basis/spyfnd_Time.h>
#include <nn/spy/detail/fnd/binary/spyfnd_Binary.h>
#include <nn/spy/detail/fnd/string/spyfnd_String.h>
#include <nn/spy/spy_SpyController.h>
#include <nn/spy/spy_TimeModule.h>

namespace {

//! @briefprivate
struct TimePacketData
{
    static const uint32_t Version = NN_DETAIL_SPY_FND_BINARY_MAKE_U32_SIGNATURE(0, 3, 0, 0);

    int32_t applicationFrame;
    int32_t audioFrame;
};

} // namespace

namespace nn {
namespace spy {

//----------------------------------------------------------
TimeModule::TimeModule() NN_NOEXCEPT
    : SpyModule("Time", TimePacketData::Version)
    , m_CurrentApplicationFrame(0)
    , m_CurrentAudioFrame(0)
{
}

//----------------------------------------------------------
void TimeModule::SetCurrentApplicationFrame(int value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(value, 0);

    nn::os::Tick tick;
    int audioFrame;

    {
        detail::fnd::ScopedCriticalSection lock(m_Mutex);

        if (value <= m_CurrentApplicationFrame)
        {
            return;
        }

        tick = nn::os::GetSystemTick();

        m_CurrentApplicationFrame = value;

        audioFrame = m_CurrentAudioFrame;
    }

    TimeModule::PushCurrentTimeData(tick, value, audioFrame);
}

//----------------------------------------------------------
void TimeModule::SetCurrentAudioFrame(int value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(value, 0);

    nn::os::Tick tick;
    int appFrame;

    {
        detail::fnd::ScopedCriticalSection lock(m_Mutex);

        if (value <= m_CurrentAudioFrame)
        {
            return;
        }

        tick = nn::os::GetSystemTick();

        m_CurrentAudioFrame = value;

        appFrame = m_CurrentApplicationFrame;
    }

    TimeModule::PushCurrentTimeData(tick, appFrame, value);
}

//----------------------------------------------------------
void
TimeModule::PushCurrentTimeData(
    nn::os::Tick tick,
    int appFrame,
    int audioFrame) NN_NOEXCEPT
{
    if(!IsRegistered())
    {
        return;
    }

    TimePacketData packetData;
    packetData.applicationFrame = appFrame;
    packetData.audioFrame = audioFrame;

    if(PushDataAt(&packetData, sizeof(TimePacketData), tick))
    {
#ifdef BUFFER_DEBUG_ENABLED
        NN_DETAIL_SPY_INFO(
            "[nn::spy::SpyController] Pushed TimePacket : app=%8d, audio=%8d.\n",
            packetData.applicationFrame,
            packetData.audioFrame);
#endif
    }
}

} // namespace nn::spy
} // namespace nn

#endif // NN_BUILD_CONFIG_SPY_ENABLED
