﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"

#include <nn/diag/text/diag_SdkTextSpy.h>

#include <nn/spy/detail/fnd/string/spyfnd_String.h>
#include <nn/spy/spy_PlotItem.h>
#include <nn/spy/spy_PlotNode.h>

namespace nn {
namespace spy {

NN_DEFINE_STATIC_CONSTANT(const int PlotItem::MaxNameLength);

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

//------------------------------------------------------------------------------
PlotItem::PlotItem() NN_NOEXCEPT
    : m_pModule(NULL)
    , m_pParent(NULL)
    , m_pPrevious(NULL)
    , m_pNext(NULL)
    , m_pItemMetaData(NULL)
    , m_IsMetadataPushed(false)
{
}

//------------------------------------------------------------------------------
PlotItem::~PlotItem() NN_NOEXCEPT
{
    if(IsAttached())
    {
        m_pModule->DetachItem(*this);
    }
}

//------------------------------------------------------------------------------
void PlotItem::InitializeItemMetaData(nn::spy::detail::SpyPlotItemPacket::ItemMetaData* pItemMetaData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pItemMetaData);

    pItemMetaData->id.Set(this);
    pItemMetaData->parentId.Set(nullptr);
    pItemMetaData->name.Set("");
    pItemMetaData->color.Set(0xFF, 0, 0);

    m_pItemMetaData = pItemMetaData;
}

//------------------------------------------------------------------------------
nn::spy::detail::SpyPlotItemPacket::ItemMetaData& PlotItem::GetItemMetaData() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pItemMetaData);
    return *m_pItemMetaData;
}

//------------------------------------------------------------------------------
const nn::spy::detail::SpyPlotItemPacket::ItemMetaData& PlotItem::GetItemMetaData() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pItemMetaData);
    return *m_pItemMetaData;
}

//------------------------------------------------------------------------------
void PlotItem::SetName(const char* name) NN_NOEXCEPT
{
    if(IsRequested())
    {
        NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] 通信中は、SpyPlotItem の名前を変更できません。\n"), NN_CURRENT_FUNCTION_NAME);
        return;
    }

    if(name == NULL)
    {
        NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] 名前が指定されていません。\n"), NN_CURRENT_FUNCTION_NAME);
        return;
    }

    GetItemMetaData().name.Set(name);
}

//------------------------------------------------------------------------------
const char* PlotItem::GetName() const NN_NOEXCEPT
{
    return GetItemMetaData().name.text;
}

//------------------------------------------------------------------------------
void PlotItem::SetColor(uint8_t r, uint8_t g, uint8_t b) NN_NOEXCEPT
{
    if(IsRequested())
    {
        NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] 通信中は、SpyPlotItem の色を変更できません。\n"), NN_CURRENT_FUNCTION_NAME);
        return;
    }

    GetItemMetaData().color.Set(r, g, b);
}

//------------------------------------------------------------------------------
void PlotItem::SetParent(PlotNode* pNode) NN_NOEXCEPT
{
    if(IsRequested())
    {
        NN_DETAIL_SPY_WARN(NN_TEXT_SPY("[%s] 通信中は、SpyPlotItem のグループを設定できません。\n"), NN_CURRENT_FUNCTION_NAME);
        return;
    }

    m_pParent = pNode;

    if (pNode != NULL)
    {
        GetItemMetaData().parentId = pNode->GetItemMetaData().id;
    }
    else
    {
        GetItemMetaData().parentId.Set(nullptr);
    }
}

//------------------------------------------------------------------------------
void PlotItem::Attach(PlotModule& module) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pModule == NULL || m_pModule == &module, NN_TEXT_SPY("この SpyPlotItem は、すでに他の SpyPlotModule にアタッチされています。"));

    m_pModule = &module;
    m_LastTick = nn::os::Tick(0);
    m_IsCheckTickEnabled = true;
    InvalidateMetadata();
}

//------------------------------------------------------------------------------
void PlotItem::Detach() NN_NOEXCEPT
{
    m_pModule = NULL;
    SetPrevious(NULL);
    SetNext(NULL);
}

//------------------------------------------------------------------------------
void PlotItem::PushMetaData() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsAttached());

    if (m_pParent != NULL && !m_pParent->IsMetadataPushed())
    {
        m_pParent->PushMetaData();
    }

    PushOwnMetaData();

    m_IsMetadataPushed = true;
}

//------------------------------------------------------------------------------
bool PlotItem::PushData(const void* buffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsAttached());

    return GetModule()->PushData(buffer, length);
}

//------------------------------------------------------------------------------
bool PlotItem::PushDataAt(const void* buffer, size_t length, nn::os::Tick tick) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsAttached());

    return GetModule()->PushDataAt(buffer, length, tick);
}

//------------------------------------------------------------------------------
void PlotItem::CheckTick(nn::os::Tick tick) NN_NOEXCEPT
{
    if (m_IsCheckTickEnabled)
    {
        if (tick < m_LastTick)
        {
            // ワーニングは一度だけ出力します。
            m_IsCheckTickEnabled = false;

            NN_DETAIL_SPY_WARN(
                NN_TEXT_SPY("[%s] 以前のプッシュよりも過去の tick を指定することはできません。\n  Item = \"%s\"\n  Node = \"%s\"\n"),
                NN_CURRENT_FUNCTION_NAME,
                GetName(),
                GetParent() != nullptr ? GetParent()->GetName() : "");
        }

        if (m_LastTick < tick)
        {
            m_LastTick = tick;
        }
    }
}

#endif // NN_BUILD_CONFIG_SPY_ENABLED

} // namespace nn::spy
} // namespace nn
