﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NN_BUILD_CONFIG_SPY_ENABLED

#include <nn/spy/detail/fnd/basis/spyfnd_Memory.h>
#include <nn/spy/detail/fnd/string/spyfnd_String.h>
#include <nn/spy/detail/spy_Packet.h>

#include <nn/spy/detail/spy_SpyDataFileChannel.h>

#if defined(NN_SPY_DATA_FILE_CHANNEL_AVAILABLE)

#if defined(NN_SDK_BUILD_DEBUG)
//#define STATE_DEBUG_ENABLED
//#define BUFFER_DEBUG_ENABLED
#endif

#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
#include <nn/config.h>
#endif

namespace {

#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
static uint32_t DefaultDataFileSizeLimit = 32 * 1024 * 1024;
#endif

}

namespace nn {
namespace spy {

//----------------------------------------------------------
SpyDataFileChannelParam::SpyDataFileChannelParam() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
    fsClient = NULL;
    dataFileSizeLimit = DefaultDataFileSizeLimit;
    hfioMountPath = NULL;
#endif
}

namespace detail {

//----------------------------------------------------------
SpyDataFileChannel::SpyDataFileChannel() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
    m_pFsClient = NULL;
    m_pFsCmdBlock = NULL;
    m_DataOutputFileOpen = false;
#endif
}

//----------------------------------------------------------
size_t
SpyDataFileChannel::GetRequiredMemorySize() NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    return 0;

#else

    return sizeof(FSCmdBlock);

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::Initialize(void* buffer) NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    NN_UNUSED(buffer);

#else

    NN_SDK_ASSERT_NOT_NULL(buffer);

    m_pFsClient = NULL;
    m_HfioMountPath[0] = '\0';
    m_OutputDirMountPath[0] = '\0';
    m_DataOutputFileOpen = false;

    m_pFsCmdBlock = reinterpret_cast<FSCmdBlock*>(buffer);
    FSInitCmdBlock(m_pFsCmdBlock);

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::Finalize() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_CAFE)

    if (this->IsOpened())
    {
        this->Close();
    }

    m_pFsCmdBlock = NULL;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::Open(const SpyDataFileChannelParam& param) NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    NN_UNUSED(param);
    return false;

#else

    NN_SDK_ASSERT(!this->IsOpened());

    m_pFsClient = NULL;

    if (param.fsClient != NULL)
    {
        if (param.hfioMountPath == NULL)
        {
            NN_SDK_ASSERT_NOT_NULL(param.hfioMountPath);
            return false;
        }

        if (!m_pFsCmdBlock)
        {
            NN_SDK_ASSERT_NOT_NULL(m_pFsCmdBlock);
            return false;
        }

        m_pFsClient = param.fsClient;
        m_MaxWriteLength = param.dataFileSizeLimit;
        nn::spy::detail::fnd::String::Copy(m_HfioMountPath, sizeof(m_HfioMountPath), param.hfioMountPath);
    }

    return true;
#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::IsOpened() const NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    return false;

#else

    return m_pFsClient != NULL;

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::Close() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
    if (this->IsOpened())
    {
        if (this->IsActive())
        {
            this->EndSession();
        }

        m_pFsClient = NULL;
    }
#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::BeginSession(const char* outputDir) NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    NN_UNUSED(outputDir);
    return false;

#else

    if (this->IsOpened())
    {
        return this->SetOutputDirPath(outputDir);
    }

    return false;

#endif
}

//----------------------------------------------------------
void
SpyDataFileChannel::EndSession() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
    if (this->IsOpened())
    {
        if (this->IsActive())
        {
            this->CloseFiles();
        }
    }
#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::IsActive() const NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    return false;

#else

    return m_DataOutputFileOpen;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::WriteData(void* buffer, size_t length) NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    NN_UNUSED(buffer);
    NN_UNUSED(length);
    return false;

#else

    nn::spy::detail::fnd::ScopedCriticalSection lock(m_DataFileLock);

    if (!m_DataOutputFileOpen)
    {
        return false;
    }

    FSFileHandle fileHandle = m_DataOutputFile[m_CurrentWriteFile];
    FSStatus result;

    result = FSWriteFile(
        m_pFsClient,
        m_pFsCmdBlock,
        buffer,
        1, // size
        length,
        fileHandle,
        0, // flag
        FS_RET_ALL_ERROR);

    if (result != length)
    {
        if (result >= 0)
        {
            NN_ABORT("FSWriteFile partial (%d)\n", result);
        }
        else
        {
            NN_ABORT("FSWriteFile failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
        }
    }

    m_WriteLength += result;

    // ファイルサイズが上限に達したら、出力ファイルを切り替える。
    if (m_WriteLength >= m_MaxWriteLength)
    {
        // Spy.exe のデータ読み込みを追い越さないようにするため、
        // Spy.exe が現在のファイルを読み込み始めてから出力ファイルを切り替える。
        // それまでは、上限をオーバーしても現在のファイルに追記を続ける。
        if (m_CurrentWriteFile == m_CurrentReadFile)
        {
            m_CurrentWriteFile = (m_CurrentWriteFile + 1) & 1;
            m_WriteLength = 0;

            // 次の出力ファイルを空にする。
            FSFileHandle newFileHandle = m_DataOutputFile[m_CurrentWriteFile];
            result = FSSetPosFile(m_pFsClient, m_pFsCmdBlock, newFileHandle, 0, FS_RET_ALL_ERROR);
            NN_SDK_ASSERT(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);
            result = FSTruncateFile(m_pFsClient, m_pFsCmdBlock, newFileHandle, FS_RET_ALL_ERROR);
            NN_SDK_ASSERT(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);

            // 出力ファイル切り替えパケットを送信。
#ifdef STATE_DEBUG_ENABLED
            NN_DETAIL_SPY_INFO("[nn::spy::SpyController] WriteData: add DataEndPacket\n");
#endif
            detail::DataEndPacket* packet = new(buffer) detail::DataEndPacket();
            result = FSWriteFile(
                m_pFsClient,
                m_pFsCmdBlock,
                buffer,
                1, // size
                sizeof(detail::DataEndPacket),
                fileHandle,
                0, // flag
                FS_RET_ALL_ERROR);

            if (result != sizeof(detail::DataEndPacket))
            {
                if (result >= 0)
                {
                    NN_ABORT("FSWriteFile partial (%d)\n", result);
                }
                else
                {
                    NN_ABORT("FSWriteFile failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
                }
            }
        }
    }

    return true;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::SetOutputDirPath(const char* path) NN_NOEXCEPT
{
#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    NN_UNUSED(path);
    return false;

#else

    NN_DETAIL_SPY_INFO("[nn::spy::detail::SpyDataFileChannel] SetOutputDirPath(path = %s)\n", path? path : "<null>");

    FSStatus result;

    // 未初期化状態に。
    m_OutputDirMountPath[0] = '\0';
    this->CloseFiles();

    if (!m_pFsClient)
    {
        return false;
    }

    if (path != NULL)
    {
        // FSにおける出力ディレクトリのパスを生成する。
        m_OutputDirMountPath[sizeof(m_OutputDirMountPath) - 1] = '\0';
        nw::ut::snprintf(m_OutputDirMountPath, sizeof(m_OutputDirMountPath), "%s/%s", m_HfioMountPath, path);
        int i = 0;
        for (; i < sizeof(m_OutputDirMountPath); ++i)
        {
            char c = m_OutputDirMountPath[i];
            if (c == '\0')
            {
                break;
            }
            else if (c == '\\' || c == ':')
            {
                m_OutputDirMountPath[i] = '/';
            }
        }

        // ヌル終端されていない。
        if (i == sizeof(m_OutputDirMountPath))
        {
            NN_DETAIL_SPY_WARN("[nn::spy::detail::SpyDataFileChannel] SetOutputDirPath: path too long.\n");
            m_OutputDirMountPath[0] = '\0';
            return false;
        }

#ifdef STATE_DEBUG_ENABLED
        NN_DETAIL_SPY_INFO("[nn::spy::detail::SpyDataFileChannel] SetOutputDirPath: m_OutputDirMountPath %s\n", m_OutputDirMountPath);
#endif

        // 出力ディレクトリが存在するかチェックする。
        // 見つからなければ、作成する。
        FSDirHandle handle;
        result = FSOpenDir(m_pFsClient, m_pFsCmdBlock, m_OutputDirMountPath, &handle, FS_RET_NOT_DIR | FS_RET_NOT_FOUND);
        if (result == FS_STATUS_NOT_FOUND)
        {
            result = FSMakeDir(m_FsClient, m_FsCmdBlock, m_OutputDirMountPath, FS_RET_NOT_DIR | FS_RET_NOT_FOUND);
            if (result == FS_STATUS_OK)
            {
                result = FSOpenDir(m_FsClient, m_FsCmdBlock, m_OutputDirMountPath, &handle, FS_RET_NOT_DIR | FS_RET_NOT_FOUND);
            }
        }
        if (result != FS_STATUS_OK)
        {
            NN_DETAIL_SPY_WARN("[nn::spy::detail::SpyDataFileChannel] SetOutputDirPath: dir not found (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
            m_OutputDirMountPath[0] = '\0';
            return false;
        }

        result = FSCloseDir(m_pFsClient, m_pFsCmdBlock, handle, FS_RET_ALL_ERROR);
        NN_SDK_ASSERT(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);

        return this->OpenFiles();
    }

    return false;

#endif
}

//----------------------------------------------------------
bool
SpyDataFileChannel::SetCurrentReadFile(int fileIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(fileIndex, 0);

#if !defined(NN_BUILD_CONFIG_SPEC_CAFE)

    NN_UNUSED(fileIndex);
    return false;

#else

    NW_ASSERT_MAX(fileIndex, UCHAR_MAX); // 負の値は不可。
    m_CurrentReadFile = static_cast<uint8_t>(fileIndex);
    return true;

#endif
}

#if defined(NN_BUILD_CONFIG_SPEC_CAFE)

//----------------------------------------------------------
bool
SpyDataFileChannel::OpenFiles() NN_NOEXCEPT
{
    if (!m_pFsClient)
    {
        return false;
    }

    if (m_DataOutputFileOpen)
    {
        return true;
    }

    FSStatus result;
    char fileName[FS_MAX_FULLPATH_SIZE];

    nn::spy::detail::fnd::ScopedCriticalSection dataLock(m_DataFileLock);

    nw::ut::snprintf(fileName, sizeof(fileName), "%s/SpyData0.spychan", m_OutputDirMountPath);
    result = FSOpenFile(m_pFsClient, m_pFsCmdBlock, fileName, "w", &(m_DataOutputFile[0]), FS_RET_ALL_ERROR);
    if (result != FS_STATUS_OK)
    {
        NN_DETAIL_SPY_WARN("[nn::spy::detail::SpyDataFileChannel] OpenFile: open file0 failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
        return false;
    }

    nw::ut::snprintf(fileName, sizeof(fileName), "%s/SpyData1.spychan", m_OutputDirMountPath);
    result = FSOpenFile(m_pFsClient, m_pFsCmdBlock, fileName, "w", &(m_DataOutputFile[1]), FS_RET_ALL_ERROR);
    if (result != FS_STATUS_OK)
    {
        NN_DETAIL_SPY_WARN("[nn::spy::detail::SpyDataFileChannel] OpenFile: open file1 failed (FS_STATUS_ERROR_BASE %d)\n", result - FS_STATUS_ERROR_BASE);
        FSCloseFile(m_pFsClient, m_pFsCmdBlock, m_DataOutputFile[0], FS_RET_ALL_ERROR);
        return false;
    }

    m_DataOutputFileOpen = true;
    m_CurrentWriteFile = 0;
    m_CurrentReadFile = static_cast<uint8_t>(-1);
    m_WriteLength = 0;

    return true;
}

//----------------------------------------------------------
void
SpyDataFileChannel::CloseFiles() NN_NOEXCEPT
{
    if (!m_DataOutputFileOpen)
    {
        return;
    }

    nn::spy::detail::fnd::ScopedCriticalSection dataLock(m_DataFileLock);

    FSStatus result;
    result = FSCloseFile(m_pFsClient, m_pFsCmdBlock, m_DataOutputFile[0], FS_RET_ALL_ERROR);
    NN_SDK_ASSERT(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);
    result = FSCloseFile(m_pFsClient, m_pFsCmdBlock, m_DataOutputFile[1], FS_RET_ALL_ERROR);
    NN_SDK_ASSERT(result == FS_STATUS_OK, "FSStatus FS_STATUS_ERROR_BASE %d\n", result - FS_STATUS_ERROR_BASE);

    m_DataOutputFileOpen = false;
}

#endif

} // namespace nn::spy::detail
} // namespace nn::spy
} // namespace nn

#endif // NN_SPY_DATA_FILE_CHANNEL_AVAILABLE

#endif // NN_BUILD_CONFIG_SPY_ENABLED
