﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_Endian.h>
#include <nn/util/util_BytePtr.h>
#include <nn/ige/ige_Buffer.h>

namespace nn { namespace ige { namespace detail {

class BufferReader
{
    NN_DISALLOW_COPY(BufferReader);
public:
    explicit BufferReader(BufferView workBuffer) NN_NOEXCEPT
        : m_WorkBuffer(workBuffer)
        , m_Position(0)
    {
    }

    int ReadInt() NN_NOEXCEPT
    {
        int value;
        Read(&value);
        return value;
    }

    bool ReadBool() NN_NOEXCEPT
    {
        return ReadInt8() != 0;
    }

    int8_t ReadInt8() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS(m_Position, m_WorkBuffer.GetSize());
        return *util::ConstBytePtr(m_WorkBuffer.GetPtr(), m_Position++).Get<int8_t>();
    }

    uint8_t ReadUInt8() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS(m_Position, m_WorkBuffer.GetSize());
        return *util::ConstBytePtr(m_WorkBuffer.GetPtr(), m_Position++).Get<uint8_t>();
    }

    int16_t ReadInt16() NN_NOEXCEPT
    {
        int16_t value;
        Read(&value);
        return value;
    }

    uint16_t ReadUInt16() NN_NOEXCEPT
    {
        uint16_t value;
        Read(&value);
        return value;
    }

    int32_t ReadInt32() NN_NOEXCEPT
    {
        int32_t value;
        Read(&value);
        return value;
    }

    uint32_t ReadUInt32() NN_NOEXCEPT
    {
        uint32_t value;
        Read(&value);
        return value;
    }

    int64_t ReadInt64() NN_NOEXCEPT
    {
        int64_t value;
        Read(&value);
        return value;
    }

    uint64_t ReadUInt64() NN_NOEXCEPT
    {
        uint64_t value;
        Read(&value);
        return value;
    }

    float ReadFloat() NN_NOEXCEPT
    {
        float value;
        Read(&value);
        return value;
    }

    double ReadDouble() NN_NOEXCEPT
    {
        double value;
        Read(&value);
        return value;
    }

    util::string_view ReadString() NN_NOEXCEPT
    {
        const char* ptr = util::ConstBytePtr(m_WorkBuffer.GetPtr(), m_Position).Get<char>();
        int length = util::Strnlen(ptr, static_cast<int>(m_WorkBuffer.GetSize() - m_Position));

        util::string_view str(ptr, length);

        m_Position += str.size();

        if (m_Position < m_WorkBuffer.GetSize())
        {
            ++m_Position; // '\0' 分
        }

        return str;
    }

    BufferView ReadBuffer(size_t size) NN_NOEXCEPT
    {
        if (size == 0)
        {
            return BufferView();
        }

        NN_SDK_ASSERT_LESS_EQUAL(m_Position + size, m_WorkBuffer.GetSize());
        BufferView buffer(util::ConstBytePtr(m_WorkBuffer.GetPtr(), m_Position).Get(), size);
        m_Position += size;
        return buffer;
    }

    BufferView ReadBuffer() NN_NOEXCEPT
    {
        return ReadBuffer(m_WorkBuffer.GetSize() - m_Position);
    }

    BufferView GetBuffer() const NN_NOEXCEPT
    {
        return m_WorkBuffer;
    }

    size_t GetPosition() const NN_NOEXCEPT
    {
        return m_Position;
    }

    size_t GetRemainingSize() const NN_NOEXCEPT
    {
        return m_WorkBuffer.GetSize() - m_Position;
    }

    template<typename T>
    void Read(T* ptr) NN_NOEXCEPT
    {
        size_t size = sizeof(T);

        NN_SDK_ASSERT_LESS_EQUAL(m_Position + size, m_WorkBuffer.GetSize());

        T value;
        std::memcpy(&value, util::ConstBytePtr(m_WorkBuffer.GetPtr(), m_Position).Get(), size);

        m_Position += size;

        util::StoreLittleEndian(ptr, value);
    }

private:
    BufferView m_WorkBuffer;
    size_t m_Position;
};

template<> void BufferReader::Read<bool>(bool* ptr) NN_NOEXCEPT;
template<> void BufferReader::Read<uint8_t>(uint8_t* ptr) NN_NOEXCEPT;
template<> void BufferReader::Read<int8_t>(int8_t* ptr) NN_NOEXCEPT;

}}} // namespace nn::ige::detail
