﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/ige/ige_Buffer.h>
#include "ige_IgeMemoryResource.h"

namespace nn { namespace ige { namespace detail {

class IgeAllocator
{
public:
    explicit IgeAllocator(IgeMemoryResource* pResource) NN_NOEXCEPT
        : m_pResource(pResource)
    {
        NN_SDK_REQUIRES_NOT_NULL(pResource);
    }

    NN_FORCEINLINE void* Allocate(size_t size) NN_NOEXCEPT
    {
        return Allocate(size, DefaultAlignment);
    }

    NN_FORCEINLINE void* Allocate(size_t size, size_t alignment) NN_NOEXCEPT
    {
        return m_pResource->allocate(size, alignment);
    }

    template<typename T>
    NN_FORCEINLINE void* Allocate() NN_NOEXCEPT
    {
        return m_pResource->allocate(sizeof(T), NN_ALIGNOF(T));
    }

    NN_FORCEINLINE void Free(void* ptr) NN_NOEXCEPT
    {
        m_pResource->Free(ptr);
    }

    NN_FORCEINLINE BufferReference AllocateBuffer(size_t size) NN_NOEXCEPT
    {
        void* ptr = m_pResource->allocate(size);
        return BufferReference(ptr, size);
    }

    NN_FORCEINLINE void FreeBuffer(BufferReference* pBuffer) NN_NOEXCEPT
    {
        if (pBuffer == NULL)
        {
            return;
        }

        if (void* ptr = pBuffer->GetPtr())
        {
            m_pResource->deallocate(ptr, pBuffer->GetSize());
            pBuffer->Set(NULL, 0);
        }
    }

    IgeMemoryResource* GetResource() const NN_NOEXCEPT
    {
        return m_pResource;
    }

private :
    IgeMemoryResource* m_pResource;
};

template <typename T>
class IgeStlAllocator : public std::allocator<T>
{
public:
    template <typename U> struct rebind
    {
        typedef IgeStlAllocator<U> other;
    };

    IgeStlAllocator()
        : m_pResource(NULL)
    {
        NN_SDK_ASSERT(false);
    }

    explicit IgeStlAllocator(MemoryResource* pResource)
        : m_pResource(pResource)
    {
        NN_SDK_REQUIRES_NOT_NULL(pResource);
    }

    template <typename U>
    NN_IMPLICIT IgeStlAllocator(const IgeStlAllocator<U>& other)
        : m_pResource(other.GetResource())
    {
    }

    NN_IMPLICIT IgeStlAllocator(const IgeAllocator& other)
        : m_pResource(other.GetResource())
    {
    }

    T* allocate(size_t count) NN_NOEXCEPT
    {
        return static_cast<T*>(m_pResource->allocate(count * sizeof(T), NN_ALIGNOF(T)));
    }

    T* allocate(size_t count, const void*) NN_NOEXCEPT
    {
        return allocate(count);
    }

    void deallocate(T* ptr, size_t count) NN_NOEXCEPT
    {
        m_pResource->deallocate(ptr, count * sizeof(T), NN_ALIGNOF(T));
    }

    MemoryResource* GetResource() const NN_NOEXCEPT
    {
        return m_pResource;
    }

private :
    MemoryResource* m_pResource;
};

// 比較演算子
template <class T, class U>
bool operator==(const IgeStlAllocator<T>& lhs, const IgeStlAllocator<U>& rhs)
{
    if (lhs.GetResource() == rhs.GetResource()) { return true; }

    return lhs.GetResource()->is_equal(*rhs.GetResource());
}

template <class T, class U>
bool operator!=(const IgeStlAllocator<T>& lhs, const IgeStlAllocator<U>& rhs)
{
    return !(lhs == rhs);
}

}}} // namespace nn::ige::detail
