﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "g3d_EditWorkBuffer.h"
#include "g3d_SynchronizedObject.h"
#include "g3d_ScopedLock.h"

namespace nn { namespace g3d { namespace viewer { namespace detail {

class SynchronizedEditWorkBuffer : public SynchronizedObject
{
public:
    explicit SynchronizedEditWorkBuffer(nn::g3d::viewer::detail::Allocator* pAllocator) NN_NOEXCEPT
        : m_WorkBuffer(pAllocator, AllocateType_DynamicBuffer)
    {
    }

    SynchronizedEditWorkBuffer(nn::g3d::viewer::detail::Allocator* pAllocator, size_t alignment) NN_NOEXCEPT
        : m_WorkBuffer(pAllocator, alignment, AllocateType_DynamicBuffer)
    {
    }

    void Clear() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_WorkBuffer.Clear();
    }

    void *GetWorkBufferPtr() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_WorkBuffer.GetWorkBufferPtr();
    }

    const void *GetWorkBufferPtr() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_WorkBuffer.GetWorkBufferPtr();
    }

    template<typename T>
    T* GetWorkBufferPtr() NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_WorkBuffer.GetWorkBufferPtr<T>();
    }

    template<typename T>
    const T* GetWorkBufferPtr() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_WorkBuffer.GetWorkBufferPtr<T>();
    }

    size_t GetSize() const NN_NOEXCEPT
    {
        ScopedLock scopedLock(*const_cast<nn::os::Mutex*>(&GetMutex()));
        return m_WorkBuffer.GetSize();
    }

    bool Resize(size_t size) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_WorkBuffer.Resize(size);
    }

    bool Resize(size_t size, size_t alignment) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        return m_WorkBuffer.Resize(size, alignment);
    }

    void CopyFrom(const void* pBuffer, size_t size) NN_NOEXCEPT
    {
        ScopedLock scopedLock(GetMutex());
        m_WorkBuffer.CopyFrom(pBuffer, size);
    }

private:
    EditWorkBuffer m_WorkBuffer;
};

}}}} // namespace nn::g3d::viewer::detail
