﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_ResourceEditUtility.h"
#include "../g3d_ViewerDetailDefine.h"
#include "../model/g3d_EditModelObj.h"
#include <nn/g3d/viewer/detail/g3d_ViewerPacketDefine.h>

#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_ResMaterial.h>
#include <nn/g3d/g3d_ResMaterialAnim.h>

using namespace nn::g3d;
using namespace nn::g3d::viewer::detail;

#define EDIT_KIND_TO_TYPE(kind, type) case kind: return type;
#define EDIT_TYPE_TO_SIZE(type, size) case type: return size;

namespace {
    template<typename TValue>
    void SetMaterialSamplerValue(
        nn::gfx::Device* pDevice,
        EditModelObj* pEditModelObj, const int materialIndices[], int indexCount, int samplerIndex, TValue value,
        void(*setSamplerValueFunction)(nn::gfx::SamplerInfo* pSamplerInfo, TValue pValue)) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(materialIndices, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_DETAIL(indexCount > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_DETAIL(samplerIndex >= 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));

        for (int modelIndex = 0, modelCount = pEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            ModelObj* pModelObj = pEditModelObj->GetTargetModelObj(modelIndex);

            for (int matIndex = 0; matIndex < indexCount; ++matIndex)
            {
                MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndices[matIndex]);

                if (samplerIndex >= pMaterialObj->GetSamplerCount())
                {
                    continue;
                }

                nn::g3d::SamplerRef samplerRef = pMaterialObj->GetSampler(samplerIndex);
                nn::gfx::Sampler* pSampler = const_cast<nn::gfx::Sampler*>(samplerRef.GetSampler());
                NN_G3D_VIEWER_ASSERT_NOT_NULL(pSampler);

                const ResMaterial* resMaterial = pMaterialObj->GetResource();
                const nn::gfx::SamplerInfo* pInfo = resMaterial->GetSamplerInfo(samplerIndex);
                NN_G3D_VIEWER_ASSERT_NOT_NULL(pInfo);

                setSamplerValueFunction(const_cast<nn::gfx::SamplerInfo*>(pInfo), value);

                pSampler->Finalize(pDevice);
                pSampler->Initialize(pDevice, *pInfo);
            }
        }
    }

    void SetSamplerAddressU(nn::gfx::SamplerInfo* pInfo, nn::gfx::TextureAddressMode value) NN_NOEXCEPT
    {
        pInfo->SetAddressU(value);
    }

    void SetSamplerAddressV(nn::gfx::SamplerInfo* pInfo, nn::gfx::TextureAddressMode value) NN_NOEXCEPT
    {
        pInfo->SetAddressV(value);
    }

    void SetSamplerAddressW(nn::gfx::SamplerInfo* pInfo, nn::gfx::TextureAddressMode value) NN_NOEXCEPT
    {
        pInfo->SetAddressW(value);
    }

    void SetSamplerFilter(nn::gfx::SamplerInfo* pInfo, nn::gfx::FilterMode value) NN_NOEXCEPT
    {
        pInfo->SetFilterMode(value);
    }

    void SetSamplerMaxAnisotoropy(nn::gfx::SamplerInfo* pInfo, int value) NN_NOEXCEPT
    {
        pInfo->SetMaxAnisotropy(value);
    }

    void SetSamplerMinLod(nn::gfx::SamplerInfo* pInfo, float value) NN_NOEXCEPT
    {
        pInfo->SetMinLod(value);
    }

    void SetSamplerMaxLod(nn::gfx::SamplerInfo* pInfo, float value) NN_NOEXCEPT
    {
        pInfo->SetMaxLod(value);
    }

    void SetSamplerLodBias(nn::gfx::SamplerInfo* pInfo, float value) NN_NOEXCEPT
    {
        pInfo->SetLodBias(value);
    }

    void SetMaterialSamplerWrapU(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, nn::gfx::TextureAddressMode value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerAddressU);
    }

    void SetMaterialSamplerWrapV(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, nn::gfx::TextureAddressMode value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerAddressV);
    }

    void SetMaterialSamplerWrapW(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, nn::gfx::TextureAddressMode value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerAddressW);
    }

    void SetMaterialSamplerFilter(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, nn::gfx::FilterMode value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerFilter);
    }

    void SetMaterialSamplerMaxAnisotoropy(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, int value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerMaxAnisotoropy);
    }

    void SetMaterialSamplerMinLod(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, float value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerMinLod);
    }

    void SetMaterialSamplerMaxLod(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, float value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerMaxLod);
    }

    void SetMaterialSamplerLodBias(
        nn::gfx::Device* pDevice,
        EditModelObj* pModelObj, const int indices[], int indexCount, int samplerIndex, float value) NN_NOEXCEPT
    {
        SetMaterialSamplerValue(pDevice, pModelObj, indices, indexCount, samplerIndex, value, SetSamplerLodBias);
    }

    void SetMaterialVisible(EditModelObj* pEditModelObj, const int indices[], int indexCount, bool visible) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_DETAIL(indexCount > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));

        for (int modelIndex = 0, modelCount = pEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            ModelObj* pModelObj = pEditModelObj->GetTargetModelObj(modelIndex);

            for (int i = 0; i < indexCount; ++i)
            {
                pModelObj->SetMaterialVisible(indices[i], visible);
                pEditModelObj->SetMaterialVisible(modelIndex, indices[i], visible);
            }
        }
    }

    nn::g3d::ResShaderParam::Type GetResShaderParamType(EditTargetKind kind) NN_NOEXCEPT
    {
        switch(kind)
        {
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL,   ResShaderParam::Type_Bool)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2,  ResShaderParam::Type_Bool2)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3,  ResShaderParam::Type_Bool3)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4,  ResShaderParam::Type_Bool4)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT,    ResShaderParam::Type_Int)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2,   ResShaderParam::Type_Int2)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3,   ResShaderParam::Type_Int3)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4,   ResShaderParam::Type_Int4)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT,   ResShaderParam::Type_Uint)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2,  ResShaderParam::Type_Uint2)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3,  ResShaderParam::Type_Uint3)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4,  ResShaderParam::Type_Uint4)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT,  ResShaderParam::Type_Float)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2, ResShaderParam::Type_Float2)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3, ResShaderParam::Type_Float3)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4, ResShaderParam::Type_Float4)

            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2, ResShaderParam::Type_Float2x2)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3, ResShaderParam::Type_Float2x3)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4, ResShaderParam::Type_Float2x4)

            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2, ResShaderParam::Type_Float3x2)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3, ResShaderParam::Type_Float3x3)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4, ResShaderParam::Type_Float3x4)

            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2, ResShaderParam::Type_Float4x2)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3, ResShaderParam::Type_Float4x3)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4, ResShaderParam::Type_Float4x4)

            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D, ResShaderParam::Type_Srt3d)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D, ResShaderParam::Type_Srt2d)

            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA,      ResShaderParam::Type_Texsrt)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX,    ResShaderParam::Type_Texsrt)
            EDIT_KIND_TO_TYPE(EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE, ResShaderParam::Type_Texsrt)

        default:
            break;
        }
        return ResShaderParam::Type_Num;
    }

    int GetArraySize(nn::g3d::ResShaderParam::Type type) NN_NOEXCEPT
    {
        switch(type)
        {
            EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Bool,   1)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Bool2,  2)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Bool3,  3)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Bool4,  4)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Int,    1)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Int2,   2)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Int3,   3)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Int4,   4)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Uint,   1)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Uint2,  2)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Uint3,  3)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Uint4,  4)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float,  1)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float2, 2)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float3, 3)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float4, 4)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float2x2, 4)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float2x3, 6)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float2x4, 8)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float3x2, 6)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float3x3, 9)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float3x4, 12)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float4x2, 8)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float4x3, 12)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Float4x4, 16)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Srt3d, 9)
                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Srt2d, 5)

                EDIT_TYPE_TO_SIZE(ResShaderParam::Type_Texsrt,           6)

        default:
            break;
        }
        return 0;
    }

    template<typename TValue>
    void SetMaterialShaderParamValue(
        EditModelObj* pEditModelObj,
        const char* paramName,
        const int matIndices[],
        int matIndexCount,
        ResShaderParam::Type paramType,
        const TValue value[],
        int valueSize) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(paramName, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(matIndices, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_DETAIL(matIndexCount > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));

        for (int modelIndex = 0, modelCount = pEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            for (int index = 0; index < matIndexCount; ++index)
            {
                EditMaterialObj* pEditMaterialObj = pEditModelObj->GetEditMaterialObj(modelIndex, matIndices[index]);
                pEditMaterialObj->EditShaderParam<TValue>(paramName, paramType, value, valueSize);
            }
        }
    }

    void SetMaterialShaderParamTexSrt(
        EditModelObj* pEditModelObj,
        const char* paramName,
        const int matIndices[],
        int indexCount,
        TexSrt::Mode mode,
        ResShaderParam::Type paramType,
        const float value[],
        int valueSize) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(paramName, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(matIndices, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_DETAIL(indexCount > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pEditModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT(paramType == ResShaderParam::Type_Texsrt);

        for (int modelIndex = 0, modelCount = pEditModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            for (int index = 0; index < indexCount; ++index)
            {
                EditMaterialObj* pEditMaterialObj = pEditModelObj->GetEditMaterialObj(modelIndex, matIndices[index]);
                pEditMaterialObj->EditTextureSrt(paramName, mode, paramType, value, valueSize);
            }
        }
    }

    void SetBoneVisibility(ModelObj* pModelObj, const int indices[], int indexCount, bool visible) NN_NOEXCEPT
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NN_G3D_VIEWER_RES_NAME(pModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_DETAIL(indexCount > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pModelObj->GetResource(), GetName()));

        for (int i = 0; i < indexCount; ++i)
        {
            pModelObj->SetBoneVisible(indices[i], visible);
        }
    }

    void SetBoneBillboard(ModelObj* pModelObj, const int indices[], int indexCount, uint32_t flag)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(indices, "%s\n", NN_G3D_VIEWER_RES_NAME(pModelObj->GetResource(), GetName()));
        NN_G3D_VIEWER_ASSERT_DETAIL(indexCount > 0, "%s\n", NN_G3D_VIEWER_RES_NAME(pModelObj->GetResource(), GetName()));

        const uint32_t clearMask = static_cast<uint32_t>(~ResBone::Flag_BillboardMax);
        SkeletonObj* skeletonObj = pModelObj->GetSkeleton();
        for (int i = 0; i < indexCount; ++i)
        {
            ResBone* resBone = const_cast<ResBone*>(skeletonObj->GetResBone(indices[i]));
            if (resBone != nullptr)
            {
                resBone->ToData().flag &= clearMask;
                resBone->ToData().flag |= flag;
            }
        }
        nn::g3d::ResSkeleton* resSkeleton = const_cast<nn::g3d::ResSkeleton*>(skeletonObj->GetResource());
        resSkeleton->UpdateBillboardMode();
    }
}

bool nn::g3d::viewer::detail::EditMaterials(nn::gfx::Device* pDevice, EditModelObj* pModelObj, const EditMaterialArg& arg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pModelObj);
    bool result = true;
    switch(arg.editTargetKind)
    {
    case EDIT_TARGET_MATERIAL_VISIBILITY:
        {
            const EditSimpleValue* editValue =
                static_cast<const EditSimpleValue*>(arg.value);
            SetMaterialVisible(pModelObj, arg.index, arg.indexCount, editValue->bValue);
        }
        break;
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
        {
            const EditSamplerValue* edit_value =
                static_cast<const EditSamplerValue*>(arg.value);
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
                SetMaterialSamplerWrapU(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->addressMode);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
                SetMaterialSamplerWrapV(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->addressMode);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
                SetMaterialSamplerWrapW(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->addressMode);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
                SetMaterialSamplerFilter(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->fiterMode);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
                SetMaterialSamplerFilter(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->fiterMode);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
                SetMaterialSamplerFilter(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->fiterMode);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
                SetMaterialSamplerMaxAnisotoropy(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->maxAnisotoropy);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
                SetMaterialSamplerMinLod(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->lod);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
                SetMaterialSamplerMaxLod(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->lod);
                break;
            case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
                SetMaterialSamplerLodBias(pDevice, pModelObj, arg.index, arg.indexCount, edit_value->samplerIndex, edit_value->bias);
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4:
        {
            const EditShaderParamVectorValue* edit_value =
                static_cast<const EditShaderParamVectorValue*>(arg.value);
            ResShaderParam::Type type = GetResShaderParamType(static_cast<EditTargetKind>(arg.editTargetKind));
            int valueSize = GetArraySize(type);
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4:
                SetMaterialShaderParamValue<uint32_t>(
                    pModelObj, reinterpret_cast<const char*>(edit_value->paramName),
                    arg.index, arg.indexCount,
                    type,
                    edit_value->value.uValue, valueSize);
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4:
                SetMaterialShaderParamValue<int32_t>(
                    pModelObj, reinterpret_cast<const char*>(edit_value->paramName),
                    arg.index, arg.indexCount,
                    type,
                    edit_value->value.sValue, valueSize);
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3:
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4:
                SetMaterialShaderParamValue<float>(
                    pModelObj, reinterpret_cast<const char*>(edit_value->paramName),
                    arg.index, arg.indexCount,
                    type,
                    edit_value->value.fValue, valueSize);
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D:
        {
            const EditShaderParamMatrixValue* editValue =
                static_cast<const EditShaderParamMatrixValue*>(arg.value);
            ResShaderParam::Type type = GetResShaderParamType(static_cast<EditTargetKind>(arg.editTargetKind));
            int valueSize = GetArraySize(type);
            SetMaterialShaderParamValue<float>(
                pModelObj, reinterpret_cast<const char*>(editValue->paramName),
                arg.index, arg.indexCount,
                type,
                editValue->value.a, valueSize);
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE:
        {
            // あとでパケットに含める形に変更
            TexSrt::Mode mode;
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX:
                mode = TexSrt::Mode_3dsmax;
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE:
                mode = TexSrt::Mode_Softimage;
                break;
            case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA:
                mode = TexSrt::Mode_Maya;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            const EditShaderParamMatrixValue* editValue =
                static_cast<const EditShaderParamMatrixValue*>(arg.value);
            ResShaderParam::Type type = GetResShaderParamType(static_cast<EditTargetKind>(arg.editTargetKind));
            int valueSize = GetArraySize(type) - 1; // パケットに含めていないのでサイズが一つ余分なので減らす
            SetMaterialShaderParamTexSrt(
                pModelObj, reinterpret_cast<const char*>(editValue->paramName),
                arg.index, arg.indexCount,
                mode,
                type,
                editValue->value.a, valueSize);
        }
        break;
    default:
        result = false;
    }
    return result;
} // NOLINT (readability/fn_size)

bool nn::g3d::viewer::detail::EditBones(ModelObj* pModelObj, const EditBoneArg& arg) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pModelObj, "%s\n", NN_G3D_VIEWER_RES_NAME(pModelObj->GetResource(), GetName()));
    bool result = true;
    switch(arg.editTargetKind)
    {
    case EDIT_TARGET_BONE_VISIBILITY:
    case EDIT_TARGET_BONE_BILLBOARD:
        {
            const EditSimpleValue* editValue =
                static_cast<const EditSimpleValue*>(arg.value);
            switch(arg.editTargetKind)
            {
            case EDIT_TARGET_BONE_VISIBILITY:
                SetBoneVisibility(pModelObj, arg.index, arg.indexCount, editValue->bValue);
                break;
            case EDIT_TARGET_BONE_BILLBOARD:
                SetBoneBillboard(pModelObj, arg.index, arg.indexCount, editValue->flag);
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        break;
    default:
        result = false;
    }
    return result;
}

bool nn::g3d::viewer::detail::EditRenderInfoArraySize(ModelObj* pModelObj, const char* labelName, const RenderInfoEditInfo* info) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pModelObj);
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(labelName, "%s\n", NN_G3D_VIEWER_RES_NAME(pModelObj->GetResource(), GetName()));
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(info, "%s\n", NN_G3D_VIEWER_RES_NAME(pModelObj->GetResource(), GetName()));
    MaterialObj* materialObj = pModelObj->GetMaterial(info->materialIndex);

    ResRenderInfo* resRenderInfo = const_cast<ResRenderInfo*>(materialObj->GetResource()->FindRenderInfo(labelName));
    if (resRenderInfo == nullptr)
    {
        return false;
    }
    resRenderInfo->ToData().arrayLength = static_cast<uint16_t>(info->arraySize);
    return true;
}

void nn::g3d::viewer::detail::SetShaderParamAnimationCurve(ResMaterialAnim* resShaderParamAnim, int materialIndex) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL(resShaderParamAnim);
    ResPerMaterialAnim* resShaderParamMatAnim = resShaderParamAnim->GetPerMaterialAnim(materialIndex);
    if (resShaderParamMatAnim == nullptr)
    {
        return;
    }

    ResAnimCurve* resAnimCurve = resShaderParamMatAnim->GetCurve(0);
    (void)resAnimCurve;
}


