﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/viewer/g3d_ViewerDefine.h>



#include "../util/g3d_DynamicPtrArray.h"
#include "../util/g3d_DynamicLengthString.h"
#include "../util/g3d_EditWorkBuffer.h"

#include <nn/g3d/g3d_ResMaterial.h>

namespace nn { namespace g3d { namespace viewer { namespace detail {

class EditSocketBase;
class Allocator;

class EditRenderInfo
{
public:
    explicit EditRenderInfo(Allocator* allocator) NN_NOEXCEPT
        : m_pAllocator(allocator)
        , m_WorkBuffer(allocator, AllocateType_DynamicBuffer)
        , m_Labels(allocator, nn::g3d::detail::Alignment_Default)
        , m_Items(allocator, nn::g3d::detail::Alignment_Default)
        , m_Values(allocator, nn::g3d::detail::Alignment_Default)
        , m_RenderInfoBufferSize(0)
        , m_ModelKey(0)
        , m_MaterialIndex(-1)
    {
        NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
    }

    ~EditRenderInfo() NN_NOEXCEPT
    {
        Clear();
        m_pAllocator = nullptr;
    }

    ViewerResult PushChoice(const char* labelName, const char* itemName, const char* aliasItemName) NN_NOEXCEPT;
    ViewerResult PushChoice(const char* labelName, int minValue, int maxValue) NN_NOEXCEPT;
    ViewerResult PushChoice(const char* labelName, float minValue, float maxValue) NN_NOEXCEPT;

    ViewerResult PushDefault(const char* labelName, const char* value) NN_NOEXCEPT;
    ViewerResult PushDefault(const char* labelName, int value) NN_NOEXCEPT;
    ViewerResult PushDefault(const char* labelName, float value) NN_NOEXCEPT;

    void Clear() NN_NOEXCEPT;

    bool MakeRenderInfoPacket() NN_NOEXCEPT;

    bool SendRenderInfo(EditSocketBase* socket) NN_NOEXCEPT;

    void SetModelKey(ViewerKeyType modelKey) NN_NOEXCEPT
    {
        m_ModelKey = modelKey;
    }
    void SetMaterialIndex(int index) NN_NOEXCEPT
    {
        m_MaterialIndex = index;
    }

private:
    ViewerResult PushLabel(const char* labelName, ResRenderInfo::Type type) NN_NOEXCEPT;
    int FindLabelIndex(const char* labelName) const NN_NOEXCEPT;

    ViewerResult PushItem(int labelIndex, const char* itemName, const char* aliasItemName) NN_NOEXCEPT;
    int FindItemIndex(int labelIndex, const char* itemName) const NN_NOEXCEPT;

    ViewerResult PushValue(int labelIndex, const char* value) NN_NOEXCEPT;
    ViewerResult PushValue(int labelIndex, int value) NN_NOEXCEPT;
    ViewerResult PushValue(int labelIndex, float value) NN_NOEXCEPT;

    size_t CalculateWorkBufferSize() NN_NOEXCEPT;
    void MakeRenderInfoBuffer() NN_NOEXCEPT;
private:
    class RenderInfoLabel
    {
        friend class EditRenderInfo;
    public:
        explicit RenderInfoLabel(Allocator* allocator) NN_NOEXCEPT
            : m_pAllocator(allocator)
            , m_Label(allocator)
            , m_LabelBinarySize(0)
            , m_Offset(0)
            , m_ItemOffset(0)
            , m_ItemCount(0)
            , m_ValueOffset(0)
            , m_ValueCount(0)
            , m_Type(ResRenderInfo::Type_String)
        {
            NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
        }

        ~RenderInfoLabel() NN_NOEXCEPT
        {
            m_pAllocator = nullptr;
        }

        ViewerResult Setup(const char* labelName, ResRenderInfo::Type type) NN_NOEXCEPT;

        bool IsEqual(const char* labelName) const NN_NOEXCEPT;

        const char* GetName() const NN_NOEXCEPT
        {
            return m_Label.GetStr();
        }
        size_t GetNameLength() const NN_NOEXCEPT
        {
            return m_Label.GetLength();
        }
    private:
        Allocator* m_pAllocator;
        DynamicLengthString m_Label;
        size_t m_LabelBinarySize;
        size_t m_Offset;
        size_t m_ItemOffset;
        int m_ItemCount;
        size_t m_ValueOffset;
        int m_ValueCount;
        ResRenderInfo::Type m_Type;

        union
        {
            float fValue;
            int iValue;
        } m_Min;

        union
        {
            float fValue;
            int iValue;
        } m_Max;
    };

    class RenderInfoItem
    {
        friend class EditRenderInfo;
    public:
        explicit RenderInfoItem(Allocator* allocator) NN_NOEXCEPT
            : m_pAllocator(allocator)
            , m_LabelIndex(-1)
            , m_Item(allocator)
            , m_Alias(allocator, nn::g3d::detail::Alignment_Default, AllocateType_Communication)
            , m_ItemBinarySize(0)
            , m_AliasBinarySize(0)
            , m_AliasLength(0)
            , m_Offset(0)
            , m_AliasOffset(0)
            , m_ItemIndex(0)
        {
            NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
        }

        ~RenderInfoItem() NN_NOEXCEPT
        {
            m_LabelIndex = -1;
            m_Alias.Clear();
            m_pAllocator = nullptr;
        }

        ViewerResult Setup(int labelIndex, const char* itemName, const char* aliasItemName) NN_NOEXCEPT;

        bool IsEqual(int labelIndex, const char* itemName) const NN_NOEXCEPT;

        const char* GetName() const NN_NOEXCEPT
        {
            return m_Item.GetStr();
        }
        size_t GetNameLength() const NN_NOEXCEPT
        {
            return m_Item.GetLength();
        }
        int GetLabelIndex() const NN_NOEXCEPT
        {
            return m_LabelIndex;
        }
    private:
        Allocator* m_pAllocator;
        int m_LabelIndex;
        DynamicLengthString m_Item;
        EditWorkBuffer m_Alias;
        size_t m_ItemBinarySize;
        size_t m_AliasBinarySize;
        size_t m_AliasLength;
        size_t m_Offset;
        size_t m_AliasOffset;
        int m_ItemIndex;
    };

    class RenderInfoValue
    {
        friend class EditRenderInfo;
    public:
        RenderInfoValue(ResRenderInfo::Type type, Allocator* allocator) NN_NOEXCEPT
            : m_pAllocator(allocator)
            , m_LabelIndex(-1)
            , m_DefaultString(allocator)
            , m_DefaultBinarySize(0)
            , m_Offset(0)
            , m_ValueIndex(0)
            , m_Type(type)
        {
            NN_G3D_VIEWER_ASSERT_NOT_NULL(allocator);
        }

        ~RenderInfoValue() NN_NOEXCEPT
        {
            m_LabelIndex = -1;
            m_pAllocator = nullptr;
        }

        ViewerResult Setup(int labelIndex, const char* value) NN_NOEXCEPT;
        ViewerResult Setup(int labelIndex, int value) NN_NOEXCEPT;
        ViewerResult Setup(int labelIndex, float value) NN_NOEXCEPT;

        const char* GetStringDefault() const NN_NOEXCEPT
        {
            return m_DefaultString.GetStr();
        }
        size_t GetStringDefaultLength() const NN_NOEXCEPT
        {
            return m_DefaultString.GetLength();
        }
        int GetLabelIndex() const NN_NOEXCEPT
        {
            return m_LabelIndex;
        }
    private:
        Allocator* m_pAllocator;
        int m_LabelIndex;
        DynamicLengthString m_DefaultString;
        size_t m_DefaultBinarySize;

        union
        {
            size_t m_Offset;
            int m_iValue;
            float m_fValue;
        };

        int m_ValueIndex;
        ResRenderInfo::Type m_Type;
    };

    Allocator* m_pAllocator;
    EditWorkBuffer m_WorkBuffer;
    DynamicPtrArray<RenderInfoLabel> m_Labels;
    DynamicPtrArray<RenderInfoItem> m_Items;
    DynamicPtrArray<RenderInfoValue> m_Values;
    size_t m_RenderInfoBufferSize;
    ViewerKeyType m_ModelKey;
    int m_MaterialIndex;
};

}}}} // namespace nn::g3d::viewer::detail


