﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditCommandExecutor.h"



#include <nn/g3d/g3d_ResShader.h>
#include "util/g3d_ScopedAllocator.h"
#include "g3d_EditCommandManager.h"
#include "model/g3d_EditModelObj.h"
#include "shader/g3d_EditShaderArchive.h"
#include <nn/g3d/g3d_ResAnimCurve.h>
#include "model/g3d_EditRenderInfo.h"
#include "command/g3d_SendRenderInfoDefinitionCommand.h"
#include "g3d_ResourceManager.h"

using namespace nn::g3d;
using namespace nn::g3d::viewer;
using namespace nn::g3d::viewer::detail;

EditCommandExecutor::EditCommandExecutor(
    Allocator* pAllocator,
    CallbackCaller* pCallback,
    EditCommandManager* pEditCommandManager) NN_NOEXCEPT
    : m_pAllocator(pAllocator)
    , m_pCallbackCaller(pCallback)
    , m_pEditCommandManager(pEditCommandManager)
{
}

ResourceManager* EditCommandExecutor::GetResourceManager()
{
    return m_pEditCommandManager->GetResourceManager();
}

void EditCommandExecutor::ExecuteUpdateShadingModels(EditValueInfoBlock* pBlock) NN_NOEXCEPT
{
    switch(pBlock->editTargetKind)
    {
    case EDIT_TARGET_UPDATE_SHADING_MODEL:
        {
            EditShadingModelValueBlock* valueBlock = reinterpret_cast<EditShadingModelValueBlock*>(pBlock + 1);
            EditShaderArchive* pTargetEditShaderArchive = GetResourceManager()->FindEditShaderArchive(pBlock->key);
            if (pTargetEditShaderArchive == nullptr)
            {
                NN_G3D_VIEWER_INTERNAL_WARNING("EditShaderArchive key %d was not found\n", pBlock->key);
                ViewerKeyManager::GetInstance().DumpKeyDataMap();
                m_pEditCommandManager->QueueErrorCommand(RUNTIME_ERROR_CODE_UNKNOWN_ERROR);
                return;
            }

            int modifiedShadingModelIndexCount = pBlock->indexCount;
            if (pTargetEditShaderArchive->GetEditShadingModelCount() < modifiedShadingModelIndexCount)
            {
                // 与えられたシェーディングモデル数がおかしいのでランタイムエラーにする
                m_pEditCommandManager->QueueErrorCommand(RUNTIME_ERROR_CODE_INVALID_SHADER_DETECTED);
                return;
            }

            GetResourceManager()->UpdateEditShaderArchive(pBlock->key, valueBlock->index, modifiedShadingModelIndexCount);

            {
                ShadingModelUpdatedArg arg;
                arg.pShaderArchive = GetResourceManager()->FindEditShaderArchive(pBlock->key)->GetTargetResShaderArchive();
                arg.shadingModelIndices = valueBlock->index;
                arg.shadingModelCount = pBlock->indexCount;
                m_pCallbackCaller->Call(CallbackType_ShadingModelUpdated, &arg);
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteEditRenderInfo(RenderInfoEditInfo* pInfo) NN_NOEXCEPT
{
    const char* labelName = AddOffset<const char>(pInfo, pInfo->labelOffset - sizeof(PacketHeader));

    EditModelObj* editModelObj =
        GetResourceManager()->FindEditModelObj(pInfo->modelKey);
    bool isValid = true;
    if (editModelObj == nullptr)
    {
        isValid = false;
    }

    if (isValid)
    {
        for (int modelIndex = 0, modelCount = editModelObj->GetAttachedModelObjCount(); modelIndex < modelCount; ++modelIndex)
        {
            ModelObj* pModelObj = editModelObj->GetTargetModelObj(modelIndex);
            MaterialObj* pMaterialObj = pModelObj->GetMaterial(pInfo->materialIndex);
            NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pMaterialObj, "%s\n", NN_G3D_VIEWER_RES_NAME(pMaterialObj->GetResource(), GetName()));

            nn::g3d::ResRenderInfo* pResRenderInfo = const_cast<nn::g3d::ResRenderInfo*>(pMaterialObj->GetResource()->FindRenderInfo(labelName));
            if (pResRenderInfo)
            {
                NN_G3D_VIEWER_ASSERT_DETAIL(pResRenderInfo->GetType() == static_cast<ResRenderInfo::Type>(pInfo->type), NN_G3D_VIEWER_RES_NAME(pMaterialObj->GetResource(), GetName()));

                if (pInfo->slotIndex >= 0 && pInfo->slotIndex < pResRenderInfo->GetArrayLength())
                {
                    // 数値型はここで直接 ResRenderInfo を書き換えます。
                    switch(static_cast<ResRenderInfo::Type>(pInfo->type))
                    {
                    case ResRenderInfo::Type_Int:
                        pResRenderInfo->ToData().intValueArray.Get()[pInfo->slotIndex] = pInfo->iValue;
                        break;
                    case ResRenderInfo::Type_Float:
                        pResRenderInfo->ToData().floatValueArray.Get()[pInfo->slotIndex] = pInfo->fValue;
                        break;
                    case ResRenderInfo::Type_String:
                        break;
                    default:
                        NN_UNEXPECTED_DEFAULT;
                    }
                }
            }

            if (m_pCallbackCaller)
            {
                RenderInfoUpdatedArg arg;
                arg.pModelObj = pModelObj;
                arg.materialIndex = pInfo->materialIndex;
                arg.name = labelName;
                m_pCallbackCaller->Call(CallbackType_RenderInfoUpdated, &arg);
            }
        }
    }
}

void EditCommandExecutor::ExecuteUnloadFile(FileDataBlock* pBlock) NN_NOEXCEPT
{
    switch(pBlock->kind)
    {
    case FILEDATA_MODEL:
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Unload Target Model Key: %d\n", pBlock->key);
            NN_G3D_VIEWER_DEBUG_PRINT("Listing All EditModelObj ---\n");
            GetResourceManager()->PrintAllEditModelObjs();
            NN_G3D_VIEWER_DEBUG_PRINT("---\n");
            m_pEditCommandManager->ScheduleDestructEditModelObj(pBlock->key);
            GetResourceManager()->ScheduleDestroyUnusedResFiles();
        }
        break;
    case FILEDATA_TEXTURE:
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Unload Target Texture Key: %d\n", pBlock->key);
            m_pEditCommandManager->UnloadResTextureFile(pBlock->key);
        }
        break;
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
    case FILEDATA_MATERIAL_ANIM:
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Unload Target Anim Key: %d\n", pBlock->key);
            GetResourceManager()->DeleteEditAnimObj(pBlock->key);
            GetResourceManager()->ScheduleDestroyAnimResFile(pBlock->key);
        }
        break;
    case FILEDATA_SHADER_ARCHIVE:
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Unload Target Shader Key: %d\n", pBlock->key);
            GetResourceManager()->DeleteEditShaderArchive(pBlock->key);
            GetResourceManager()->ScheduleDestroyUnusedResFiles();
        }
        break;
    case FILEDATA_SCENE_ANIM:
        {
            NN_G3D_VIEWER_DEBUG_PRINT("Unload Target Scene Anim Key: %d\n", pBlock->key);
            GetResourceManager()->DeleteEditSceneAnimObj(pBlock->key);
            GetResourceManager()->ScheduleDestroyUnusedResFiles();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteEditMaterial(EditValueInfoBlock* pBlock) NN_NOEXCEPT
{
    switch(pBlock->editTargetKind)
    {
    case EDIT_TARGET_MATERIAL_VISIBILITY:
        {
            EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexCount = pBlock->indexCount;

            GetResourceManager()->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_CONSTANT_COLOR:
        {
            EditVector4ValueBlock* valueBlock = reinterpret_cast<EditVector4ValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexCount = pBlock->indexCount;

            GetResourceManager()->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V:
    case EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO:
    case EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD:
    case EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS:
        {
            EditSamplerValueBlock* valueBlock = reinterpret_cast<EditSamplerValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexCount = pBlock->indexCount;

            GetResourceManager()->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4:
        {
            EditShaderParamVectorValueBlock* valueBlock = reinterpret_cast<EditShaderParamVectorValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexCount = pBlock->indexCount;

            GetResourceManager()->EditMaterials(arg);
        }
        break;
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX:
    case EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE:
        {
            EditShaderParamMatrixValueBlock* valueBlock = reinterpret_cast<EditShaderParamMatrixValueBlock*>(pBlock + 1);
            EditMaterialArg arg;
            arg.key = pBlock->key;
            arg.valueKind = pBlock->valueKind;
            arg.editTargetKind = pBlock->editTargetKind;
            arg.value = static_cast<void*>(&valueBlock->value);
            arg.index = valueBlock->index;
            arg.indexCount = pBlock->indexCount;

            GetResourceManager()->EditMaterials(arg);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
} // NOLINT (readability/fn_size)

void EditCommandExecutor::ExecuteEditAnimationCurve(CommandFlag command, AnimCurveBlock* pBlock) NN_NOEXCEPT
{
    EditAnimCurveArg arg;
    arg.animationKey = pBlock->info.animationKey;
    arg.animationKind = pBlock->info.animationKind;
    arg.curveData = nullptr;
    arg.curveDataSize = 0;
    arg.value = 0;

    switch(arg.animationKind)
    {
    case EDIT_TARGET_MODEL_ANIMATION_SHADER_PARAM_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_MATERIAL_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_TEX_PATTERN_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_BONE_VISIBILITY_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_MAT_VISIBILITY_CURVE:
    case EDIT_TARGET_MODEL_ANIMATION_SHAPE_CURVE:
    case EDIT_TARGET_SCENE_ANIMATION_CAMERA_CURVE:
    case EDIT_TARGET_SCENE_ANIMATION_LIGHT_CURVE:
    case EDIT_TARGET_SCENE_ANIMATION_FOG_CURVE:
        {
            arg.value = &pBlock->value;
            arg.curveData = pBlock->curveData;
            arg.curveDataSize = pBlock->value.curveSize;

            // ここでリロケートしておく
            nn::g3d::ResAnimCurveData* resAnimCurve = static_cast<nn::g3d::ResAnimCurveData*>(arg.curveData);
            RelocatePacketBinPtr(&resAnimCurve->pFrameArray);
            RelocatePacketBinPtr(&resAnimCurve->pKeyArray);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    switch(command)
    {
    case MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        GetResourceManager()->EditAnimCurve(arg);
        break;
    case SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        GetResourceManager()->EditSceneAnimCurve(arg);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteBindAnimation(CommandFlag command, BindAnimInfoBlock* pBlock) NN_NOEXCEPT
{
    BindAnimArg arg;
    arg.modelKey = pBlock->info.modelKey;
    arg.animationKeys = pBlock->animationKeys;
    arg.animationKeySize = pBlock->info.animationKeySize;
    switch(command)
    {
    case MODEL_ANIMATION_BIND_COMMAND_FLAG:
        {
            GetResourceManager()->BindAnimations(arg);

            NN_G3D_VIEWER_DEBUG_PRINT("---\n");
            NN_G3D_VIEWER_DEBUG_PRINT("Listing Bound Animations\n");
            for (int keyIndex = 0, end = arg.animationKeySize; keyIndex < end; ++keyIndex)
            {
                ViewerKeyType resFileKey = arg.animationKeys[keyIndex];
                NN_G3D_VIEWER_DEBUG_PRINT("key: %d bound at model %d\n", resFileKey, arg.modelKey);
                GetResourceManager()->PrintEditAnimObjs(resFileKey, arg.modelKey);
            }
        }
        break;
    case MODEL_ANIMATION_UNBIND_COMMAND_FLAG:
        {
            NN_G3D_VIEWER_DEBUG_PRINT("---\n");
            NN_G3D_VIEWER_DEBUG_PRINT("Listing Unbind Target Animations\n");
            for (int keyIndex = 0, end = arg.animationKeySize; keyIndex < end; ++keyIndex)
            {
                ViewerKeyType resFileKey = arg.animationKeys[keyIndex];
                NN_G3D_VIEWER_DEBUG_PRINT("key: %d bound at model %d\n", resFileKey, arg.modelKey);
                GetResourceManager()->PrintEditAnimObjs(resFileKey, arg.modelKey);
            }

            GetResourceManager()->UnbindAnimations(arg);
        }
        break;
    case SCENE_ANIMATION_BIND_COMMAND_FLAG:
        {
            GetResourceManager()->BindSceneAnimations(arg);
        }
        break;
    case SCENE_ANIMATION_UNBIND_COMMAND_FLAG:
        {
            GetResourceManager()->UnbindSceneAnimations(arg);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_G3D_VIEWER_DEBUG_PRINT("---\n");
    NN_G3D_VIEWER_DEBUG_PRINT("Listing All Animations\n");
    GetResourceManager()->PrintAllEditAnimObjs();

    NN_G3D_VIEWER_DEBUG_PRINT("---\n");
    NN_G3D_VIEWER_DEBUG_PRINT("Listing All Resource Files\n");
    GetResourceManager()->PrintAllResFiles();
    NN_G3D_VIEWER_DEBUG_PRINT("---\n");
}

void EditCommandExecutor::ExecutePickup(CommandFlag command, EditValueInfoBlock* pBlock) NN_NOEXCEPT
{
    EditValueBlock* pValueBlock = reinterpret_cast<EditValueBlock*>(pBlock + 1);
    TargetSelectedArg arg;
    arg.pModelObj = ViewerKeyManager::GetInstance().FindData<nn::g3d::ModelObj>(pBlock->key);

    switch (command)
    {
    case PICK_RUNTIME_MODEL_COMMAND_FLAG:
        {
            arg.targetKind = TargetSelectedArg::TargetKind_Model;
            arg.index = nullptr;
            arg.indexCount = 0;
            m_pCallbackCaller->Call(CallbackType_TargetSelected, &arg);

            GetResourceManager()->StartBlinking(arg);
        }
        break;

    case PICK_RUNTIME_MATERIAL_COMMAND_FLAG:
        {
            arg.targetKind = TargetSelectedArg::TargetKind_Material;
            arg.index = pValueBlock->index;
            arg.indexCount = pBlock->indexCount;
            m_pCallbackCaller->Call(CallbackType_TargetSelected, &arg);

            GetResourceManager()->StartBlinking(arg);
        }
        break;

    case PICK_RUNTIME_BONE_COMMAND_FLAG:
        {
            arg.targetKind = TargetSelectedArg::TargetKind_Bone;
            arg.index = pValueBlock->index;
            arg.indexCount = pBlock->indexCount;
            m_pCallbackCaller->Call(CallbackType_TargetSelected, &arg);
        }
        break;

    case PICK_RUNTIME_SHAPE_COMMAND_FLAG:
        {
            arg.targetKind = TargetSelectedArg::TargetKind_Shape;
            arg.index = pValueBlock->index;
            arg.indexCount = pBlock->indexCount;
            m_pCallbackCaller->Call(CallbackType_TargetSelected, &arg);

            GetResourceManager()->StartBlinking(arg);
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EditCommandExecutor::ExecuteUserScript(UserScriptBlock* pBlock) NN_NOEXCEPT
{
    NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(m_pAllocator, "Internal error: Allocator must be set before receive commands");

    ModelUserScriptExecutedArg arg;
    ScopedAllocator commandTextAllocator(*m_pAllocator, pBlock->scriptTextLength + 1, nn::g3d::detail::Alignment_Default, AllocateType_Other);
    ScopedAllocator targetNameAllocator(*m_pAllocator, pBlock->targetNameLength + 1, nn::g3d::detail::Alignment_Default, AllocateType_Other);

    ScopedAllocator boneIndicesAllocator(*m_pAllocator, pBlock->selectedBoneCount * sizeof(int), nn::g3d::detail::Alignment_Default, AllocateType_Other);
    ScopedAllocator shapeIndicesAllocator(*m_pAllocator, pBlock->selectedShapeCount * sizeof(int), nn::g3d::detail::Alignment_Default, AllocateType_Other);
    ScopedAllocator materialIndicesAllocator(*m_pAllocator, pBlock->selectedMaterialCount * sizeof(int), nn::g3d::detail::Alignment_Default, AllocateType_Other);

    int32_t* pInputSelectedBoneIndices = reinterpret_cast<int32_t*>(pBlock->dynamicData);
    int32_t* pInputSelectedShapeIndices = pInputSelectedBoneIndices + pBlock->selectedBoneCount;
    int32_t* pInputSelectedMaterialIndices = pInputSelectedShapeIndices + pBlock->selectedShapeCount;

    char* pInputScriptTextData = reinterpret_cast<char*>(pInputSelectedMaterialIndices + pBlock->selectedMaterialCount);
    char* pInputTargetNameData = pInputScriptTextData + pBlock->scriptTextLength + 1;

    {
        char* scriptText = reinterpret_cast<char*>(commandTextAllocator.GetAllocatedBuffer());
        memset(scriptText, 0, commandTextAllocator.GetAllocatedSize());
        memcpy(scriptText, pInputScriptTextData, pBlock->scriptTextLength);
        arg.scriptText = scriptText;
    }

    // モデル以外に適用するケースが出てきたらargにtargetNameのようなメンバを追加して、targetName、targetKindを渡す
    {
        char* targetName = reinterpret_cast<char*>(targetNameAllocator.GetAllocatedBuffer());
        memset(targetName, 0, targetNameAllocator.GetAllocatedSize());
        memcpy(targetName, pInputTargetNameData, pBlock->targetNameLength);
    }

    {
        EditModelObj* pTargetEditModelObj = GetResourceManager()->FindEditModelObj(pBlock->targetModelObjKey);
        NN_G3D_VIEWER_ASSERT_NOT_NULL_DETAIL(pTargetEditModelObj, "ModelObj at 0x%08x is not attached\n", pBlock->targetModelObjKey);

        arg.pTargetModelObj = ViewerKeyManager::GetInstance().FindData<nn::g3d::ModelObj>(pBlock->targetModelObjKey);
    }

    {
        int selectedBoneCount = pBlock->selectedBoneCount;
        int* pSelectedBoneIndices = reinterpret_cast<int*>(boneIndicesAllocator.GetAllocatedBuffer());
        for (int index = 0; index < selectedBoneCount; ++index, ++pInputSelectedBoneIndices)
        {
            pSelectedBoneIndices[index] = *pInputSelectedBoneIndices;
        }

        arg.selectedBoneCount = selectedBoneCount;
        arg.pSelectedBoneIndices = pSelectedBoneIndices;
    }

    {
        int selectedShapeCount = pBlock->selectedShapeCount;
        int* pSelectedShapeIndices = reinterpret_cast<int*>(shapeIndicesAllocator.GetAllocatedBuffer());
        for (int index = 0; index < selectedShapeCount; ++index, ++pInputSelectedShapeIndices)
        {
            pSelectedShapeIndices[index] = *pInputSelectedShapeIndices;
        }

        arg.selectedShapeCount = selectedShapeCount;
        arg.pSelectedShapeIndices = pSelectedShapeIndices;
    }

    {
        int selectedMaterialCount = pBlock->selectedMaterialCount;
        int* pSelectedMaterialIndices = reinterpret_cast<int*>(materialIndicesAllocator.GetAllocatedBuffer());
        for (int index = 0; index < selectedMaterialCount; ++index, ++pInputSelectedMaterialIndices)
        {
            pSelectedMaterialIndices[index] = *pInputSelectedMaterialIndices;
        }

        arg.selectedMaterialCount = selectedMaterialCount;
        arg.pSelectedMaterialIndices = pSelectedMaterialIndices;
    }

    m_pCallbackCaller->Call(CallbackType_ModelUserScriptExecuted, &arg);
}

void EditCommandExecutor::ExecuteSendRenderInfoDefinition(RenderInfoRecvBlock* pBlock, CommandMonitor* pCommandMonitor) NN_NOEXCEPT
{
    detail::EditModelObj* pEditModelObj = GetResourceManager()->FindEditModelObj(pBlock->modelKey);
    if (pEditModelObj == nullptr)
    {
        // TODO: 本来はアサートにすべきだが、3DEditor と接続を再開したタイミングで前回接続していた際の
        //       コマンドが残っていて、実行されてしまうケースがあったので、暫定的にスキップする
        return;
    }

    SendRenderInfoDefinitionCommand* pCommand = m_pEditCommandManager->QueueSendRenderInfoDefinitionCommand(pCommandMonitor);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(pCommand);// 今は止める。

    SendRenderInfoRequestedArg arg;
    arg.pModelObj = ViewerKeyManager::GetInstance().FindData<nn::g3d::ModelObj>(pBlock->modelKey);
    NN_G3D_VIEWER_ASSERT_NOT_NULL(arg.pModelObj);
    int32_t* materialIndices = static_cast<int32_t*>(pBlock->ofsMaterialIndexArray.Get());

    int materialIndexCount = pBlock->numMaterialIndex;
    for (int materialIndex = 0; materialIndex < materialIndexCount; ++materialIndex)
    {
        EditRenderInfo* pEditRenderInfo = pCommand->NewEditRenderInfo(static_cast<ViewerKeyType>(pBlock->modelKey), static_cast<int>(materialIndices[materialIndex]));
        m_pCurrentEditTargetRenderInfoDefinition = pEditRenderInfo;

        // コールバックの中でユーザに 3DEditor に送信する描画情報定義のパラメータをセットしてもらう
        {
            arg.materialIndex = materialIndices[materialIndex];
            m_pCallbackCaller->Call(CallbackType_SendRenderInfoRequested, &arg);
        }

        // セットされたパラメータを元に 3DEditor に送信するパケットを作成
        bool success = pEditRenderInfo->MakeRenderInfoPacket();
        if (!success)
        {
            pCommand->DeleteEditRenderInfo(pEditRenderInfo);
        }

        m_pCurrentEditTargetRenderInfoDefinition = nullptr;
    }

    pCommand->SetExecutable(true);
}
