﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <algorithm>
#include <nn/nn_SdkAssert.h>
#include <nn/mii.h>
#include <nn/util/util_BitArray.h>
#include <nn/util/util_MathTypes.h>
#include <nn/g3d/detail/g3d_Flag.h>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/mii/g3d_MiiConverter.h>

namespace nn { namespace g3d { namespace mii {
namespace {
void BoundingVolumeToAabb( nn::g3d::Bounding* pBounding, const nn::mii::BoundingBox& boundingVolume )
{
    NN_SDK_ASSERT_NOT_NULL( pBounding );
    nn::util::Vector3fType bbMax = NN_UTIL_VECTOR_3F_INITIALIZER( boundingVolume.max.x, boundingVolume.max.y, boundingVolume.max.z );
    nn::util::Vector3fType bbMin = NN_UTIL_VECTOR_3F_INITIALIZER( boundingVolume.min.x, boundingVolume.min.y, boundingVolume.min.z );
    nn::util::Vector3fType center;
    nn::util::Vector3fType extent;
    nn::util::VectorAdd( &center, bbMax, bbMin );
    nn::util::VectorSubtract( &extent, bbMax, bbMin );
    pBounding->center.x = nn::util::VectorGetX( center ) / 2.f;
    pBounding->center.y = nn::util::VectorGetY( center ) / 2.f;
    pBounding->center.z = nn::util::VectorGetZ( center ) / 2.f;
    pBounding->extent.x = nn::util::VectorGetX( extent ) / 2.f;
    pBounding->extent.y = nn::util::VectorGetY( extent ) / 2.f;
    pBounding->extent.z = nn::util::VectorGetZ( extent ) / 2.f;
}

bool IsMaterialVisible( uint32_t flag )
{
    return ( flag & nn::g3d::ResMaterial::Flag_Visibility ) == nn::g3d::ResMaterial::Flag_Visibility;
}

// shapeIndexEnd までに shapeIndexEnd と同じマテリアル参照が無いかを調べる。
bool HasCommonMaterial( nn::g3d::ResModel* pResModel, int shapeIndexEnd )
{
    for( int idxShape = 0; idxShape < shapeIndexEnd; ++idxShape )
    {
        int matIdx = pResModel->GetShape( idxShape )->GetMaterialIndex();
        if( matIdx == pResModel->GetShape( shapeIndexEnd )->GetMaterialIndex() )
        {
            return true;
        }
    }
    return false;
}

}


void AssignCharModel( nn::g3d::ResModel* pResModel, const nn::mii::CharModel* pCharModel, MaterialCallbackType pMaterialCallback,
                        void* pMaterialCallbackParam, DrawParamCallbackType pDrawParamCallback, void* pDrawParamCallbackParam ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pResModel );
    NN_SDK_REQUIRES_NOT_NULL( pCharModel );
    NN_SDK_REQUIRES( pCharModel->IsInitialized(), "pCharModel isn't initialized.\n" );
    NN_SDK_REQUIRES_NOT_NULL( pMaterialCallback );
    NN_SDK_REQUIRES_NOT_NULL( pDrawParamCallback );

    auto pModel = pResModel;
    const int shapeCount = pModel->GetShapeCount();
    for( int idxShape = 0; idxShape < shapeCount; ++idxShape )
    {
        auto pMaterial = pModel->GetMaterial( pModel->GetShape( idxShape )->GetMaterialIndex() );
        DrawParamCallbackOutput drawParamCallbackOutput = {};
        pDrawParamCallback( &drawParamCallbackOutput, pResModel, idxShape, pCharModel, pDrawParamCallbackParam );

        // Mii のパーツでない時は何もしない。
        if( !drawParamCallbackOutput.isMiiFace )
        {
            continue;
        }

        // Mii のパーツが無い時はボーンのビジビリティを落とす。
        auto pDrawParam = drawParamCallbackOutput.pDrawParam;
        if( pDrawParam == nullptr )
        {
            // Bone
            auto pBone = pModel->GetSkeleton()->GetBone( pModel->GetShape( idxShape )->GetBoneIndex() );
            auto& boneData = pBone->ToData();
            boneData.flag &= ~nn::g3d::ResBone::Flag_Visibility;

            // Material Visibility
            if( IsMaterialVisible( pMaterial->ToData().flag ) && ( !HasCommonMaterial( pModel, idxShape ) ) )
            {
                auto& materialData = pMaterial->ToData();
                nn::g3d::SetBit( &materialData.flag, 0, 0 );
            }
            continue;
        }

        // Material
        auto& materialData = pMaterial->ToData();
        nn::g3d::SetBit( &materialData.flag, 0, nn::g3d::ResMaterial::Flag_Visibility );

        pMaterialCallback( pMaterial, pDrawParam, pMaterialCallbackParam );

        // Shape
        auto pShape = pModel->GetShape( idxShape );
        auto& shapeData = pShape->ToData();
        const int shapeBoundingIdx = pShape->GetSubMeshCount();
        NN_SDK_ASSERT( drawParamCallbackOutput.pBoundingBox != nullptr, "nn::mii::DrawParamCallbackOutput::pBoundingBox should not be nullptr. Check if your nn::g3d::mii::DrawParamCallback sets valid ptr.\n" );
        auto pSubMeshBoundingArray = ( shapeData.pSubMeshBoundingArray ).Get();
        auto pShapeBounding = &pSubMeshBoundingArray[shapeBoundingIdx];
        BoundingVolumeToAabb( pShapeBounding, *drawParamCallbackOutput.pBoundingBox );
        for( int idxSubMesh = 0; idxSubMesh < shapeBoundingIdx; ++idxSubMesh )    // サブメッシュを持つ場合はシェイプのバウンディングで上書きする。
        {
            memcpy( &pSubMeshBoundingArray[idxSubMesh], pShapeBounding, sizeof( nn::g3d::Bounding ) );
        }

        // バウンディング球の計算
        float radius = 0.f;
        radius = std::sqrt( pShapeBounding->extent.x * pShapeBounding->extent.x +
                            pShapeBounding->extent.y * pShapeBounding->extent.y +
                            pShapeBounding->extent.z * pShapeBounding->extent.z );
        for( int idxMesh = 0; idxMesh < pShape->GetMeshCount(); ++idxMesh )
        {
            shapeData.pRadiusArray.Get()[idxMesh] = radius;
        }

        // Mesh
        NN_SDK_ASSERT( pShape->GetMeshCount() == 1, "Mii model cannot have vertex lod\n" );
        const int idxMesh = 0;    // mii モデルは頂点 LOD は現状持たないので index は 0 とする。
        auto pMesh = pShape->GetMesh( idxMesh );
        auto& mesh = pMesh->ToData();
        mesh.pIndexBuffer.Set( const_cast<nn::gfx::Buffer*>( pDrawParam->GetIndexBuffer() ) );
        mesh.count = pDrawParam->GetIndexCount();
        mesh.primType = nn::mii::DrawParam::PrimitiveTopology;
        mesh.format = nn::mii::DrawParam::IndexFormat;
        auto pSubmeshArray = mesh.pSubMeshArray.Get();
        NN_SDK_ASSERT( mesh.subMeshCount == 1, "libnn_g3dmii doesn't accept bfres that contains multiple submesh under a mesh.\n" );
        pSubmeshArray[0].ToData().count = mesh.count;
        pSubmeshArray[0].ToData().offset = 0;

        // Vertex
        auto pVertex = pShape->GetVertex();
        auto& vertex = pVertex->ToData();

        // libnn_mii では頂点バッファのパッキングは行っていない。1 attribute 1 頂点バッファになっている。
        auto vtxBuffSize = pDrawParam->GetBufferSize( nn::mii::DrawParam::AttributeType_Position );
        vertex.count = static_cast<uint32_t>( vtxBuffSize / nn::mii::DrawParam::PositionStride );    // 頂点数は attribute によらず同じなので position で計算する。

        const char* miiAttribNameArray[] =
        {
            "_p0",    // idxMiiAttrib: 0 for mii position
            "_n0",    // idxMiiAttrib: 1 for mii normal
            "_u0",    // idxMiiAttrib: 2 for mii uv
            "_t0",    // idxMiiAttrib: 3 for mii tangent
            "_c0",    // idxMiiAttrib: 4 for mii param
        };
        const int miiAttribCount = sizeof( miiAttribNameArray ) / sizeof( char* );
        for( int idxMiiAttrib = 0; idxMiiAttrib < miiAttribCount; ++idxMiiAttrib )
        {
            const char* miiAttribName = miiAttribNameArray[idxMiiAttrib];
            int idxVtxAttrib = pVertex->FindVertexAttrIndex( miiAttribName );
            if( idxVtxAttrib == nn::util::ResDic::Npos )
            {
                continue;
            }

            auto miiAttribType = static_cast<nn::mii::DrawParam::AttributeType>( idxMiiAttrib );
            if( !pDrawParam->IsValidAttribute( miiAttribType ) )
            {
                continue;
            }
            int idxBuff = pVertex->GetVertexAttr( idxVtxAttrib )->GetBufferIndex();
            auto pVtxBufferStateInfo = &vertex.pVertexBufferStateInfoArray.Get()[idxBuff];  // アクセサが private なので直接取得する
            auto ppVtxBuffer = vertex.pVertexBufferPtrArray.Get();  // 未初期化の ResFile なのでアクセサ使用不可
            auto pVtxBufferInfo = pVertex->GetVertexBufferInfo( idxBuff );
            ppVtxBuffer[idxBuff] = const_cast<nn::gfx::Buffer*>( pDrawParam->GetAttribute( miiAttribType ) );
            pVtxBufferStateInfo->ToData()->stride = static_cast<uint32_t>( pDrawParam->GetVertexAttributeStride( miiAttribType ) );
            pVtxBufferInfo->ToData()->size = static_cast<uint32_t>( vertex.count * pVtxBufferStateInfo->GetStride() );

            auto& vtxAttrib = pVertex->GetVertexAttr( idxMiiAttrib )->ToData();
            vtxAttrib.format = pDrawParam->GetVertexAttributeFormat( miiAttribType );
            vtxAttrib.offset = static_cast<uint16_t>( pDrawParam->GetVertexAttributeOffset( miiAttribType ) );
        }
    }

} //NOLINT(impl/function_size)

}}}    // namespace nn::g3d::mii
