﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/g3d/g3d_ShapeAnimObj.h>
#include <algorithm>
#include <nn/g3d/detail/g3d_Perf.h>
#include <nn/g3d/g3d_ModelObj.h>

namespace nn { namespace g3d {

void ShapeAnimObj::InitializeArgument::CalculateMemorySize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValid() == true);

    int bindCount = std::max(GetMaxShapeCount(), GetMaxVertexShapeAnimCount());
    int curveCount = GetMaxCurveCount();
    int keyShapeAnimCount = GetMaxKeyShapeAnimCount();

    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_MemoryBlock[blockIndex].Initialize();
    }

    m_MemoryBlock[MemoryBlockIndex_ResultBuffer].SetSizeBy<float>(1, keyShapeAnimCount);
    m_MemoryBlock[MemoryBlockIndex_BindTable].SetSizeBy<Bit32>(1,  bindCount);
    m_MemoryBlock[MemoryBlockIndex_SubBindTable].SetSizeBy<int8_t>(Alignment_Default, keyShapeAnimCount);
    size_t size = IsContextEnabled() ? nn::util::align_up(sizeof(AnimFrameCache) * curveCount, Alignment_Default) : 0;
    m_MemoryBlock[MemoryBlockIndex_FrameCacheArray].SetSize(size);

    m_WorkMemory.Initialize();
    for (int blockIndex = 0; blockIndex < MemoryBlockIndex_End; ++blockIndex)
    {
        m_WorkMemory.Append(&m_MemoryBlock[blockIndex]);
    }
}

bool ShapeAnimObj::Initialize(const InitializeArgument& arg, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pBuffer != NULL);
    NN_SDK_REQUIRES(IsAligned(pBuffer, Alignment_Buffer));

    if (!arg.IsMemoryCalculated() || !arg.IsValid())
    {
        return false;
    }
    if (arg.GetWorkMemorySize() > bufferSize)
    {
        // バッファーが必要なサイズに満たない場合は失敗。
        return false;
    }

    int bindCount = std::max(arg.GetMaxShapeCount(), arg.GetMaxVertexShapeAnimCount());
    int curveCount = arg.GetMaxCurveCount();

    // メンバの初期化。
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetBindTable().Initialize(arg.GetBuffer<Bit32>(pBuffer, InitializeArgument::MemoryBlockIndex_BindTable), bindCount);
    GetContext().Initialize(arg.GetBuffer<AnimFrameCache>(pBuffer, InitializeArgument::MemoryBlockIndex_FrameCacheArray), curveCount);
    SetResultBuffer(arg.GetBuffer(pBuffer, InitializeArgument::MemoryBlockIndex_ResultBuffer));
    m_MaxVtxShpAnim = arg.GetMaxVertexShapeAnimCount();
    m_MaxSubBind = arg.GetMaxKeyShapeAnimCount();
    m_pSubBindIndexArray = arg.GetBuffer<int8_t>(pBuffer, InitializeArgument::MemoryBlockIndex_SubBindTable);

    return true;
}

void ShapeAnimObj::SetResource(const ResShapeAnim* pRes) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pRes) == true, NN_G3D_RES_GET_NAME(pRes, GetName()));

    m_pRes = pRes;
    m_pVertexShapeAnimArray = pRes->ToData().pVertexShapeAnimArray.Get();

    SetTargetUnbound();

    bool loop = (pRes->ToData().flag & AnimFlag_PlayPolicyLoop) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetBindTable().SetAnimCount(pRes->GetVertexShapeAnimCount());
    GetContext().SetCurveCount(pRes->GetCurveCount());
}

BindResult ShapeAnimObj::Bind(const ResModel* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel               != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(IsAcceptable(pModel) == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pModel->GetShapeCount());

    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
        const char* pName = pVertexShapeAnim->ToData().pName.Get()->GetData();
        int idxTarget = pModel->FindShapeIndex(pName);
        if (idxTarget >= 0)
        {
            // KeyShape のバインド
            const ResShape* pShape = pModel->GetShape(idxTarget);
            BindResult subResult = SubBind(pVertexShapeAnim, pShape);
            if (subResult.IsBound())
            {
                // 1つも関連付けられなかった場合はバインドしない。
                bindTable.Bind(idxAnim, idxTarget);
            }
            result |= subResult;
        }
        else
        {
            result.SetFailureBit();
        }
    }

    SetTargetBound();
    ShapeAnimObj::ClearResult();
    return result;
}

BindResult ShapeAnimObj::Bind(const ModelObj* pModel) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(pModel != NULL, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    return Bind(pModel->GetResource());
}

void ShapeAnimObj::BindFast(const ResModel* pTarget) NN_NOEXCEPT
{
    NN_G3D_REQUIRES(IsAcceptable(pTarget) == true,     NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_G3D_REQUIRES(pTarget == m_pRes->GetBindModel(), NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pTarget->GetShapeCount());
    bindTable.BindAll(m_pRes->ToData().pBindIndexArray.Get());

    // VtxAttrib のバインド
    for (int idxAnim = 0, animCount = bindTable.GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        int idxTarget = bindTable.GetTargetIndex(idxAnim);
        if (idxTarget < AnimBindTable::Flag_NotBound)
        {
            const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
            BindResult result = SubBindFast(pVertexShapeAnim);
            if (result.IsMissed())
            {
                // 1つも関連付けられなかった場合はバインド解除する。
                bindTable.Unbind(idxAnim, idxTarget);
            }
        }
    }

    SetTargetBound();
    ShapeAnimObj::ClearResult();
}

BindResult ShapeAnimObj::SubBind(
    const ResVertexShapeAnim* pVertexShapeAnim, const ResShape* pShape) NN_NOEXCEPT
{
    BindResult result;
    const ResKeyShapeAnimInfo* pInfo =
        pVertexShapeAnim->ToData().pKeyShapeAnimInfoArray.Get();
    int beginKeyShapeAnim = pVertexShapeAnim->ToData().beginKeyShapeAnim;

    for (int idxKeyShapeAnim = 0, keyShapeAnimCount = pVertexShapeAnim->GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
    {
        const char* pName = pInfo->pName.Get()->GetData();
        int idxTarget = pShape->FindKeyShapeIndex(pName);
        m_pSubBindIndexArray[beginKeyShapeAnim + idxKeyShapeAnim] = static_cast<int8_t>(idxTarget);
        if (idxTarget >= 0)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    return result;
}

BindResult ShapeAnimObj::SubBindFast(const ResVertexShapeAnim* pVertexShapeAnim) NN_NOEXCEPT
{
    BindResult result;
    const ResKeyShapeAnimInfo* pInfo =
        pVertexShapeAnim->ToData().pKeyShapeAnimInfoArray.Get();
    int beginKeyShapeAnim = pVertexShapeAnim->ToData().beginKeyShapeAnim;

    for (int idxKeyShapeAnim = 0, keyShapeAnimCount = pVertexShapeAnim->GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
    {
        m_pSubBindIndexArray[beginKeyShapeAnim + idxKeyShapeAnim] = pInfo->subbindIndex;
        if (pInfo->subbindIndex >= 0)
        {
            result.SetSuccessBit();
        }
        else
        {
            result.SetFailureBit();
        }
    }

    return result;
}

void ShapeAnimObj::ClearResult() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);

    // バインドされていなくても初期化してしまう。
    for (int idxAnim = 0, animCount = GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
        int beginKeyShapeAnim = pVertexShapeAnim->ToData().beginKeyShapeAnim;
        float* pResult = AddOffset<float>(GetResultBuffer(), sizeof(float) * beginKeyShapeAnim);
        pVertexShapeAnim->Initialize(pResult);
    }
}

void ShapeAnimObj::Calculate() NN_NOEXCEPT
{
    NN_G3D_PERF_LEVEL1("ShapeAnimObj::Calculate");
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_G3D_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (IsFrameChanged())
    {
        float frame = GetFrameCtrl().GetFrame();
        const AnimBindTable& bindTable = GetBindTable();
        AnimContext& context = GetContext();
        int animCount = bindTable.GetAnimCount();
        float* pResultArray = static_cast<float*>(GetResultBuffer());
        if (context.IsFrameCacheValid())
        {
            for (int idxAnim = 0; idxAnim < animCount; ++idxAnim)
            {
                const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
                int beginKeyShapeAnim = pVertexShapeAnim->ToData().beginKeyShapeAnim;
                int beginCurve = pVertexShapeAnim->ToData().beginCurve;

                if (!bindTable.IsCalculateEnabled(idxAnim))
                {
                    continue;
                }

                float* pResult = &pResultArray[beginKeyShapeAnim];
                pVertexShapeAnim->Evaluate(pResult, frame, &m_pSubBindIndexArray[beginKeyShapeAnim],
                    context.GetFrameCacheArray(beginCurve));
            }
        }
        else
        {
            for (int idxAnim = 0; idxAnim < animCount; ++idxAnim)
            {
                const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
                int beginKeyShapeAnim = pVertexShapeAnim->ToData().beginKeyShapeAnim;

                if (!bindTable.IsCalculateEnabled(idxAnim))
                {
                    continue;
                }

                float* pResult = &pResultArray[beginKeyShapeAnim];
                pVertexShapeAnim->Evaluate(pResult, frame, &m_pSubBindIndexArray[beginKeyShapeAnim]);
            }
        }

        UpdateLastFrame();
    }
}

void ShapeAnimObj::ApplyTo(ModelObj* pModelObj) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pRes);
    NN_SDK_REQUIRES(IsTargetBound() == true, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    NN_SDK_REQUIRES(pModelObj != NULL,       NN_G3D_RES_GET_NAME(m_pRes, GetName()));

    const AnimBindTable& bindTable = GetBindTable();
    for (int idxAnim = 0, animCount = GetAnimCount(); idxAnim < animCount; ++idxAnim)
    {
        if (!bindTable.IsApplyEnabled(idxAnim))
        {
            continue;
        }

        int idxShape = bindTable.GetTargetIndex(idxAnim);
        ShapeObj* pShape = pModelObj->GetShape(idxShape);
        ApplyTo(pShape, idxAnim);
    }
}

void ShapeAnimObj::ApplyTo(ShapeObj* pShapeObj, int animIndex) const NN_NOEXCEPT
{
    // カーブの評価結果
    const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(animIndex);
    int beginKeyShapeAnim = pVertexShapeAnim->ToData().beginKeyShapeAnim;
    const float* pResult = AddOffset<float>(GetResultBuffer(), sizeof(float) * beginKeyShapeAnim);

    // 書き込み先と KeyShape の情報
    const ResKeyShapeAnimInfo* pInfo =
        pVertexShapeAnim->ToData().pKeyShapeAnimInfoArray.Get();
    // ベースキーシェイプを除いたキーシェイプでループし、ベースシェイプのウェイトを計算する。
    float baseWeight = 1.0f;
    for (int idxKeyShapeAnim = 1, keyShapeAnimCount = pVertexShapeAnim->GetKeyShapeAnimCount();
        idxKeyShapeAnim < keyShapeAnimCount; ++idxKeyShapeAnim, ++pInfo)
    {
        int subBindIndex = m_pSubBindIndexArray[beginKeyShapeAnim + idxKeyShapeAnim];
        if (subBindIndex == -1)
        {
            continue;
        }

        float weight = pResult[idxKeyShapeAnim];
        baseWeight -= weight;
        pShapeObj->SetBlendWeight(subBindIndex, weight);
    }

    int baseIndex = m_pSubBindIndexArray[beginKeyShapeAnim];
    NN_G3D_ASSERT(baseIndex >= 0, NN_G3D_RES_GET_NAME(m_pRes, GetName()));
    pShapeObj->SetBlendWeight(baseIndex, baseWeight);
}

}} // namespace nn::g3d
