﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <nn/font/font_TextureCache.h>
#include <nn/font/detail/font_Log.h>

#include <nn/gfx.h>

#if defined(NN_FONT_USE_FONTLL2)
#include <nn/fontll2/fontll2_ScalableFontEngine.h>
#else
#include <nn/fontll/fontll_ScalableFontEngine.h>
#endif

#include <nn/os.h>

namespace {

bool IsPos2(uint32_t n)
{
    return (((n - 1) & n) == 0);
}

void SetTextureInfo(nn::gfx::Texture::InfoType* pInfo, int width, int height)
{
    nn::gfx::Texture::InfoType info;
    pInfo->SetDefault();
    pInfo->SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    pInfo->SetWidth(width);
    pInfo->SetHeight(height);
    pInfo->SetDepth(1);
    pInfo->SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    pInfo->SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);
    pInfo->SetTileMode(nn::gfx::TileMode_Linear);
    pInfo->SetMipCount(1);
    pInfo->SetArrayLength(1);
}

// fontll のエラーコードをメッセージに変換する
const char* ChangeErrorCodeToMessage(int errorCode)
{
    switch (errorCode)
    {
        case 0: return "SUCCESS";
        case 101: return "ERR_FILE_CREATE";
        case 102: return "ERR_FILE_OPEN";
        case 103: return "ERR_FILE_CLOSE";
        case 104: return "ERR_FILE_SEEK";
        case 105: return "ERR_FILE_TELL";
        case 106: return "ERR_FILE_READ";
        case 107: return "ERR_FILE_WRITE";
        case 201: return "ERR_MALLOC_FAIL";
        case 202: return "ERR_REALLOC_FAIL";
        case 204: return "ERR_FREE_BAD_PTR";
        case 206: return "ERR_RESIZE_FAIL";
        case 207: return "ERR_BLOCK_GRANULARITY";
        case 208: return "ERR_BLOCK_SIZE";
        case 209: return "ERR_BLOCK_BEFORE_FIRST";
        case 210: return "ERR_BLOCK_AFTER_LAST";
        case 211: return "ERR_BLOCK_DAMAGED";
        case 212: return "ERR_BLOCK_IS_FREE";
        case 213: return "ERR_BLOCK_IN_USE";
        case 214: return "ERR_FREE_LIST_DAMAGED";
        case 301: return "ERR_FONT_NOT_FOUND";
        case 302: return "ERR_NO_CURRENT_SFNT";
        case 303: return "ERR_NO_CURRENT_LFNT";
        case 304: return "ERR_BAD_LFNT";
        case 305: return "ERR_NO_CURRENT_FNTSET";
        case 306: return "ERR_DUPLICATED_FONT_NAME";
        case 310: return "ERR_FONT_BUFFER_TOO_SMALL";
        case 311: return "ERR_FONT_NAME_NOT_UNIQUE";
        case 312: return "ERR_FONT_NAME_NOT_FOUND";
        case 313: return "ERR_DELETE_FONT";
        case 314: return "ERR_FONT_NAME_IN_USE";
        case 315: return "ERR_BAD_TYPESET";
        case 316: return "ERR_BAD_FNTSET";
        case 317: return "ERR_BAD_SFNT";
        case 402: return "ERR_CMAP_UNSUPPORTED";
        case 403: return "ERR_NOT_A_TTF";
        case 404: return "ERR_BAD_TTC_INDEX";
        case 405: return "ERR_TABLE_NOT_FOUND";
        case 406: return "ERR_BAD_GLYF_INDEX";
        case 407: return "ERR_BAD_GLYF_FORMAT";
        case 412: return "OUT_OFF_SEQUENCE_CALL_ERR";
        case 416: return "UNDEFINED_INSTRUCTION_ERR";
        case 417: return "TRASHED_MEM_ERR";
        case 418: return "POINTS_DATA_ERR";
        case 419: return "CONTOUR_DATA_ERR";
        case 426: return "EBLC_VERSION";
        case 427: return "EBLC_NO_METRICS";
        case 428: return "EBLC_COMP_SIZE";
        case 429: return "ERR_INDEX_FORMAT";
        case 430: return "ERR_ACT3_DISK";
        case 431: return "ERR_ACT3_UNDEF";
        case 432: return "ERR_STIK_UNDEF";
        case 433: return "ERR_CCC_UNDEF";
        case 435: return "ERR_PHASED_UNDEF";
        case 436: return "ERR_PFR_UNDEF";
        case 437: return "ERR_BOLD_UNDEF";
        case 438: return "ERR_CONTOURCHECK_UNDEF";
        case 439: return "ERR_FS_EDGE_HINTS_UNDEF";
        case 440: return "ERR_FS_EDGE_RENDER_UNDEF";
        case 441: return "ERR_TABLE_UNSUPPORTED";
        case 442: return "ERR_NOT_A_LTT";
        case 443: return "ERR_BAD_LTT_NUM_FONTS";
        case 444: return "ERR_BAD_LTT_METRIC_FONT";
        case 445: return "ERR_BAD_LTT_INDEX";
        case 446: return "ERR_BAD_FONT_TYPE";
        case 447: return "ERR_INVALID_FILE_SIZE";
        case 448: return "ERR_BAD_TABLE_DIR";
        case 449: return "ERR_BAD_TABLE_CHECKSUM";
        case 450: return "ERR_LINKED_FONTS_UNDEF";
        case 451: return "ERR_BAD_LTT_COMPONENT";
        case 452: return "ERR_CFF_UNDEF";
        case 453: return "ERR_CFF_MAXP";
        case 456: return "ERR_CFF_BAD_OPCODE";
        case 457: return "ERR_CFF_CID";
        case 458: return "ERR_CFF_MULTIPLE_FONTS";
        case 459: return "ERR_CFF_CHARSTRINGTYPE";
        case 460: return "ERR_CFF_BAD_SUBRSPTR";
        case 461: return "ERR_CFF_BAD_OFFSET_SIZE";
        case 502: return "ERR_SCALE_LIMIT";
        case 503: return "ERR_SCALE_DEGENERATE";
        case 505: return "ERR_NO_MATCHING";
        case 508: return "ERR_BAD_REF_COUNT";
        case 512: return "ERR_POSIX_THREADS_NOT_SUPP";
        case 513: return "ERR_NO_ICONS";
        case 514: return "ERR_BAD_ICON_INDEX";
        case 516: return "ERR_NOT_VANILLA";
        case 517: return "ERR_VDMX_RATIO";
        case 518: return "ERR_yPelHeight_NOT_FOUND";
        case 519: return "ERR_NOT_SUPPORTED";
        case 520: return "ERR_INVALID_CMAP";
        case 521: return "ERR_INVALID_ADDRESS";
        case 522: return "ERR_xPelHeight_NOT_FOUND";
        case 523: return "ERR_BAD_EXTERNAL_HEAP";
        case 524: return "ERR_RASTER_RESOURCE_LIMIT";
        case 601: return "ERR_TINY_TYPE";
        case 602: return "ERR_BYTE_TYPE";
        case 603: return "ERR_SHORT_TYPE";
        case 604: return "ERR_USHORT_TYPE";
        case 605: return "ERR_LONG_TYPE";
        case 606: return "ERR_ULONG_TYPE";
        case 607: return "ERR_SIZEOF_LONG";
        case 608: return "ERR_BYTE_ORDER";
        case 609: return "ERR_INT_64";
        case 610: return "ERR_FONT_IN_USE";
        case 612: return "ERR_BAD_EFFECT";
        case 613: return "ERR_BAD_PERCENT";
        case 614: return "ERR_BAD_OUTLINE_WIDTH";
        case 615: return "ERR_BAD_OUTLINE_OPACITY";
        case 701: return "ERR_CCC_BAD_TABLE";
        case 702: return "ERR_FREE_BAD_COUNT";
        case 801: return "ERR_MUTEX_CREATE";
        case 802: return "ERR_MUTEX_GONE";
        case 803: return "ERR_MUTEX_TIMEOUT";
        case 804: return "ERR_MUTEX_RELEASE";
        case 821: return "ERR_SHMEM_CREATE";
        case 822: return "ERR_SHMEM_ATTACH";
        case 823: return "ERR_SHMEM_DETACH";
        case 824: return "ERR_SHMEM_DELETE";
        case 825: return "ERR_SHMEM_OPEN";
        case 831: return "ERR_FILE_UNMAP";
        case 832: return "ERR_CREATE_FILE_MAPPING";
        case 833: return "ERR_MAP_FILE_VIEW";
        case 1030: return "ERR_bad_token_code";
        default: return "Unknown error.";
    }
}// NOLINT(impl/function_size)

}

namespace nn
{
namespace font
{

namespace detail
{
#if defined(NN_FONT_USE_FONTLL2)
    typedef nn::fontll2::ScalableFontEngine ScalableFontEngine;
    typedef nn::fontll2::OtfKerningTable OtfKerningTable;
    typedef nn::fontll2::Metrics Metrics;
    typedef nn::fontll2::GlyphMap GlyphMap;
    const int FormatGrayMap8 = nn::fontll2::FormatGrayMap8;
    const int Success = nn::fontll2::Success;
    const int ErrorMemoryAllocationFail = nn::fontll2::ErrorMemoryAllocationFail;
#else
    typedef nn::fontll::ScalableFontEngine ScalableFontEngine;
    typedef nn::fontll::OtfKerningTable OtfKerningTable;
    typedef nn::fontll::Metrics Metrics;
    typedef nn::fontll::GlyphMap GlyphMap;
    const int FormatGrayMap8 = nn::fontll::FormatGrayMap8;
    const int Success = nn::fontll::Success;
    const int ErrorMemoryAllocationFail = nn::fontll::ErrorMemoryAllocationFail;
#endif
}

void TextureCache::InitializeArg::SetDefault()
{
    textureCacheWidth = 1024;
    textureCacheHeight = 1024;
    allocateFunction = NULL;
    pUserDataForAllocateFunction = NULL;
    for (int i = 0; i < FontFaceCountMax; i++)
    {
        for (int j = 0; j < MultiScalableFontCountMax; j++)
        {
            pFontDatas[i][j] = NULL;
            pFontDataSizes[i][j] = 0;
            pBoldWeights[i][j] = 0.0f;
            pBorderWidths[i][j] = 0;
            scaleWidths[i][j] = 1.0f;
            scaleHeights[i][j] = 1.0f;
            pFontDataTypes[i][j] = FontDataType_Ttf;
            ignorePalt[i][j] = false;
            deleteBearingX[i][j] = false;
            bearingOffsetX[i][j] = 0;
            forceMonospacedEnabled[i][j] = false;
            forceMonospacedWidth[i][j] = 0;
            baselineOffset[i][j] = 0;
            charCodeRangeCount[i][j] = 0;
            for (int k = 0; k < CharCodeRangeCountMax; k++)
            {
                charCodeRangeFirst[i][j][k] = 0;
                charCodeRangeLast[i][j][k] = 0;
            }
            fontIndex[i][j] = 0;
        }
        fontListCount[i] = 1;
    }
    fontCount = 1;
    workMemorySize = WorkMemorySizeDefault;
    noPlotWorkMemorySize = WorkMemorySizeNoPlotDefault;
    glyphNodeCountMax = GlyphNodeCountMaxDefault;
    isDrawMultiCore = false;
    coreCount = 1;
    pGetCoreNumberFunction = NULL;
    pCalculateLineKindFunction = NULL;
    pCalculateLineHeightFunction = NULL;
    isAutoHint = false;
    isExactWidthReturnedAfterPlot = true;
}

TextureCache::TextureCache()
: m_pFontEngine(NULL)
, m_pFontEngineNoPlot(NULL)
, m_pFontNameBuffers(NULL)
, m_pTextureBitmap(NULL)
, m_TextureCacheWidth(0)
, m_TextureCacheHeight(0)
, m_LineCurrentPos(0)
, m_FontCount(0)
, m_CurrentFontFace(0)
, m_NoSpaceError(NoSpaceError_NoError)
, m_IsWorkMemoryExhausted(false)
, m_IsDrawMultiCore(false)
, m_CoreCount(1)
, m_pGetCoreNumberFunction(NULL)
, m_pCalculateLineKindFunction(NULL)
, m_pCalculateLineHeightFunction(NULL)
, m_pFontMetrics(NULL)
, m_pBoldWeights(NULL)
, m_pBorderWidths(NULL)
, m_pDeleteBearingX(NULL)
, m_pBearingOffsetX(NULL)
, m_pForceMonospacedEnabled(NULL)
, m_pForceMonospacedWidth(NULL)
, m_pCharCodeRangeCount(NULL)
, m_pCharCodeRangeFirst(NULL)
, m_pCharCodeRangeLast(NULL)
, m_pOtfKerningTable(NULL)
, m_IsExactWidthReturnedAfterPlot(true)
, m_pActiveMemoryPoolForTexture(NULL)
, m_OffsetOfMemoryPoolForTexture(0)
#if NN_GFX_IS_TARGET_VK || NN_GFX_IS_TARGET_GL
, m_pDevice(NULL)
, m_pTemporalCommandMemory(NULL)
, m_pTemporalControlMemory(NULL)
#endif
{
    for (int i = 0; i < FontFaceCountMax; i++)
    {
        m_FontListCount[i] = 0;
        m_InnerFontFaceHeadTable[i] = NULL;
    }
    for (int i = 0; i < FontFaceCountMax * MultiScalableFontCountMax; i++)
    {
        m_InnerFontFaceTable[i] = 0;
    }
}

TextureCache::~TextureCache()
{
}

void TextureCache::SetMemoryPoolInfo(nn::gfx::MemoryPoolInfo* pInfo)
{
    pInfo->SetDefault();
    pInfo->SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);
}

size_t TextureCache::CalculateMemoryPoolAlignment(nn::gfx::Device* pDevice, int width, int height)
{
    nn::gfx::Texture::InfoType info;
    SetTextureInfo(&info, width, height);

    nn::gfx::MemoryPoolInfo memPoolInfo;
    SetMemoryPoolInfo(&memPoolInfo);

    return std::max(nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info), nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo));
}

size_t TextureCache::CalculateMemoryPoolSize(nn::gfx::Device* pDevice, int width, int height)
{
    nn::gfx::MemoryPoolInfo memPoolInfo;
    SetMemoryPoolInfo(&memPoolInfo);

    return nn::util::align_up(width * height * 1, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo));
}

void TextureCache::Initialize(nn::gfx::Device* pDevice, const InitializeArg& arg, nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize)
{
    // 引数のチェック
    if ( ! IsPos2(arg.textureCacheWidth))
    {
        NN_SDK_ASSERT(false, "arg.width[%u] is not power of 2.", arg.textureCacheWidth);
        return;
    }
    if (arg.allocateFunction == NULL)
    {
        NN_SDK_ASSERT(false, "arg.allocateFunction must be set.");
        return;
    }
    if (arg.fontCount == 0 || arg.fontCount > FontFaceCountMax)
    {
        NN_SDK_ASSERT(false, "arg.fontCount must be from 1 to %d.", FontFaceCountMax);
        return;
    }

    int fontFaceCount = InitializeFontFaceTable(arg);
    m_FontCount = arg.fontCount;
    m_IsDrawMultiCore = arg.isDrawMultiCore;
    m_CoreCount = arg.coreCount;
    NN_SDK_ASSERT(0 < m_CoreCount && m_CoreCount <= CoreCountMax, "nn::font::TextureCache::InitializeArg::coreCount(%d) must not be over %d. Please set the minimum number of cores to use in the actual machine.", m_CoreCount, CoreCountMax);
    m_pGetCoreNumberFunction = arg.pGetCoreNumberFunction == NULL ? nn::os::GetCurrentCoreNumber : arg.pGetCoreNumberFunction;
    m_pCalculateLineKindFunction = arg.pCalculateLineKindFunction == NULL ? GlyphNode::CalculateLineKind : arg.pCalculateLineKindFunction;
    m_pCalculateLineHeightFunction = arg.pCalculateLineHeightFunction == NULL ? GlyphNode::CalculateLineHeight : arg.pCalculateLineHeightFunction;
    m_pFontNameBuffers = static_cast<char*>(arg.allocateFunction(fontFaceCount * FontNameLengthMax, 4, arg.pUserDataForAllocateFunction));
    m_pFontMetrics = static_cast<FontMetrics*>(arg.allocateFunction(fontFaceCount * sizeof(FontMetrics), 4, arg.pUserDataForAllocateFunction));
    m_pBoldWeights = static_cast<int32_t*>(arg.allocateFunction(fontFaceCount * sizeof(int32_t), 4, arg.pUserDataForAllocateFunction));
    m_pBorderWidths = static_cast<uint8_t*>(arg.allocateFunction(fontFaceCount * sizeof(uint8_t), 4, arg.pUserDataForAllocateFunction));
    m_pDeleteBearingX = static_cast<bool*>(arg.allocateFunction(fontFaceCount * sizeof(bool), 4, arg.pUserDataForAllocateFunction));
    m_pBearingOffsetX = static_cast<int16_t*>(arg.allocateFunction(fontFaceCount * sizeof(int16_t), 4, arg.pUserDataForAllocateFunction));
    m_pForceMonospacedEnabled = static_cast<bool*>(arg.allocateFunction(fontFaceCount * sizeof(bool), 4, arg.pUserDataForAllocateFunction));
    m_pForceMonospacedWidth = static_cast<int16_t*>(arg.allocateFunction(fontFaceCount * sizeof(int16_t), 4, arg.pUserDataForAllocateFunction));
    m_pCharCodeRangeCount = static_cast<int*>(arg.allocateFunction(fontFaceCount * sizeof(int), 4, arg.pUserDataForAllocateFunction));
    m_pCharCodeRangeFirst = static_cast<uint32_t(*)[CharCodeRangeCountMax]>(arg.allocateFunction(fontFaceCount * sizeof(uint32_t[CharCodeRangeCountMax]), 4, arg.pUserDataForAllocateFunction));
    m_pCharCodeRangeLast = static_cast<uint32_t(*)[CharCodeRangeCountMax]>(arg.allocateFunction(fontFaceCount * sizeof(uint32_t[CharCodeRangeCountMax]), 4, arg.pUserDataForAllocateFunction));
    m_pOtfKerningTable = static_cast<detail::OtfKerningTable**>(arg.allocateFunction(fontFaceCount * sizeof(detail::OtfKerningTable*), 4, arg.pUserDataForAllocateFunction));

    m_TextureCacheWidth = arg.textureCacheWidth;
    m_TextureCacheHeight = arg.textureCacheHeight;
    m_LineCurrentPos = 0;
    m_CurrentFontFace = UINT_MAX;
    m_NoSpaceError = NoSpaceError_NoError;
    m_IsWorkMemoryExhausted = false;
    for (int k = 0; k < CoreCountMax; k++)
    {
        m_CurrentFontFacesNoPlot[k] = UINT_MAX;
    }
    m_IsExactWidthReturnedAfterPlot = arg.isExactWidthReturnedAfterPlot;

    // ステート構造体の確保
    {
        m_pFontEngine = static_cast<detail::ScalableFontEngine*>(arg.allocateFunction(sizeof(detail::ScalableFontEngine), 4, arg.pUserDataForAllocateFunction));
        new(m_pFontEngine)detail::ScalableFontEngine();

        m_pFontEngineNoPlot = static_cast<detail::ScalableFontEngine*>(arg.allocateFunction(sizeof(detail::ScalableFontEngine) * GetFontEngineNoPlotCount(), 4, arg.pUserDataForAllocateFunction));
        for (uint32_t k = 0; k < GetFontEngineNoPlotCount(); k++)
        {
            new(&m_pFontEngineNoPlot[k])detail::ScalableFontEngine();
        }
    }

    // ステート構造体の初期化
    m_pFontEngine->Initialize(arg.allocateFunction(arg.workMemorySize, 4, arg.pUserDataForAllocateFunction), static_cast<uint32_t>(arg.workMemorySize));
    AssertFsError("Initialize");
    // このステート構造体ではプロットは行わないので、ワークエリアは固定の小さい領域にする。フォントを複数扱う場合は消費されるサイズはその数の倍数必要。
    for (uint32_t k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        m_pFontEngineNoPlot[k].Initialize(arg.allocateFunction(arg.noPlotWorkMemorySize * fontFaceCount, 4, arg.pUserDataForAllocateFunction), static_cast<uint32_t>(arg.noPlotWorkMemorySize) * fontFaceCount);
        AssertFsErrorNoPlot("Initialize", k);
    }

    // フォントの登録
    {
        int idx = 0;
        for (uint32_t i = 0; i < m_FontCount; i++)
        {
            for (uint32_t j = 0; j < static_cast<uint32_t>(m_FontListCount[i]); j++)
            {
                const void* pFontData;
                if (arg.pFontDataTypes[i][j] == FontDataType_Bfttf || arg.pFontDataTypes[i][j] == FontDataType_Bfotf)
                {
#if defined(NN_FONT_USE_FONTLL2)
                    pFontData = nn::fontll2::ScalableFontEngineHelper::Decode(arg.pFontDatas[i][j], static_cast<uint32_t>(arg.pFontDataSizes[i][j]));
#else
                    pFontData = nn::fontll::ScalableFontEngineHelper::Decode(arg.pFontDatas[i][j], static_cast<uint32_t>(arg.pFontDataSizes[i][j]));
#endif
                    NN_SDK_ASSERT(pFontData != NULL, "Font decoding failed. [FontFaceIndex=%d, MultiScalableFontIndex=%d] The font format may be wrong, or the font size may be set to 0.", i, j);
                }
                else
                {
                    pFontData = arg.pFontDatas[i][j];
                }
                m_pBoldWeights[idx] = static_cast<int32_t>(static_cast<float>(1 << 16) * arg.pBoldWeights[i][j]);
                m_pBorderWidths[idx] = arg.pBorderWidths[i][j];
                m_pDeleteBearingX[idx] = arg.deleteBearingX[i][j];
                m_pBearingOffsetX[idx] = arg.bearingOffsetX[i][j];
                m_pForceMonospacedEnabled[idx] = arg.forceMonospacedEnabled[i][j];
                m_pForceMonospacedWidth[idx] = arg.forceMonospacedWidth[i][j];

                NN_SDK_ASSERT(arg.charCodeRangeCount[i][j] <= CharCodeRangeCountMax);
                m_pCharCodeRangeCount[idx] = arg.charCodeRangeCount[i][j];
                for (int k = 0; k < CharCodeRangeCountMax; k++)
                {
                    m_pCharCodeRangeFirst[idx][k] = arg.charCodeRangeFirst[i][j][k];
                    m_pCharCodeRangeLast[idx][k] = arg.charCodeRangeLast[i][j][k];
                }

                m_pFontEngine->LoadFont(GetFontNameBuf(idx), pFontData, arg.fontIndex[i][j], FontNameLengthMax);
                AssertFsError("LoadFont");
                for (uint32_t k = 0; k < GetFontEngineNoPlotCount(); k++)
                {
                    m_pFontEngineNoPlot[k].LoadFont(GetFontNameBuf(idx), pFontData, arg.fontIndex[i][j], FontNameLengthMax);
                    AssertFsErrorNoPlot("LoadFont", k);
                }

                // フォントのアセントと高さを求めておく
                SetFontFace(idx);
                detail::Metrics metrics;
                m_pFontEngine->GetFontMetrics(&metrics);
                AssertFsError("GetFontMetrics");

                m_pFontMetrics[idx].boundingBoxAscentRatio = static_cast<float>(metrics.fontBoundingBox.yMax) / metrics.metricsResolution;
                m_pFontMetrics[idx].boundingBoxHeightRatio = static_cast<float>(metrics.fontBoundingBox.yMax - metrics.fontBoundingBox.yMin) / metrics.metricsResolution;
                m_pFontMetrics[idx].ascent_ratio = static_cast<float>(metrics.os2WinAscent) / metrics.metricsResolution;
                m_pFontMetrics[idx].height_ratio = static_cast<float>(metrics.os2WinAscent + metrics.os2WinDescent) / metrics.metricsResolution;
                m_pFontMetrics[idx].scaleWidth = arg.scaleWidths[i][j];
                m_pFontMetrics[idx].scaleHeight = arg.scaleHeights[i][j];
                m_pFontMetrics[idx].baselineOffset = arg.baselineOffset[i][j];

                if (CheckNecessaryToReadOtfKerningTable(arg, i, j))
                {
                    m_pOtfKerningTable[idx] = m_pFontEngine->InitializeOtfKerningTable(arg.allocateFunction, arg.pUserDataForAllocateFunction, arg.ignorePalt[i][j]);
                }
                else
                {
                    m_pOtfKerningTable[idx] = NULL;
                }

                idx++;
            }
        }
    }

    // 高さが最大のものにベースラインを合わせる
    {
        int idx = 0;
        for (uint32_t i = 0; i < m_FontCount; i++)
        {
            float maxBoundingBoxAscentRatio = 0.0f;
            float maxBoundingBoxHeightRatio = 0.0f;
            float maxAscentRatio = 0.0f;
            float maxHeightRatio = 0.0f;
            for (uint32_t j = 0; j < static_cast<uint32_t>(m_FontListCount[i]); j++)
            {
                if (maxBoundingBoxHeightRatio < m_pFontMetrics[idx + j].boundingBoxHeightRatio)
                {
                    maxBoundingBoxAscentRatio = m_pFontMetrics[idx + j].boundingBoxAscentRatio;
                    maxBoundingBoxHeightRatio = m_pFontMetrics[idx + j].boundingBoxHeightRatio;
                    maxAscentRatio = m_pFontMetrics[idx + j].ascent_ratio;
                    maxHeightRatio = m_pFontMetrics[idx + j].height_ratio;
                }
            }
            for (uint32_t j = 0; j < static_cast<uint32_t>(m_FontListCount[i]); j++)
            {
                m_pFontMetrics[idx + j].boundingBoxAscentRatio = maxBoundingBoxAscentRatio;
                m_pFontMetrics[idx + j].boundingBoxHeightRatio = maxBoundingBoxHeightRatio;
                m_pFontMetrics[idx + j].ascent_ratio = maxAscentRatio;
                m_pFontMetrics[idx + j].height_ratio = maxHeightRatio;
            }
            idx += m_FontListCount[i];
        }
    }

    const size_t TextureSize = m_TextureCacheWidth * m_TextureCacheHeight * 1;

    m_TexObj.Set(m_pTextureBitmap, font::FontSheetFormat_A8, static_cast<uint16_t>(m_TextureCacheWidth), static_cast<uint16_t>(m_TextureCacheHeight), 1, true);
    {
        nn::gfx::Texture::InfoType info;
        SetTextureInfo(&info, m_TextureCacheWidth, m_TextureCacheHeight);

        {
            const size_t Alignment = CalculateMemoryPoolAlignment(pDevice, m_TextureCacheWidth, m_TextureCacheHeight);
            const size_t MemPoolSize = CalculateMemoryPoolSize(pDevice, m_TextureCacheWidth, m_TextureCacheHeight);

            if (pMemoryPool == NULL)
            {
                m_pTextureBitmap = static_cast<uint8_t*>(arg.allocateFunction(MemPoolSize, Alignment, arg.pUserDataForAllocateFunction));
                std::memset(m_pTextureBitmap, 0x00, MemPoolSize);

                nn::gfx::MemoryPoolInfo memPoolInfo;
                SetMemoryPoolInfo(&memPoolInfo);

                memPoolInfo.SetPoolMemory(m_pTextureBitmap, MemPoolSize);
                m_MemoryPoolForTexture.Initialize(pDevice, memPoolInfo);

                m_Texture.Initialize(pDevice, info, &m_MemoryPoolForTexture, 0, TextureSize);

                m_pActiveMemoryPoolForTexture = &m_MemoryPoolForTexture;
                m_OffsetOfMemoryPoolForTexture = 0;
            }
            else {

                auto pBuffer = static_cast<uint8_t*>(pMemoryPool->Map());
                m_pTextureBitmap = pBuffer + memoryPoolOffset;
                std::memset(m_pTextureBitmap, 0x00, MemPoolSize);
                pMemoryPool->Unmap();

                m_Texture.Initialize(pDevice, info, pMemoryPool, memoryPoolOffset, memoryPoolSize);

                m_pActiveMemoryPoolForTexture = pMemoryPool;
                m_OffsetOfMemoryPoolForTexture = memoryPoolOffset;
            }
        }
    }
    // TextureView を初期化
    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetChannelMapping(nn::gfx::ChannelMapping_One, nn::gfx::ChannelMapping_One, nn::gfx::ChannelMapping_One, nn::gfx::ChannelMapping_Red);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_Unorm);

        // シェーダーが 2DArray としてアクセスするので 2dArray として参照する(そうしないと正しく表示されない)。
#if NN_GFX_IS_TARGET_GL || NN_GFX_IS_TARGET_VK
        info.SetImageDimension(nn::gfx::ImageDimension_2dArray);
#elif NN_GFX_IS_TARGET_NVN
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
#else
        #error Un konown API Type.
#endif

        info.SetTexturePtr(&m_Texture);
        m_TextureView.Initialize(pDevice, info);
    }

    m_GlyphTreeMap.Initialize(arg.allocateFunction, arg.pUserDataForAllocateFunction, arg.glyphNodeCountMax);
    if (arg.isAutoHint)
    {
        m_pFontEngine->SetAutoHint(true);
    }

#if NN_GFX_IS_TARGET_VK || NN_GFX_IS_TARGET_GL
    m_pDevice = pDevice;

    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( m_pDevice, info );
        m_pTemporalCommandMemory = arg.allocateFunction( TemporalCommandMemorySize, alignment, arg.pUserDataForAllocateFunction );
        info.SetPoolMemory( m_pTemporalCommandMemory, TemporalCommandMemorySize );
        m_TemporalMemoryPool.Initialize( m_pDevice, info );

        m_pTemporalControlMemory = arg.allocateFunction( TemporalControlMemorySize, 8, arg.pUserDataForAllocateFunction );
    }

    {
        nn::gfx::CommandBuffer::InfoType info;
        info.SetDefault();
        info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
        info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
        m_TemporalCommandBuffer.Initialize( m_pDevice, info );
    }

    {
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize( TextureSize );
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_Write | nn::gfx::GpuAccess_Read );
        m_TemporalBuffer.Initialize( m_pDevice, info, NULL, 0, 0 );
    }

    {
        nn::gfx::Queue::InfoType info;
        info.SetDefault();
        info.SetCapability( nn::gfx::QueueCapability_Graphics );
        m_TemporalQueue.Initialize( m_pDevice, info );
    }

#endif

}// NOLINT(impl/function_size)

void TextureCache::Finalize(nn::gfx::Device* pDevice, nn::FreeFunctionWithUserData freeFunction, void* pUserDataForFreeFunction)
{
    void* pWorkBuffer = reinterpret_cast<void*>(m_pFontEngine->GetPointerToWorkBuffer());
    void* pNoPlotWorkBuffers[CoreCountMax] = { NULL };
    for (uint32_t k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        pNoPlotWorkBuffers[k] = reinterpret_cast<void*>(m_pFontEngineNoPlot[k].GetPointerToWorkBuffer());
    }

    {
        int idx = 0;
        for (uint32_t i = 0; i < m_FontCount; i++)
        {
            for (uint32_t j = 0; j < static_cast<uint32_t>(m_FontListCount[i]); j++)
            {
                if (m_pOtfKerningTable[idx] != NULL)
                {
                    m_pFontEngine->FinalizeOtfKerningTable(m_pOtfKerningTable[idx], freeFunction, pUserDataForFreeFunction);
                    m_pOtfKerningTable[idx] = NULL;
                }
                idx++;
            }
        }
    }

    m_pFontEngine->Finalize();
    AssertFsError("Finalize");
    for (uint32_t k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        m_pFontEngineNoPlot[k].Finalize();
        AssertFsErrorNoPlot("Finalize", k);
    }

    freeFunction( m_pFontEngine, pUserDataForFreeFunction );
    m_pFontEngine = NULL;
    freeFunction( m_pFontEngineNoPlot, pUserDataForFreeFunction );
    m_pFontEngineNoPlot = NULL;
    freeFunction( pWorkBuffer, pUserDataForFreeFunction );
    for (uint32_t k = 0; k < GetFontEngineNoPlotCount(); k++)
    {
        freeFunction( pNoPlotWorkBuffers[k], pUserDataForFreeFunction );
    }

    // m_GlyphTreeMap 内のノードを保持するリストは m_GlyphTreeMap.Finalize を呼ぶ前にクリアしなければならない
    for (uint32_t i = 0; i < TextureCacheLineCountMax; i++)
    {
        m_LineInfos[i].list.clear();
    }
    m_NeedPlotGlyphList.clear();
    m_NeedEraseGlyphList.clear();
    m_NotInFontGlyphList.clear();

    m_GlyphTreeMap.Finalize(freeFunction, pUserDataForFreeFunction);
    freeFunction( m_pFontNameBuffers, pUserDataForFreeFunction );
    m_pFontNameBuffers = NULL;
    freeFunction( m_pFontMetrics, pUserDataForFreeFunction );
    m_pFontMetrics = NULL;
    freeFunction( m_pBoldWeights, pUserDataForFreeFunction );
    m_pBoldWeights = NULL;
    freeFunction( m_pBorderWidths, pUserDataForFreeFunction );
    m_pBorderWidths = NULL;
    freeFunction( m_pDeleteBearingX, pUserDataForFreeFunction );
    m_pDeleteBearingX = NULL;
    freeFunction( m_pBearingOffsetX, pUserDataForFreeFunction );
    m_pBearingOffsetX = NULL;
    freeFunction( m_pForceMonospacedEnabled, pUserDataForFreeFunction );
    m_pForceMonospacedEnabled = NULL;
    freeFunction( m_pForceMonospacedWidth, pUserDataForFreeFunction );
    m_pForceMonospacedWidth = NULL;
    freeFunction( m_pCharCodeRangeCount, pUserDataForFreeFunction );
    m_pCharCodeRangeCount = NULL;
    freeFunction( m_pCharCodeRangeFirst, pUserDataForFreeFunction );
    m_pCharCodeRangeFirst = NULL;
    freeFunction( m_pCharCodeRangeLast, pUserDataForFreeFunction );
    m_pCharCodeRangeLast = NULL;
    freeFunction( m_pOtfKerningTable, pUserDataForFreeFunction);
    m_pOtfKerningTable = NULL;

    if (m_TexObj.IsInitialized())
    {
        m_TextureView.Finalize(pDevice);
        m_Texture.Finalize(pDevice);

        if (IsInternalMemoryPoolUsed())
        {
            m_MemoryPoolForTexture.Finalize(pDevice);
        }
    }

    if (IsInternalMemoryPoolUsed())
    {
        freeFunction(m_pTextureBitmap, pUserDataForFreeFunction);
        m_pTextureBitmap = NULL;
    }

#if NN_GFX_IS_TARGET_VK || NN_GFX_IS_TARGET_GL

    m_TemporalQueue.Finalize( m_pDevice );
    m_TemporalBuffer.Finalize( m_pDevice );
    m_TemporalCommandBuffer.Finalize( m_pDevice );
    m_TemporalMemoryPool.Finalize( m_pDevice );

    freeFunction( m_pTemporalCommandMemory, pUserDataForFreeFunction );
    freeFunction( m_pTemporalControlMemory, pUserDataForFreeFunction );

#endif
}

void TextureCache::RegisterTextureViewToDescriptorPool(nn::font::RegisterTextureViewSlot registerTextureViewSlot, void* pUserData)
{
    registerTextureViewSlot(&m_TexObj.GetDescriptorSlotForTexture(), m_TextureView, pUserData);
}

void TextureCache::UnregisterTextureViewFromDescriptorPool(nn::font::UnregisterTextureViewSlot unregisterTextureViewSlot, void* pUserData)
{
    unregisterTextureViewSlot(&m_TexObj.GetDescriptorSlotForTexture(), m_TextureView, pUserData);
}

bool TextureCache::RegisterGlyph(uint32_t code, uint32_t fontSize, uint16_t fontFace, int lockGroup, bool isDrawingHyphenMinusInsteadOfNonBreakingHyphen)
{
    code = ConvertNonBreakingHyphenToHyphenMinus(code, isDrawingHyphenMinusInsteadOfNonBreakingHyphen);
    uint32_t innerFontFace;
    if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code))
    {
        // 文字が存在しなかった場合は登録を行わずに抜ける。
        return false;
    }

    GlyphNode* pNode = m_GlyphTreeMap.Find(code, static_cast<uint16_t>(fontSize), static_cast<uint16_t>(innerFontFace));

    if (pNode)
    {
        // 見つかった -> 既に登録されている文字
        pNode->SetRequested(true);
        if (lockGroup >= 0)
        {
            pNode->GetLockGroup().SetBit(lockGroup, true);
        }
        // 既に登録されている文字の場合、IsPlottingがtrueのときだけまだ描画できない
        return pNode->IsPlotting();
    }
    else
    {
        // 見つからない -> まだ登録されていない文字
        pNode = m_GlyphTreeMap.Insert(code, static_cast<uint16_t>(fontSize), static_cast<uint16_t>(innerFontFace));
        if (pNode)
        {
            pNode->SetRequested(true);
            pNode->SetPlotting(true);
            if (lockGroup >= 0)
            {
                pNode->GetLockGroup().SetBit(lockGroup, true);
            }
            // 未プロットリストに入れる
            m_NeedPlotGlyphList.push_back(*pNode);
            return true;
        }
        else
        {
            // 平衡二分木のノードに空きがない。
            if (EraseNotInFontGlyphs() > 0) {
                // フォントにない文字のノードを削除し、空きができたら登録を行う。(次は登録できるはず)
                return RegisterGlyph(code, fontSize, static_cast<uint16_t>(innerFontFace), lockGroup);
            } else {
                // 空きができなかったら、文字の登録を行わずに抜ける。
                NN_DETAIL_FONT_ERROR("nn::font::TextureCache: Lack of the glyph node space. Please increase InitializeArg::glyphNodeCountMax.\n");
                m_NoSpaceError = NoSpaceError_LackOfGlyphNodeSpace;
                return false;
            }
        }
    }
}

uint32_t TextureCache::RegisterGlyphsWithLength(const uint16_t* pCode, uint32_t codeLength, uint32_t fontSize, uint16_t fontFace, int lockGroup, bool isDrawingHyphenMinusInsteadOfNonBreakingHyphen)
{
    uint32_t count = 0;
    for (uint32_t i = 0; i < codeLength; i++)
    {
        if (pCode[i] == 0)
        {
            break;
        }

        count += static_cast<uint32_t>(RegisterGlyph(pCode[i], fontSize, fontFace, lockGroup, isDrawingHyphenMinusInsteadOfNonBreakingHyphen));
    }
    return count;
}

uint32_t TextureCache::RegisterGlyphsWithLengthUtf8(const char* pCode, uint32_t codeLength, uint32_t fontSize, uint16_t fontFace, int lockGroup, bool isDrawingHyphenMinusInsteadOfNonBreakingHyphen)
{
    const char* pPtr = pCode;
    uint32_t count = 0;
    while (static_cast<uint32_t>(pPtr - pCode) != codeLength)
    {
        uint32_t code;
        {
            char buffer[4] = { 0 };
            nn::util::PickOutCharacterFromUtf8String(buffer, &pPtr);
            if (*buffer == 0)
            {
                break;
            }
            NN_SDK_ASSERT(static_cast<uint32_t>(pPtr - pCode) <= codeLength, "Wrong character code of UTF-8.");
            code = ConvertCharUtf8ToUtf32(buffer);
        }
        count += static_cast<uint32_t>(RegisterGlyph(code, fontSize, fontFace, lockGroup, isDrawingHyphenMinusInsteadOfNonBreakingHyphen));
    }
    return count;
}

void TextureCache::UpdateTextureCache()
{
    // 結局テクスチャキャッシュ内の文字の配置はUpdateTextureCache内で行わなければならない。
    // 文字の横幅を取る処理が重く、AcquireGlyphmap() で取ってきてその場で配置したほうがトータルのコストが小さいため。
    GlyphList::iterator stepIter = m_NeedPlotGlyphList.begin();
    GlyphList::iterator endIter = m_NeedPlotGlyphList.end();
    for (;stepIter != endIter;)
    {
        GlyphList::iterator iter = stepIter;
        ++stepIter;
        m_NeedPlotGlyphList.erase(iter);

        SetFontFace(iter->GetFace());

        // まず、文字がフォントにあるかどうかを調べる。ここで調べているのは、RegisterGlyphでやると時間がかかり、
        // メインスレッドの遅延に繋がるため。UpdateTextureCacheの中であれば、プロットの処理に紛れる
        if ( ! m_pFontEngine->CheckGlyphExist(iter->GetCode())) {
            // ない場合、削除は行わない。(削除してしまうとその文字のRegisterGlyphが常にtrueになってしまうため。)
            // フォントにない文字フラグを立て、フォントにない文字リストに入れておく。
            iter->SetNotInFont(true);
            m_NotInFontGlyphList.push_back(*iter);
            continue;
        }

        // フォントサイズの設定
        m_pFontEngine->SetScale(iter->GetSize() << 16, 0, 0, iter->GetSize() << 16);
        AssertFsError("SetScale");

        // このフォントのアセントを求める
        const int fontAscent = static_cast<int>(m_pFontMetrics[iter->GetFace()].boundingBoxAscentRatio * iter->GetSize() + 0.5f);

        // GRAYMAP8固定にする。そのままA8のテクスチャにコピーできるという利点がある。
        // GRAYMAP8の場合は、ScGlyphMapのメンバbplとwidthは同じ値になる。
        detail::GlyphMap *pGlyphMap = m_pFontEngine->AcquireGlyphmap(iter->GetCode(), detail::FormatGrayMap8);
        AssertFsError("AcquireGlyphmap");

        iter->SetLineKind(m_pCalculateLineKindFunction(static_cast<uint16_t>(pGlyphMap->height + fontAscent - pGlyphMap->hiY)));

        // 文字をプロットするラインを決める
        LineInfo* pMatchedLine = NULL;
        for (uint32_t i = 0; i < m_LineCurrentPos; i++)
        {
            if (m_LineInfos[i].kind == iter->GetLineKind() && (static_cast<uint32_t>(m_LineInfos[i].currentX + pGlyphMap->width + GlyphPadding) < m_TextureCacheWidth))
            {
                pMatchedLine = &m_LineInfos[i];
                break;
            }
        }
        // マッチするラインがなかった場合
        if ( ! pMatchedLine)
        {
            if (m_LineCurrentPos >= TextureCacheLineCountMax)
            {
                // ラインの最大数を超えた。エラー扱い
                NN_DETAIL_FONT_ERROR("nn::font::TextureCache: The number of the texture cache lines[%d] is over the max[%d].\n", m_LineCurrentPos, TextureCacheLineCountMax);
                m_NeedEraseGlyphList.push_back(*iter);
                m_pFontEngine->ReleasesGlyph(pGlyphMap);
                AssertFsError("ReleasesGlyph");
                m_NoSpaceError = NoSpaceError_TextureCacheLineIsOverMax;
                continue;
            }

            // 新しいラインを作成する
            pMatchedLine = CreateNewLineImpl(iter->GetLineKind());

            if (pMatchedLine == NULL)
            {
                // 新しいラインを作成できなかった。
                // 現在プロットされているものを消して、その上にプロットする必要がある。

                // 消去可能な文字を探す
                GlyphNode* pNode = NULL;
                pNode = FindAndReserveEraseGlyph(iter->GetLineKind(), pGlyphMap->width);
                if (pNode)
                {
                    // 元々あったグリフの領域に上書きする
                    GlyphLineList* list = &m_LineInfos[pNode->GetLineNo()].list;
                    iter->SetCachePosX(pNode->GetCachePosX());
                    iter->SetCachePosY(pNode->GetCachePosY());
                    iter->SetLineNo(pNode->GetLineNo());
                    list->insert(list->iterator_to(*pNode), *iter);
                }
                else
                {
                    // 入るスペースがないので抜ける
                    NN_DETAIL_FONT_ERROR("nn::font::TextureCache: Lack of the texture cache's space. A line, which has the height(%d), for a character(0x%x) could not be created.\n", m_pCalculateLineHeightFunction(iter->GetLineKind()), iter->GetCode());
                    m_NeedEraseGlyphList.push_back(*iter);
                    m_pFontEngine->ReleasesGlyph(pGlyphMap);
                    AssertFsError("ReleasesGlyph");
                    m_NoSpaceError = NoSpaceError_LackOfTextureCacheSpace;
                    continue;
                }
            }
        }

        if (pMatchedLine)
        {
            iter->SetCachePosX(pMatchedLine->currentX);
            iter->SetCachePosY(pMatchedLine->y);
            iter->SetLineNo(pMatchedLine->no);
            pMatchedLine->list.push_back(*iter);
            pMatchedLine->currentX += pGlyphMap->width + GlyphPadding;
        }

        const int offsetFromTop = fontAscent - pGlyphMap->hiY;
        const int renderOriginY = iter->GetCachePosY() + offsetFromTop;

        const float scaleWidth = m_pFontMetrics[iter->GetFace()].scaleWidth;
        const float scaleHeight = m_pFontMetrics[iter->GetFace()].scaleHeight;
        iter->SetGlyphWidth(static_cast<uint16_t>(pGlyphMap->width * scaleWidth));
        iter->SetGlyphHeight(static_cast<uint16_t>((pGlyphMap->height + offsetFromTop) * scaleHeight));
        iter->SetCacheWidth(pGlyphMap->width);
        iter->SetCacheHeight(static_cast<uint16_t>(pGlyphMap->height + offsetFromTop));

        if (m_pDeleteBearingX[iter->GetFace()] && pGlyphMap->width != 0)
        {
            // 文字幅ではなくグリフ幅で描画する。
            // スペースなどの幅 0 のグリフは文字幅で描画する。
            iter->SetAdvanceX(static_cast<uint16_t>((pGlyphMap->width + m_pBearingOffsetX[iter->GetFace()]) * scaleWidth));
            iter->SetLeftOffset(static_cast<int16_t>(m_pBearingOffsetX[iter->GetFace()] / 2 * scaleWidth));
        }
        else if (m_pForceMonospacedEnabled[iter->GetFace()])
        {
            // 強制的に等幅で描画する。
            iter->SetAdvanceX(static_cast<uint16_t>(m_pForceMonospacedWidth[iter->GetFace()] * scaleWidth));
            iter->SetLeftOffset(static_cast<int16_t>((m_pForceMonospacedWidth[iter->GetFace()] - pGlyphMap->width) / 2 * scaleWidth));
        }
        else
        {
            if (m_IsExactWidthReturnedAfterPlot)
            {
                iter->SetAdvanceX(static_cast<uint16_t>((pGlyphMap->idX + m_pBearingOffsetX[iter->GetFace()]) * scaleWidth));
            }
            else
            {
                int16_t idX;
                int16_t idY;
                int32_t dx;
                int32_t dy;
                m_pFontEngine->GetAdvance(&idX, &idY, &dx, &dy, iter->GetCode(), detail::FormatGrayMap8);
                iter->SetAdvanceX(static_cast<uint16_t>((idX + m_pBearingOffsetX[iter->GetFace()]) * scaleWidth));
            }
            iter->SetLeftOffset(static_cast<int16_t>((pGlyphMap->loX + m_pBearingOffsetX[iter->GetFace()] / 2) * scaleWidth));
        }
        iter->SetBaselineOffset(static_cast<int16_t>(fontAscent * scaleHeight + m_pFontMetrics[iter->GetFace()].baselineOffset));

        NN_SDK_ASSERT_NOT_NULL(m_pActiveMemoryPoolForTexture);

        const uint32_t copySize = pGlyphMap->width + 2;
        uint8_t* pTextureCache = static_cast<uint8_t*>(m_pActiveMemoryPoolForTexture->Map());
        pTextureCache += m_OffsetOfMemoryPoolForTexture;

        // 前の文字が描画されている場合があるので、文字の上のスペースをクリアする
        for (int l = iter->GetCachePosY(); l < renderOriginY; l++) {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            uint8_t* pLineStart = &pTextureCache[(m_TextureCacheHeight - l - 1) * m_TextureCacheWidth + iter->GetCachePosX()];
            std::memset(pLineStart, 0x00, copySize);
        }

        const int copyStartY = std::max(0, - renderOriginY);
        if (m_pBorderWidths[iter->GetFace()])
        {
            // フチを含めた全領域を白で塗りつぶす
            for (int l = copyStartY; l < pGlyphMap->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                uint8_t* pLineStart = &pTextureCache[(m_TextureCacheHeight - (l + renderOriginY) - 1) * m_TextureCacheWidth + iter->GetCachePosX()];

                // 文字をコピーして左右の1ピクセルずつを0でクリアする
                std::memcpy(pLineStart + 1, &pGlyphMap->bits[l * pGlyphMap->width], pGlyphMap->width);
                pLineStart[0] = 0;
                pLineStart[pGlyphMap->width + 1] = 0;
            }

            m_pFontEngine->ReleasesGlyph(pGlyphMap);
            m_pFontEngine->SetFlags(detail::ScalableFontEngine::Flags_OutlinedUnFilled);
            pGlyphMap = m_pFontEngine->AcquireGlyphmap(iter->GetCode(), detail::FormatGrayMap8);
            AssertFsError("AcquireGlyphmap");
            m_pFontEngine->SetFlags(detail::ScalableFontEngine::Flags_OutlinedFilled);

            // フチの部分だけを黒くする
            for (int l = copyStartY; l < pGlyphMap->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                uint8_t* pLineStart = &pTextureCache[(m_TextureCacheHeight - (l + renderOriginY) - 1) * m_TextureCacheWidth + iter->GetCachePosX()];

                for (int x = 0; x < pGlyphMap->width; ++x)
                {
                    // フチ描画の場合は、値を半分未満にすることでシェーダ内でフチとして扱っている
                    pLineStart[x + 1] -= pGlyphMap->bits[l * pGlyphMap->width + x] >> 1;
                }
            }
        }
        else
        {
            for (int l = copyStartY; l < pGlyphMap->height; l++) {
                // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
                uint8_t* pLineStart = &pTextureCache[(m_TextureCacheHeight - (l + renderOriginY) - 1) * m_TextureCacheWidth + iter->GetCachePosX()];

                // 文字をコピーして左右の1ピクセルずつを0でクリアする
                std::memcpy(pLineStart + 1, &pGlyphMap->bits[l * pGlyphMap->width], pGlyphMap->width);
                pLineStart[0] = 0;
                pLineStart[pGlyphMap->width + 1] = 0;
            }
        }
        // heightがそのラインの高さより小さい場合は、最大2ドット0でクリアする
        const uint32_t lineHeight = m_pCalculateLineHeightFunction(iter->GetLineKind());
        for (uint32_t l = pGlyphMap->height; l < static_cast<uint32_t>(pGlyphMap->height + 2) && l < lineHeight; l++)
        {
            // fontライブラリがy軸反転したテクスチャを要求するため、書き込み先のY軸を反転させる
            uint8_t* pLineStart = &pTextureCache[(m_TextureCacheHeight - (l + renderOriginY) - 1) * m_TextureCacheWidth + iter->GetCachePosX()];
            std::memset(pLineStart, 0, copySize);
        }
        m_pActiveMemoryPoolForTexture->Unmap();

        m_pFontEngine->ReleasesGlyph(pGlyphMap);
        AssertFsError("ReleasesGlyph");
    }
}// NOLINT(impl/function_size)

TextureCache::LineInfo* TextureCache::CreateNewLineImpl(uint8_t lineKind)
{
    uint32_t nextLineY;
    if (m_LineCurrentPos == 0)
    {
        nextLineY = 0;
    }
    else
    {
        nextLineY = m_LineInfos[m_LineCurrentPos - 1].y + m_pCalculateLineHeightFunction(m_LineInfos[m_LineCurrentPos - 1].kind) + GlyphPadding;
    }
    uint32_t endY = nextLineY + m_pCalculateLineHeightFunction(lineKind);

    if (endY >= m_TextureCacheHeight)
    {
        // 全てのラインが埋まっていて、新たにラインを作ることができない。
        return NULL;
    }
    else
    {
        // 新しいラインを作成できる
        LineInfo* pNewLine = &m_LineInfos[m_LineCurrentPos];
        pNewLine->currentX = 0;
        pNewLine->y = static_cast<uint16_t>(nextLineY);
        pNewLine->kind = lineKind;
        pNewLine->no = static_cast<uint8_t>(m_LineCurrentPos);

        m_LineCurrentPos++;
        return pNewLine;
    }
}

GlyphNode* TextureCache::FindAndReserveEraseGlyph(uint8_t lineKind, uint16_t glyphWidth)
{
    for (uint32_t l = 0; l < m_LineCurrentPos; l++)
    {
        LineInfo* pLineInfo = &m_LineInfos[l];
        if (pLineInfo->kind == lineKind)
        {
            // 種類が一致するラインを見つけた、消去可能な文字がないか探す
            GlyphLineList::iterator iter = pLineInfo->list.begin();
            GlyphLineList::iterator beginIter = pLineInfo->list.begin();
            GlyphLineList::iterator endIter = pLineInfo->list.end();
            GlyphLineList::iterator mostLeftErasableNodeIter = iter;
            uint32_t erasableNum = 0;
            for (;iter != endIter; ++iter)
            {
                if (iter->IsErasable())
                {
                    uint32_t erasableWidth;
                    if (erasableNum == 0)
                    {
                        mostLeftErasableNodeIter = iter;
                        erasableWidth = iter->GetCacheWidth();
                    }
                    else
                    {
                        erasableWidth = iter->GetCacheWidth() + iter->GetCachePosX() - mostLeftErasableNodeIter->GetCachePosX();
                    }
                    int mostLeftMoveValue = 0;
                    if (mostLeftErasableNodeIter != beginIter)
                    {
                        // mostLeftErasableNodeIterが最初のノードでなければ、左側の空きスペースを使うことができる
                        GlyphLineList::iterator mostLeftPrevIter = mostLeftErasableNodeIter;
                        do {
                            --mostLeftPrevIter;
                        } while (mostLeftPrevIter->IsErased() && beginIter != mostLeftPrevIter);

                        mostLeftMoveValue = mostLeftErasableNodeIter->GetCachePosX() - (mostLeftPrevIter->GetCachePosX() + mostLeftPrevIter->GetCacheWidth() + GlyphPadding);
                        if (mostLeftMoveValue > 0)
                        {
                            erasableWidth += mostLeftMoveValue;
                        }
                    }
                    if (erasableWidth >= glyphWidth)
                    {
                        // 消去可能な幅が必要な文字幅を超えた
                        // もう一度消去されることがないように消去フラグを立て、消去予約済みリストに入れる
                        GlyphLineList::iterator erasableIter = mostLeftErasableNodeIter;
                        GlyphLineList::iterator erasableEndIter = iter;
                        ++erasableEndIter;
                        for (; erasableIter != erasableEndIter; ++erasableIter)
                        {
                            if ( ! erasableIter->IsErased())
                            {
                                erasableIter->SetErased(true);
                                m_NeedEraseGlyphList.push_back(*erasableIter);
                            }
                        }
                        if (mostLeftMoveValue > 0)
                        {
                            // mostLeftErasableNodeIterの左側の空きスペースを使っている場合はCachePosXを動かす
                            mostLeftErasableNodeIter->SetCachePosX(static_cast<uint16_t>(mostLeftErasableNodeIter->GetCachePosX() - mostLeftMoveValue));
                        }
                        return &(*mostLeftErasableNodeIter);
                    }
                    erasableNum += 1;
                }
                else
                {
                    erasableNum = 0;
                }
            }
        }
    }
    return NULL;
}

void TextureCache::CompleteTextureCache()
{
#if NN_GFX_IS_TARGET_VK || NN_GFX_IS_TARGET_GL

    // ステージング用のバッファをマップし、更新したメモリの内容を反映します。
    // ステージング用のバッファからCopyBufferToImageによりテクスチャへデータを反映します。

    void* pBufferData = m_TemporalBuffer.Map();
    std::memcpy( pBufferData, m_pTextureBitmap, m_TextureCacheWidth * m_TextureCacheHeight );
    m_TemporalBuffer.FlushMappedRange( 0, m_TextureCacheWidth * m_TextureCacheHeight );
    m_TemporalBuffer.Unmap();

    m_TemporalCommandBuffer.Reset();
    m_TemporalCommandBuffer.AddCommandMemory( &m_TemporalMemoryPool, 0, TemporalCommandMemorySize );
    m_TemporalCommandBuffer.AddControlMemory( m_pTemporalControlMemory, TemporalControlMemorySize );
    m_TemporalCommandBuffer.Begin();

    nn::gfx::BufferTextureCopyRegion copyRegion;
    copyRegion.SetDefault();
    copyRegion.SetBufferImageWidth( m_TextureCacheWidth );
    copyRegion.SetBufferImageHeight( m_TextureCacheHeight );
    copyRegion.EditTextureCopyRegion().SetWidth( m_TextureCacheWidth );
    copyRegion.EditTextureCopyRegion().SetHeight( m_TextureCacheHeight );
    m_TemporalCommandBuffer.CopyBufferToImage( &m_Texture, &m_TemporalBuffer, copyRegion );

    m_TemporalCommandBuffer.End();
    m_TemporalQueue.ExecuteCommand( &m_TemporalCommandBuffer, NULL );
    m_TemporalQueue.Sync();

#endif

    // 消去予約済みリストに入っているノードを破棄する
    // 消去予約済みリストに入るのは、新しい文字を表示する必要があって削除する場合と、
    // プロットすることができなかったために削除する場合の二通りある。
    // 前者はIsErasedがtrueになり、後者はfalseになるので区別できる。後者はラインの
    // リストから削除する必要がない。
    GlyphList::iterator stepIter = m_NeedEraseGlyphList.begin();
    GlyphList::iterator endIter = m_NeedEraseGlyphList.end();
    for (;stepIter != endIter;)
    {
        GlyphList::iterator iter = stepIter;
        ++stepIter;
        m_NeedEraseGlyphList.erase(iter);
        if (iter->IsErased())
        {
            GlyphLineList& list = m_LineInfos[iter->GetLineNo()].list;
            list.erase(list.iterator_to(*iter));
            // グリフが消えたことによりラインの右側が空く可能性があるので、currentXを設定する
            if (list.size() > 0)
            {
                m_LineInfos[iter->GetLineNo()].currentX = list.back().GetCachePosX() + list.back().GetCacheWidth() + GlyphPadding;
            }
            else
            {
                m_LineInfos[iter->GetLineNo()].currentX = 0;
            }
        }
        m_GlyphTreeMap.Erase(iter->GetCode(), iter->GetSize(), iter->GetFace());
    }

    m_GlyphTreeMap.UpdateFlagsForCompleteTextureCache();
}

void TextureCache::ClearLockAllGlyphs(int lockGroup)
{
    m_GlyphTreeMap.ClearLockGroup(1 << lockGroup);
}

void TextureCache::ClearLockGlyphsWithLength(const uint16_t* pCode, uint32_t codeLength, uint32_t fontSize, uint16_t fontFace, int lockGroup, bool isDrawingHyphenMinusInsteadOfNonBreakingHyphen)
{
    uint32_t groupLockMask = 1 << lockGroup;

    for (uint32_t i = 0; i < codeLength; i++)
    {
        const uint32_t code = ConvertNonBreakingHyphenToHyphenMinus(pCode[i], isDrawingHyphenMinusInsteadOfNonBreakingHyphen);

        if (code == 0)
        {
            break;
        }

        uint32_t innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code))
        {
            break;
        }

        GlyphNode* pNode = m_GlyphTreeMap.Find(code, static_cast<uint16_t>(fontSize), static_cast<uint16_t>(innerFontFace));
        if (pNode)
        {
            pNode->GetLockGroup().SetAllBitOff(groupLockMask);
        }
    }
}

void TextureCache::ClearLockGlyphsWithLengthUtf8(const char* pCode, uint32_t codeLength, uint32_t fontSize, uint16_t fontFace, int lockGroup, bool isDrawingHyphenMinusInsteadOfNonBreakingHyphen)
{
    uint32_t groupLockMask = 1 << lockGroup;

    const char* pPtr = pCode;
    while (static_cast<uint32_t>(pPtr - pCode) != codeLength)
    {
        uint32_t code;
        {
            char buffer[4] = { 0 };
            nn::util::PickOutCharacterFromUtf8String(buffer, &pPtr);
            if (*buffer == 0)
            {
                break;
            }
            NN_SDK_ASSERT(static_cast<uint32_t>(pPtr - pCode) <= codeLength, "Wrong character code of UTF-8.");
            code = ConvertCharUtf8ToUtf32(buffer);
        }

        code = ConvertNonBreakingHyphenToHyphenMinus(code, isDrawingHyphenMinusInsteadOfNonBreakingHyphen);

        uint32_t innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code))
        {
            break;
        }

        GlyphNode* pNode = m_GlyphTreeMap.Find(code, static_cast<uint16_t>(fontSize), static_cast<uint16_t>(innerFontFace));
        if (pNode)
        {
            pNode->GetLockGroup().SetAllBitOff(groupLockMask);
        }

    }
}

void TextureCache::ResetTextureCache()
{
    NN_SDK_ASSERT_NOT_NULL(m_pActiveMemoryPoolForTexture);

    uint8_t* pTextureCache = static_cast<uint8_t*>(m_pActiveMemoryPoolForTexture->Map());
    std::memset(pTextureCache + m_OffsetOfMemoryPoolForTexture, 0x0, m_TextureCacheWidth * m_TextureCacheHeight);
    m_pActiveMemoryPoolForTexture->Unmap();

    m_NeedPlotGlyphList.clear();
    m_NeedEraseGlyphList.clear();
    m_NotInFontGlyphList.clear();

    // ラインをリセット
    for (uint32_t i = 0; i < m_LineCurrentPos; i++)
    {
        m_LineInfos[i].list.clear();
    }
    m_LineCurrentPos = 0;

    // 平衡二分木をリセット
    m_GlyphTreeMap.Reset();
}

void TextureCache::ChangeFontListOrder(uint32_t fontFaceIndex, uint32_t* pNewFontList)
{
    NN_SDK_ASSERT(0 <= fontFaceIndex && fontFaceIndex < m_FontCount);
    uint8_t* fontFaceTable = m_InnerFontFaceHeadTable[fontFaceIndex];
    ptrdiff_t headIdx = fontFaceTable - m_InnerFontFaceTable; // 先頭の内部フォントフェース番号をポインタから逆算する
    for (int i = 0; i < m_FontListCount[fontFaceIndex]; i++)
    {
        NN_SDK_ASSERT(0 <= pNewFontList[i] && pNewFontList[i] < m_FontListCount[fontFaceIndex]);
        fontFaceTable[i] = static_cast<uint8_t>(headIdx + pNewFontList[i]);
    }
}

bool TextureCache::IsBorderEffectEnabled(uint16_t fontFace) const
{
    // 1 つでも BorderWidth が設定されていれば true を返す
    NN_SDK_ASSERT(fontFace < m_FontCount);
    const uint8_t* pInnerFontFaceTable = m_InnerFontFaceHeadTable[fontFace];
    for (uint32_t i = 0; i < static_cast<uint32_t>(m_FontListCount[fontFace]); i++)
    {
        if (0 < m_pBorderWidths[pInnerFontFaceTable[i]])
        {
            return true;
        }
    }
    return false;
}

uint32_t TextureCache::CountPlottingGlyph(const uint16_t* pCode, uint32_t codeLength, uint32_t fontSize, uint16_t fontFace)
{
    uint32_t count = 0;
    for (uint32_t i = 0; i < codeLength; i++)
    {
        if (pCode[i] == 0)
        {
            break;
        }

        const uint32_t code = static_cast<uint32_t>(pCode[i]);

        uint32_t innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code))
        {
           // フォント内に指定したグリフが存在しない。
            continue;
        }

        // Register されていないか、Register されていても利用可能でない。
        const GlyphNode* pNode = m_GlyphTreeMap.Find(code, static_cast<uint16_t>(fontSize), static_cast<uint16_t>(innerFontFace));
        if (pNode == NULL || !pNode->IsUsable())
        {
            count++;
        }
    }
    return count;
}

bool TextureCache::IsGlyphExistInFont(uint32_t code, uint16_t fontFace)
{
    uint32_t innerFontFace;
    return GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code);
}

const TextureCache::FontMetrics& TextureCache::GetFontMetrics(uint16_t fontFace) const
{
    // 高さが最大のものを返す。
    NN_SDK_ASSERT(fontFace < m_FontCount);
    const uint8_t* pInnerFontFaceTable = m_InnerFontFaceHeadTable[fontFace];
    float maxHeight = m_pFontMetrics[pInnerFontFaceTable[0]].boundingBoxHeightRatio;
    uint32_t result = pInnerFontFaceTable[0];
    for (uint32_t i = 1; i < static_cast<uint32_t>(m_FontListCount[fontFace]); i++)
    {
        float height = m_pFontMetrics[pInnerFontFaceTable[i]].boundingBoxHeightRatio;
        if (maxHeight < height)
        {
            result = pInnerFontFaceTable[i];
            maxHeight = height;
        }
    }
    return m_pFontMetrics[result];
}

int TextureCache::CalculateCharWidth(uint32_t code, uint32_t fontSize, uint16_t fontFace)
{
    uint32_t innerFontFace;
    if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code))
    {
        return 0;
    }

    int16_t i_dx, i_dy;
    int32_t dx, dy;
    uint32_t coreId = GetCoreId();
    SetFontFaceNoPlot(innerFontFace, coreId);
    m_pFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
    AssertFsErrorNoPlot("SetScale", coreId);
    int result = m_pFontEngineNoPlot[coreId].GetAdvance(&i_dx, &i_dy, &dx, &dy, code, detail::FormatGrayMap8);
    if (result == detail::Success)
    {
        return i_dx;
    }
    else
    {
        // 指定された文字がフォントに含まれていない場合は 0 を返す仕様ですが、
        // メモリ不足になっている場合は assert で止めます。
        if (result == detail::ErrorMemoryAllocationFail)
        {
            m_IsWorkMemoryExhausted = true;
            NN_SDK_ASSERT(false, "Memory allocation fail. Increase the value of the InitializeArg::noPlotWorkMemorySize.");
        }
        return 0;
    }
}

int TextureCache::CalculateKerning(uint32_t c0, uint32_t c1, uint32_t fontSize, uint16_t fontFace)
{
    if (c0 == 0)
    {
        // 先頭の文字のカーニング処理
        uint32_t innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), c1))
        {
            return 0;
        }
        if (m_pDeleteBearingX[innerFontFace] || m_pForceMonospacedEnabled[innerFontFace])
        {
            return 0;
        }

        uint32_t coreId = GetCoreId();
        SetFontFaceNoPlot(innerFontFace, coreId);
        m_pFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
        AssertFsErrorNoPlot("SetScale", coreId);

        if (m_pOtfKerningTable[innerFontFace] != NULL)
        {
            const float scaleWidth = m_pFontMetrics[innerFontFace].scaleWidth;
            return static_cast<int>(m_pFontEngineNoPlot[coreId].AcquireOtfKerningFirst(m_pOtfKerningTable[innerFontFace], c1, fontSize) * scaleWidth);
        }
    }
    else if (c1 == 0)
    {
        // 末尾の文字のカーニング処理
        uint32_t innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), c0))
        {
            return 0;
        }
        if (m_pDeleteBearingX[innerFontFace] || m_pForceMonospacedEnabled[innerFontFace])
        {
            return 0;
        }

        uint32_t coreId = GetCoreId();
        SetFontFaceNoPlot(innerFontFace, coreId);
        m_pFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
        AssertFsErrorNoPlot("SetScale", coreId);

        if (m_pOtfKerningTable[innerFontFace] != NULL)
        {
            const float scaleWidth = m_pFontMetrics[innerFontFace].scaleWidth;
            return static_cast<int>(m_pFontEngineNoPlot[coreId].AcquireOtfKerningLast(m_pOtfKerningTable[innerFontFace], c0, fontSize) * scaleWidth);
        }
    }
    else
    {
        // 通常のカーニング処理
        uint32_t innerFontFace0;
        uint32_t innerFontFace1;
        if (!GetInnerFontFace(&innerFontFace0, static_cast<uint32_t>(fontFace), c0))
        {
            return 0;
        }
        if (!GetInnerFontFace(&innerFontFace1, static_cast<uint32_t>(fontFace), c1))
        {
            return 0;
        }
        if (innerFontFace0 != innerFontFace1)
        {
            return 0;
        }
        if (m_pDeleteBearingX[innerFontFace0] || m_pForceMonospacedEnabled[innerFontFace0])
        {
            return 0;
        }

        int32_t dx, dy;
        uint32_t coreId = GetCoreId();
        SetFontFaceNoPlot(innerFontFace0, coreId);
        m_pFontEngineNoPlot[coreId].SetScale(fontSize << 16, 0, 0, fontSize << 16);
        AssertFsErrorNoPlot("SetScale", coreId);

        // OTF 用のカーニングテーブルが存在していればそちらから取得する
        if (m_pOtfKerningTable[innerFontFace0] != NULL)
        {
            int kerning = m_pFontEngineNoPlot[coreId].AcquireOtfKerning(m_pOtfKerningTable[innerFontFace0], c0, c1, fontSize);
            if (kerning != 0)
            {
                const float scaleWidth = m_pFontMetrics[innerFontFace0].scaleWidth;
                return static_cast<int>(kerning * scaleWidth);
            }
        }

        // OTF 用のカーニングに存在しなければペアカーニングから取得する。
        if (m_pFontEngineNoPlot[coreId].GetKerning(&dx, &dy, c0, c1) == detail::Success)
        {
            // CalculateKerning が int を返すインターフェースになっているため、
            // 先に scaleWidth を掛けてからビットシフトしてカーニングの精度が落ちにくくする。
            // 互換性を維持するために float を返す変更はしないでおく。
            const float scaleWidth = m_pFontMetrics[innerFontFace0].scaleWidth;
            return static_cast<int>(dx * scaleWidth) >> 16;
        }
    }

    // カーニング情報が存在しなければ 0 とする
    return 0;
}

GlyphNode* TextureCache::FindGlyphNode(uint32_t code, uint32_t fontSize, uint16_t fontFace)
{
    uint32_t innerFontFace;
    if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code))
    {
        return NULL;
    }
    return m_GlyphTreeMap.Find(code, static_cast<uint16_t>(fontSize), static_cast<uint16_t>(innerFontFace));
}

uint32_t TextureCache::CountUnusableGlyph(const uint16_t* pCode, uint32_t codeLength, uint32_t fontSize, uint16_t fontFace)
{
    uint32_t count = 0;
    for (uint32_t i = 0; i < codeLength; i++)
    {
        if (pCode[i] == 0)
        {
            break;
        }

        uint16_t code = pCode[i];

        uint32_t innerFontFace;
        if (!GetInnerFontFace(&innerFontFace, static_cast<uint32_t>(fontFace), code))
        {
            // フォントに指定したグリフが存在しない場合はここではじかれる。
            continue;
        }

        // Register されていないか、Register されていても、利用可能ではない。
        GlyphNode* pNode = m_GlyphTreeMap.Find(code, static_cast<uint16_t>(fontSize), static_cast<uint16_t>(innerFontFace));
        if (pNode == NULL || !pNode->IsUsable())
        {
            count++;
        }
    }
    return count;
}

void TextureCache::SetFontFace(uint32_t innerFontFace)
{
    if (innerFontFace != m_CurrentFontFace)
    {
        // 今から使用するフォントの設定
        // 指定する文字のエンコーディングを設定
        m_pFontEngine->SetFont(GetFontNameBuf(innerFontFace));
        AssertFsError("SetFont");

        // ボールドのウェイトの設定
        m_pFontEngine->SetBoldWeight(m_pBoldWeights[innerFontFace]);
        AssertFsError("SetBoldWeight");

        // 縁取りの設定
        if (m_pBorderWidths[innerFontFace])
        {
            m_pFontEngine->SetOutlineWidth(m_pBorderWidths[innerFontFace]);
            m_pFontEngine->SetFlags(detail::ScalableFontEngine::Flags_OutlinedFilled);
        }
        else
        {
            m_pFontEngine->SetFlags(detail::ScalableFontEngine::Flags_NoEffect);
        }
        AssertFsError("SetFlags");

        m_CurrentFontFace = innerFontFace;
    }
}

void TextureCache::SetFontFaceNoPlot(uint32_t innerFontFace, uint32_t coreId)
{
    if (innerFontFace != m_CurrentFontFacesNoPlot[coreId])
    {
        // 今から使用するフォントの設定
        // 指定する文字のエンコーディングを設定
        m_pFontEngineNoPlot[coreId].SetFont(GetFontNameBuf(innerFontFace));
        AssertFsErrorNoPlot("SetFont", coreId);

        // ボールドのウェイトの設定
        m_pFontEngineNoPlot[coreId].SetBoldWeight(m_pBoldWeights[innerFontFace]);
        AssertFsErrorNoPlot("SetBoldWeight", coreId);

        // 縁取りの設定
        if (m_pBorderWidths[innerFontFace])
        {
            m_pFontEngineNoPlot[coreId].SetOutlineWidth(m_pBorderWidths[innerFontFace]);
            m_pFontEngineNoPlot[coreId].SetFlags(detail::ScalableFontEngine::Flags_OutlinedFilled);
        }
        else
        {
            m_pFontEngineNoPlot[coreId].SetFlags(detail::ScalableFontEngine::Flags_NoEffect);
        }
        AssertFsErrorNoPlot("SetFlags", coreId);

        m_CurrentFontFacesNoPlot[coreId] = innerFontFace;
    }
}

void TextureCache::AssertFsError(const char* pApiName) const
{
    NN_UNUSED(pApiName);
    int errorCode = m_pFontEngine->GetError();
    NN_UNUSED(errorCode);
    NN_UNUSED(&ChangeErrorCodeToMessage); // 関数を NN_UNUSED する場合は & を付けないと VS で警告が出る
    NN_SDK_ASSERT(errorCode == detail::Success, "%s error(%d): %s", pApiName, errorCode, ChangeErrorCodeToMessage(errorCode));
}

void TextureCache::AssertFsErrorNoPlot(const char* pApiName, uint32_t coreId) const
{
    NN_UNUSED(pApiName);
    NN_UNUSED(coreId);
    int errorCode = m_pFontEngineNoPlot[coreId].GetError();
    NN_UNUSED(errorCode);
    NN_UNUSED(&ChangeErrorCodeToMessage); // 関数を NN_UNUSED する場合は & を付けないと VS で警告が出る
    NN_SDK_ASSERT(errorCode == detail::Success, "%s error(%d): %s", pApiName, errorCode, ChangeErrorCodeToMessage(errorCode));
}

uint32_t TextureCache::EraseNotInFontGlyphs()
{
    uint32_t count = 0;
    GlyphList::iterator stepIter = m_NotInFontGlyphList.begin();
    GlyphList::iterator endIter = m_NotInFontGlyphList.end();
    for (;stepIter != endIter;)
    {
        GlyphList::iterator iter = stepIter;
        ++stepIter;
        // 現在使われている文字については削除しない
        if ( ! iter->IsRequestedOrKeeped()) {
            m_NotInFontGlyphList.erase(iter);
            m_GlyphTreeMap.Erase(iter->GetCode(), iter->GetSize(), iter->GetFace());
            count++;
        }
    }
    return count;
}

uint32_t TextureCache::GetCoreId() const
{
    if (m_IsDrawMultiCore)
    {
        int coreNumber = m_pGetCoreNumberFunction();
        NN_SDK_ASSERT(0 <= coreNumber && coreNumber < m_CoreCount, "Wrong core number.");
        return static_cast<uint32_t>(coreNumber);
    }
    else
    {
        return 0;
    }
}

int TextureCache::InitializeFontFaceTable(const InitializeArg& arg)
{
    // FontFace の総数を求める
    int fontFaceCount = 0;
    {
        for (uint32_t i = 0; i < arg.fontCount; i++)
        {
            if (arg.fontListCount[i] == 0 || arg.fontListCount[i] > MultiScalableFontCountMax)
            {
                NN_SDK_ASSERT(false, "arg.fontListCount must be from 1 to %d.", MultiScalableFontCountMax);
                return 0;
            }
            fontFaceCount += arg.fontListCount[i];
        }
    }

    {
        // 外部用 FontFace と内部用 FontFace をマッピングするテーブルの作成
        uint8_t idx = 0;
        for (int i = 0; i < FontFaceCountMax; i++)
        {
            if (static_cast<uint32_t>(i) >= arg.fontCount)
            {
                m_FontListCount[i] = 0;
                m_InnerFontFaceHeadTable[i] = NULL;
                continue;
            }
            m_FontListCount[i] = static_cast<uint8_t>(arg.fontListCount[i]);
            m_InnerFontFaceHeadTable[i] = &m_InnerFontFaceTable[idx];
            idx += static_cast<uint8_t>(arg.fontListCount[i]);
        }
        for (int i = 0; i < FontFaceCountMax * MultiScalableFontCountMax; i++)
        {
            m_InnerFontFaceTable[i] = static_cast<uint8_t>(i);
        }
    }
    return fontFaceCount;
}

bool TextureCache::GetInnerFontFace(uint32_t* pInnerFontFace, uint32_t outerFontFace, uint32_t code)
{
    NN_SDK_ASSERT(outerFontFace < m_FontCount);
    const uint8_t* pInnerFontFaceTable = m_InnerFontFaceHeadTable[outerFontFace];
    for (int i = 0; i < static_cast<int>(m_FontListCount[outerFontFace]); i++)
    {
        if (CheckCharCodeRange(pInnerFontFaceTable[i], code))
        {
            // CharCodeRange の中に該当する場合はグリフが存在するかを判別
            uint32_t coreId = GetCoreId();
            SetFontFaceNoPlot(pInnerFontFaceTable[i], coreId);
            bool existing = m_pFontEngineNoPlot[coreId].CheckGlyphExist(code);
            if (existing)
            {
                *pInnerFontFace = pInnerFontFaceTable[i];
                return true;
            }
        }
    }
    *pInnerFontFace = 0;
    return false;
}

bool TextureCache::CheckCharCodeRange(uint32_t fontFace, uint32_t code) const
{
    int charCodeRangeCount = m_pCharCodeRangeCount[fontFace];
    if (charCodeRangeCount == 0)
    {
        // CharCodeRange の数が 0 の場合はすべての文字コードを使用
        return true;
    }

    for (int j = 0; j < charCodeRangeCount; j++)
    {
        if (m_pCharCodeRangeFirst[fontFace][j] <= code && code <= m_pCharCodeRangeLast[fontFace][j])
        {
            return true;
        }
    }
    return false;
}

} // namespace nn::font
} // namespace nn
