﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_ResTexture.h>
#include <nn/gfx/gfx_ResTextureData.h>

#include <nn/font/font_ResFontBase.h>
#include <nn/font/font_RectDrawer.h>
#include <nn/perf.h>

typedef uint32_t  TexName;

namespace nn {
namespace font {

namespace {

enum Command
{
    Command_TexNearest = 0,
    Command_TexLinear  = 1
};

#define NN_FONT_COMMAND_TEX_WRAP_FILTER( magFilter, minFilter )             \
      ( (magFilter) << 1 | (minFilter) << 2 | ( 0/*isETC1*/ ? 2 : 0 ) << 4  \
    | 0/*wrapT*/ << 8 | 0/*wrapS*/ << 12                                    \
    | 0/*minFilter2*/ << 24 )

}   //     namespace


/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

ResFontBase::ResFontBase()
: Font()
, m_pResource(NULL)
, m_pResourceOffsetBase(NULL)
, m_pFontInfo(NULL)
, m_pKerningTable(NULL)
, m_pMemoryPool(NULL)
, m_MemoryPoolOffset(0)
, m_MemoryPoolSize(0)
, m_CharCodeRangeCount(0)
{
    SetLinearFilterEnabled(true, true);     // m_Filter
}

ResFontBase::~ResFontBase()
{
}



/* ------------------------------------------------------------------------
        構築/破棄
   ------------------------------------------------------------------------ */

void
ResFontBase::SetResourceBuffer(
    void*               pUserBuffer,
    FontInformation*    pFontInfo,
    FontKerningTable*   pKerningTable,
    nn::gfx::MemoryPool* pMemoryPool,
    ptrdiff_t memoryPoolOffset,
    size_t memoryPoolSize
)
{
    NN_SDK_ASSERT_NOT_NULL(pUserBuffer);
    NN_SDK_ASSERT_NOT_NULL(pFontInfo);
    NN_SDK_ASSERT(m_pResource == NULL);
    NN_SDK_ASSERT(m_pFontInfo == NULL);

    m_pResource = pUserBuffer;
    m_pFontInfo = pFontInfo;
    m_pKerningTable = pKerningTable;
    m_pMemoryPool = pMemoryPool;
    m_MemoryPoolOffset = memoryPoolOffset;
    m_MemoryPoolSize = memoryPoolSize;
}

void*
ResFontBase::RemoveResourceBuffer()
{
    void* pUserData = m_pResource;

    m_pResource = NULL;
    m_pFontInfo = NULL;
    m_pKerningTable = NULL;
    m_pMemoryPool = NULL;
    m_MemoryPoolOffset = 0;
    m_MemoryPoolSize = 0;

    return pUserData;
}

void
ResFontBase::RegisterTextureViewToDescriptorPool(RegisterTextureViewSlot pRegisterTextureViewSlot, void* pUserData)
{
    pRegisterTextureViewSlot(&m_TexObj.GetDescriptorSlotForTexture(), *m_TexObj.GetTextureView(), pUserData);
}

void
ResFontBase::UnregisterTextureViewFromDescriptorPool(UnregisterTextureViewSlot pUnregisterTextureViewSlot, void* pUserData)
{
    pUnregisterTextureViewSlot(&m_TexObj.GetDescriptorSlotForTexture(), *m_TexObj.GetTextureView(), pUserData);
}

/* ------------------------------------------------------------------------
        フォント全体情報アクセサ
   ------------------------------------------------------------------------ */

int
ResFontBase::GetWidth() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return m_pFontInfo->width;
}

int
ResFontBase::GetHeight() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return m_pFontInfo->height;
}

int
ResFontBase::GetAscent() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return m_pFontInfo->ascent;
}

int
ResFontBase::GetDescent() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return m_pFontInfo->height - m_pFontInfo->ascent;
}

int
ResFontBase::GetBaselinePos() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return nn::util::BytePtr(m_pResourceOffsetBase).Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>()->baselinePos;
}

int
ResFontBase::GetCellHeight() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return nn::util::BytePtr(m_pResourceOffsetBase).Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>()->cellHeight;
}

int
ResFontBase::GetCellWidth() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return nn::util::BytePtr(m_pResourceOffsetBase).Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>()->cellWidth;
}

int
ResFontBase::GetMaxCharWidth() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return nn::util::BytePtr(m_pResourceOffsetBase).Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>()->maxCharWidth;
}

Font::Type
ResFontBase::GetType() const
{
    return Type_Resource;
}

TexFmt
ResFontBase::GetTextureFormat() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    uint16_t    sheetFormat = nn::util::BytePtr(m_pResourceOffsetBase)
                                .Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>()->sheetFormat;
    return static_cast<TexFmt>(sheetFormat & FontSheetFormat_Mask);
}

int
ResFontBase::GetLineFeed() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return m_pFontInfo->linefeed;
}

const CharWidths
ResFontBase::GetDefaultCharWidths() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return m_pFontInfo->defaultWidth;
}

void
ResFontBase::SetDefaultCharWidths( const CharWidths& widths )
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    NN_SDK_ASSERT_NOT_NULL( &widths );
    m_pFontInfo->defaultWidth = widths;
}

bool
ResFontBase::SetAlternateChar( uint32_t c )
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    GlyphIndex index = FindGlyphIndex(c);

    if (index != GlyphIndexNotFound)
    {
        m_pFontInfo->alterCharIndex = index;
        return true;
    }

    return false;
}

void
ResFontBase::SetLineFeed( int linefeed )
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    NN_FONT_MINMAX_ASSERT(linefeed, SCHAR_MIN, SCHAR_MAX);
    m_pFontInfo->linefeed = static_cast<int8_t>(linefeed);
}




/* ------------------------------------------------------------------------
        文字単体情報アクセサ
   ------------------------------------------------------------------------ */

int
ResFontBase::GetCharWidth( uint32_t c ) const
{
    return GetCharWidths(c).charWidth;
}

const CharWidths
ResFontBase::GetCharWidths( uint32_t c ) const
{
    GlyphIndex index = GetGlyphIndex(c);
    return GetCharWidthsFromIndex(index);
}

void
ResFontBase::GetGlyph( Glyph* glyph, uint32_t c ) const
{
    GlyphIndex index = GetGlyphIndex(c);
    GetGlyphFromIndex(glyph, index);
}

bool
ResFontBase::HasGlyph( uint32_t c ) const
{
    return ( GlyphIndexNotFound != FindGlyphIndex(c) );
}

int
ResFontBase::GetKerning(uint32_t c0, uint32_t c1) const
{
    if (m_pKerningTable == NULL || ( ! IsKerningEnabled()))
    {
        return 0;
    }

    if (!CheckCharCodeRange(c0) || !CheckCharCodeRange(c1))
    {
        return 0;
    }

    uint32_t  firstWordCount = m_pKerningTable->firstWordCount;
    const KerningFirstTableElem* pFirstTable = m_pKerningTable->firstTable;

    const KerningSecondTable* pSecondTable = NULL;
    // pFirstTableからc0を二分探索
    {
        uint32_t  middle;
        uint32_t  start = 0;
        uint32_t  end = firstWordCount;
        for(;;)
        {
            middle = ( start + end ) / 2;
            uint32_t  firstWord = pFirstTable[ middle ].firstWord;
            if ( firstWord == c0 )
            {
                break;
            }
            else if ( firstWord < c0 )
            {
                if ( start == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                end = middle;
            }
        }

        uint32_t  offset = pFirstTable[ middle ].offset;
        pSecondTable = reinterpret_cast<const KerningSecondTable*>(&(reinterpret_cast<const uint8_t *>(m_pKerningTable)[offset]));
    }

    uint32_t  secondWordCount = pSecondTable->secondWordCount;
    const KerningSecondTableElem* pSecondTableElements = pSecondTable->elems;

    // pSecondTableからc1を二分探索
    {
        uint32_t  middle;
        uint32_t  start = 0;
        uint32_t  end = secondWordCount;
        for(;;)
        {
            middle = ( start + end ) / 2;
            uint32_t  secondWord = pSecondTableElements[ middle ].secondWord;
            if ( secondWord == c1 )
            {
                // 見つかった。カーニングの値を返す
                return pSecondTableElements[ middle ].kerningValue;
            }
            else if ( secondWord < c1 )
            {
                if ( start == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                start = middle;
            }
            else
            {
                if ( end == middle )
                {
                    // 見つからなかった
                    return 0;
                }
                end = middle;
            }
        }
    }
}

/* ------------------------------------------------------------------------
        文字ストリーム
   ------------------------------------------------------------------------ */

CharacterCode
ResFontBase::GetCharacterCode() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);

    return static_cast<CharacterCode>(m_pFontInfo->characterCode);
}



/* ------------------------------------------------------------------------
        グリフインデックス
   ------------------------------------------------------------------------ */

ResFontBase::GlyphIndex
ResFontBase::GetGlyphIndex( uint32_t c ) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    GlyphIndex index = FindGlyphIndex(c);
    return (index != GlyphIndexNotFound) ? index: m_pFontInfo->alterCharIndex;
}

ResFontBase::GlyphIndex
ResFontBase::FindGlyphIndex( uint32_t c ) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);

    uint32_t    offset = m_pFontInfo->pMap;
    // 参照するデータへのオフセットに 0 が入っている場合は終端とみなす。
    while (offset > 0)
    {
        const FontCodeMap* pMap = nn::util::ConstBytePtr(m_pResourceOffsetBase).Advance(offset).Get<FontCodeMap>();

        if (pMap->codeBegin <= c && c <= pMap->codeEnd)
        {
            return FindGlyphIndex(pMap, c);
        }

        offset = pMap->pNext;
    }

    return GlyphIndexNotFound;
}

ResFontBase::GlyphIndex
ResFontBase::FindGlyphIndex(
    const FontCodeMap*  pMap,
    uint32_t            c
) const
{
    NN_SDK_ASSERT_NOT_NULL(pMap);

    if (!CheckCharCodeRange(c))
    {
        return GlyphIndexNotFound;
    }

    uint16_t  index = GlyphIndexNotFound;

    switch (pMap->mappingMethod)
    {
    //-----------------------------------------------------------
    // インデックス = 文字コード - オフセット
    case FontMapMethod_Direct:
        {
            uint16_t  offset = pMap->mapInfo[0];
            index = static_cast<uint16_t >(c - pMap->codeBegin + offset);
        }
        break;

    //-----------------------------------------------------------
    // インデックス = table[文字コード - 文字コードオフセット]
    case FontMapMethod_Table:
        {
            const int tableIndex = c - pMap->codeBegin;

            index = pMap->mapInfo[tableIndex];
        }
        break;

    //-----------------------------------------------------------
    // インデックス = 二分探索(文字コード)
    case FontMapMethod_Scan:
        {
            const CMapInfoScan* const pScanInfo
                = reinterpret_cast<const CMapInfoScan*>(pMap->mapInfo);
            const CMapScanEntry* pFirstEntry  = &(pScanInfo->entries[0]);
            const CMapScanEntry* pLastEntry   = &(pScanInfo->entries[pScanInfo->count - 1]);

            while( pFirstEntry <= pLastEntry )
            {
                const CMapScanEntry* pMiddle = pFirstEntry + (pLastEntry - pFirstEntry) / 2;

                if( pMiddle->code < c )
                {
                    pFirstEntry = pMiddle + 1;
                }
                else if( c < pMiddle->code )
                {
                    pLastEntry = pMiddle - 1;
                }
                else
                {
                    index = pMiddle->index;
                    break;
                }
            }
        }
        break;

    //-----------------------------------------------------------
    // unknown
    default:
        NN_SDK_ASSERT(false, "unknwon MAPMETHOD");
    }

    return index;
}

const CharWidths&
ResFontBase::GetCharWidthsFromIndex( GlyphIndex index ) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    uint32_t    offset = m_pFontInfo->pWidth;

    // 参照するデータへのオフセットに 0 が入っている場合は終端とみなす。
    while (offset > 0)
    {
        const FontWidth* pWidth = nn::util::ConstBytePtr(m_pResourceOffsetBase).Advance(offset).Get<FontWidth>();

        if (pWidth->indexBegin <= index && index <= pWidth->indexEnd)
        {
            return GetCharWidthsFromIndex( pWidth, index );
        }

        offset = pWidth->pNext;
    }

    return m_pFontInfo->defaultWidth;
}

const CharWidths&
ResFontBase::GetCharWidthsFromIndex(
    const FontWidth*    pWidth,
    GlyphIndex          index
) const
{
    NN_SDK_ASSERT_NOT_NULL(pWidth);
    return pWidth->widthTable[index - pWidth->indexBegin];
}

void
ResFontBase::GetGlyphFromIndex(
    Glyph*      pGlyph,
    GlyphIndex  index
) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    const FontTextureGlyph& tg = *nn::util::ConstBytePtr(m_pResourceOffsetBase)
                                    .Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>();

    const uint32_t  cellsInASheet = detail::GetCellsInASheet(tg);
    const uint32_t  sheetNo       = index       / cellsInASheet;
    const uint32_t  offsetBytes   = sheetNo * tg.sheetSize;
    const void* pSheet = NULL;

    if (tg.sheetImage > 0)
    {
        pSheet = nn::util::BytePtr(m_pResourceOffsetBase).Advance(tg.sheetImage).Get<uint8_t>() + offsetBytes;
    }

    pGlyph->pTexture         = pSheet;
    const nn::font::CharWidths& widths = GetCharWidthsFromIndex(index);
    pGlyph->widths.left          = static_cast<int16_t>(widths.left);
    pGlyph->widths.glyphWidth    = static_cast<uint16_t>(widths.glyphWidth);
    pGlyph->widths.charWidth     = static_cast<uint16_t>(widths.charWidth);
    pGlyph->widths.rawWidth      = static_cast<uint16_t>(widths.glyphWidth);
    pGlyph->pTextureObject   = &m_TexObj;
    pGlyph->sheetIndex       = static_cast<uint8_t >(sheetNo);
    pGlyph->isSheetUpdated   = false;
    pGlyph->baselineDifference = 0;
    SetGlyphMember(pGlyph, index, tg);
}

void
ResFontBase::SetGlyphMember(
    Glyph*                  pGlyph,
    GlyphIndex              index,
    const FontTextureGlyph& tg
)
{
    const uint32_t  cellNo        = index       % detail::GetCellsInASheet(tg);
    const uint32_t  cellUnitX     = cellNo      % tg.sheetRow;
    const uint32_t  cellUnitY     = cellNo      / tg.sheetRow;
    const uint32_t  cellPixelX    = cellUnitX   * (tg.cellWidth  + 1);
    const uint32_t  cellPixelY    = cellUnitY   * (tg.cellHeight + 1);

    pGlyph->height       = static_cast<uint16_t>(tg.cellHeight);
    pGlyph->rawHeight    = static_cast<uint16_t>(tg.cellHeight);
    pGlyph->texFormat    = static_cast<TexFmt>(tg.sheetFormat & FontSheetFormat_Mask);
    pGlyph->texWidth     = tg.sheetWidth;
    pGlyph->texHeight    = tg.sheetHeight;
    pGlyph->cellX        = static_cast<uint16_t >(cellPixelX + 1);
    pGlyph->cellY        = static_cast<uint16_t >(cellPixelY + 1);
}

void* ResFontBase::FindBlock(nn::font::detail::BinaryFileHeader* pFileHeader, uint32_t sigword)
{
    nn::font::detail::BinaryBlockHeader* pBlockHeader;
    int nBlocks = 0;

    pBlockHeader = reinterpret_cast<nn::font::detail::BinaryBlockHeader*>(
        reinterpret_cast<uint8_t *>(pFileHeader) + pFileHeader->headerSize
    );

    while (nBlocks < pFileHeader->dataBlocks)
    {
        NN_SDK_ASSERT_NOT_NULL( pBlockHeader );
        if (pBlockHeader->kind == sigword)
        {
            return reinterpret_cast<uint8_t *>(pBlockHeader) + sizeof(*pBlockHeader);
        }

        pBlockHeader = reinterpret_cast<nn::font::detail::BinaryBlockHeader*>(
            reinterpret_cast<uint8_t *>(pBlockHeader) + pBlockHeader->size
        );
        nBlocks++;
    }

    return NULL;
}

/* ------------------------------------------------------------------------
        テクスチャフィルタ
   ------------------------------------------------------------------------ */

void
ResFontBase::SetLinearFilterEnabled(
    bool    atSmall,
    bool    atLarge
)
{
    // 同じレジスタに以下に設定される項目がありますが、フォントではサポートしていないため、
    // 常に固定値です。
    // ・テクスチャをラップして使用することはありません。
    // ・テクスチャをミップマップで使用することはありません。
    // ・テクスチャフォーマットとして ETC1(アルファ無し)はサポートしていません。
    // ・テクスチャユニット0がシャドウテクスチャが設定されることはありません。
    const int MagFilter = atLarge ? Command_TexLinear: Command_TexNearest;
    const int MinFilter = atSmall ? Command_TexLinear: Command_TexNearest;
    m_WrapFilter = static_cast<uint32_t >(
        NN_FONT_COMMAND_TEX_WRAP_FILTER(
            MagFilter,
            MinFilter));
}

bool
ResFontBase::IsLinearFilterEnabledAtSmall() const
{
    return 0 != (m_WrapFilter & (1 << 2));
}

bool
ResFontBase::IsLinearFilterEnabledAtLarge() const
{
    return 0 != (m_WrapFilter & (1 << 1));
}

uint32_t
ResFontBase::GetTextureWrapFilterValue() const
{
    return m_WrapFilter;
}

bool
ResFontBase::IsColorBlackWhiteInterpolationEnabled() const
{
    return m_TexObj.IsColorBlackWhiteInterpolationEnabled();
}

void
ResFontBase::SetColorBlackWhiteInterpolationEnabled(bool flag)
{
    m_TexObj.SetColorBlackWhiteInterpolationEnabled(flag);
}

bool
ResFontBase::IsBorderEffectEnabled() const
{
    return m_pFontInfo->fontType == FontType_FontTypePackedTexture;
}

void
ResFontBase::SetCharCodeRange(int count, uint32_t* pFirst, uint32_t* pLast)
{
    m_CharCodeRangeCount = count;
    for (int i = 0; i < count; i++)
    {
        NN_SDK_ASSERT(pFirst[i] <= pLast[i]);
        m_CharCodeRangeFirst[i] = pFirst[i];
        m_CharCodeRangeLast[i] = pLast[i];
    }
}

int
ResFontBase::GetActiveSheetCount() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFontInfo);
    return nn::util::BytePtr(m_pResourceOffsetBase).Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>()->sheetCount;
}


// シート毎のテクスチャオブジェクトの初期化
void
ResFontBase::GenTextureNames(nn::gfx::Device* pDevice)
{
    TextureObject* pTexObj = GetTextureObject();

    if (pTexObj->IsInitialized())
    {
        // ロード済み。
        return;
    }

    const FontTextureGlyph& tg = *nn::util::ConstBytePtr(m_pResourceOffsetBase)
                                    .Advance(m_pFontInfo->pGlyph).Get<FontTextureGlyph>();
    const int sheetNum = GetActiveSheetCount();
    // TextureObjectクラスのm_SheetNumがuint8_t なので、シートの枚数は255までしか扱えない
    NN_SDK_ASSERT(sheetNum <= 256);
    const void * pImage = NULL;

    if (tg.sheetImage > 0)
    {
        pImage = nn::util::BytePtr(m_pResourceOffsetBase).Advance(tg.sheetImage).Get<uint8_t>();
    }

    const TexFmt format = static_cast<TexFmt>(tg.sheetFormat);
    pTexObj->Set(pImage, format, tg.sheetWidth, tg.sheetHeight, static_cast<uint8_t >(sheetNum), true);

    ResourceTextureObject* pResTexObj = static_cast<ResourceTextureObject*>(pTexObj);
    LoadTexture(pDevice, pResTexObj);
}

void
ResFontBase::DeleteTextureNames(nn::gfx::Device* pDevice)
{
    TextureObject* pTexObj = GetTextureObject();

    ResourceTextureObject* pResTexObj = static_cast<ResourceTextureObject*>(pTexObj);
    UnloadTexture(pDevice, pResTexObj);

#if defined(NN_SDK_BUILD_DEBUG)
    pTexObj->Reset();
#endif
}

bool
ResFontBase::LoadTexture(nn::gfx::Device* pDevice, ResourceTextureObject* pTexObj) const
{
    NN_PERF_AUTO_MEASURE_INDEX_NAME(0, "nn::font::ResFontBase::LoadTexture");
    NN_SDK_ASSERT_NOT_NULL(pTexObj);

    // サポートされていない形式は初期化を中断します。
    if (pTexObj->GetFormat() >=  FontSheetFormat_MaxFontSheetFormat ||
        pTexObj->GetFormat() == FontSheetFormat_Rgb8 ||
        pTexObj->GetFormat() == FontSheetFormat_La4 ||
        pTexObj->GetFormat() == FontSheetFormat_A4 )
    {
        return false;
    }

    // Texture
    {
        nn::util::BytePtr binTop(m_pResource);

        const void* pImagePtr = reinterpret_cast<const void*>(pTexObj->GetImage());
        pTexObj->Initialize(pDevice, m_pMemoryPool, binTop.Distance(pImagePtr) + m_MemoryPoolOffset, m_MemoryPoolSize);
    }

    return true;
}

void
ResFontBase::UnloadTexture(nn::gfx::Device* pDevice, ResourceTextureObject* pTexObj) const
{
    if (pTexObj->IsInitialized())
    {
        pTexObj->Finalize(pDevice);
        pTexObj->Reset();
    }
}

bool
ResFontBase::CheckCharCodeRange(uint32_t code) const
{
    if (m_CharCodeRangeCount == 0)
    {
        // CharCodeRange の数が 0 の場合はすべての文字コードを使用
        return true;
    }

    for (int i = 0; i < m_CharCodeRangeCount; i++)
    {
        if (m_CharCodeRangeFirst[i] <= code && code <= m_CharCodeRangeLast[i])
        {
            return true;
        }
    }
    return false;
}

}   // namespace font
}   // namespace nn
