﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_StaticAssert.h>
#include <nn/font/font_DispStringBuffer.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_BinaryFormat.h>
#include <nn/font/detail/font_Log.h>

namespace nn
{
namespace font
{
namespace detail
{

void VertexShaderCharAttribute::Set(
    const float x,
    const float y,
    const float width,
    const float height,
    const nn::util::Float4& texCoord_,
    const nn::util::Unorm8x4& upperColor,
    const nn::util::Unorm8x4& lowerColor,
    const int32_t sheetIndex_,
    const float italicOffset_,
    const float rectangleTopXOffset)
{
    posAndSize.v[0] = width;
    posAndSize.v[1] = height;
    posAndSize.v[2] = x;
    posAndSize.v[3] = y;

    texCoord = texCoord_;

    color[detail::TextColor_Start] = upperColor;
    color[detail::TextColor_End] = lowerColor;
    sheetIndex = sheetIndex_;

    italicOffset = italicOffset_;
    translate.w = rectangleTopXOffset;
}

void VertexShaderCharAttributeWithTransform::Set(
    const float x,
    const float y,
    const float width,
    const float height,
    const nn::util::Float4& texCoord_,
    const nn::util::Unorm8x4& upperColor,
    const nn::util::Unorm8x4& lowerColor,
    const int32_t sheetIndex_,
    const float italicOffset_,
    const float rectangleTopXOffset,
    const nn::util::Float4& rotateMatrixAndCenterX_,
    const nn::util::Float4& rotateMatrixAndCenterY_,
    const nn::util::Float4& translate_)
{
    posAndSize.v[0] = width;
    posAndSize.v[1] = height;
    posAndSize.v[2] = x;
    posAndSize.v[3] = y;

    texCoord = texCoord_;

    color[detail::TextColor_Start] = upperColor;
    color[detail::TextColor_End] = lowerColor;
    sheetIndex = sheetIndex_;

    italicOffset = italicOffset_;

    rotateMatrixAndCenterX = rotateMatrixAndCenterX_;
    rotateMatrixAndCenterY = rotateMatrixAndCenterY_;
    translate = translate_;
    translate.w = rectangleTopXOffset;
}

void* CalculateVertexShaderCharAttributePtr(nn::util::BytePtr constantBuffer, ptrdiff_t bufferOffset, uint32_t charAttrWriteOffset, bool withTransform)
{
    const size_t size = withTransform ?
        sizeof(detail::VertexShaderCharAttributeWithTransform) :
        sizeof(detail::VertexShaderCharAttribute);
    return constantBuffer.Advance(bufferOffset + size * charAttrWriteOffset).Get();
}

}

DispStringBuffer::DispStringBuffer()
: m_ConstantBufferOffset(0)
, m_PerCharacterParamOffset(0)
, m_DrawContentFlags(0)
, m_CharCountMax(0)
, m_CharCount(0)
, m_pCharAttrs(NULL)
, m_pCharUseTextureIndices(NULL)
, m_pConstantBuffer(NULL)
, m_ShadowEnabled(false)
, m_DoubleDrawnBorder(false)
, m_PerCharacterTransformEnabled(false)
, m_PerCharacterTransformAutoShadowAlpha(false)
, m_FontHeight(0.0f)
{
    memset(&m_VertexBufferData, 0, sizeof(VertexBufferData));
}

//---------------------------------------------------------------------------------
DispStringBuffer::~DispStringBuffer()
{
    NN_SDK_ASSERT(!IsInitialized());
}

//---------------------------------------------------------------------------------
bool
DispStringBuffer::Initialize(nn::gfx::Device* pDevice, const InitializeArg& arg)
{
    NN_UNUSED(pDevice);
    NN_SDK_ASSERT(!IsInitialized());
    NN_SDK_ASSERT(arg.GetCharCountMax() >= 1);
    NN_SDK_ASSERT_NOT_NULL(arg.GetDrawBuffer());

    if (IsInitialized())
    {
        return false;
    }

    if (arg.GetCharCountMax() <= 0)
    {
        return false;
    }

    if (arg.GetDrawBuffer() == NULL)
    {
        return false;
    }

    void*    pDrawBuffer = static_cast<uint8_t*>(arg.GetDrawBuffer());

    m_CharCountMax = arg.GetCharCountMax();
    m_CharCount = 0;
    m_pCharAttrs = static_cast<CharAttribute*>(pDrawBuffer);
    pDrawBuffer = nn::util::BytePtr(pDrawBuffer).Advance(sizeof(CharAttribute) * arg.GetCharCountMax()).Get();
    m_pCharUseTextureIndices = static_cast<uint8_t*>(pDrawBuffer);
    pDrawBuffer = nn::util::BytePtr(pDrawBuffer).Advance(sizeof(uint8_t) * arg.GetCharCountMax()).Get();
    m_ShadowEnabled = arg.m_ShadowEnabled;
    m_DoubleDrawnBorder = arg.m_DoubleDrawnBorder;
    m_PerCharacterTransformEnabled = arg.m_PerCharacterTransformEnabled;
    m_PerCharacterTransformAutoShadowAlpha = arg.m_PerCharacterTransformAutoShadowAlpha;
    m_pConstantBuffer = arg.m_pConstantBuffer;

    return true;
}

//---------------------------------------------------------------------------------
void
DispStringBuffer::Finalize(nn::gfx::Device* pDevice)
{
    NN_UNUSED(pDevice);
    if (!IsInitialized())
    {
        return;
    }

    m_CharCountMax = 0;
}

//---------------------------------------------------------------------------------
size_t
DispStringBuffer::GetRequiredDrawBufferSize(const InitializeArg& arg)
{
    size_t bufferSize = 0;

    // 文字ごとの描画情報
    bufferSize += sizeof(CharAttribute) * arg.GetCharCountMax();
    // 文字ごとの使用テクスチャインデックス
    bufferSize += sizeof(uint8_t) * arg.GetCharCountMax();

    return bufferSize;
}

//---------------------------------------------------------------------------------
size_t
DispStringBuffer::GetRequiredConstantBufferSize(nn::gfx::Device* pDevice, const InitializeArg& arg)
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);

    size_t  constantBufferSize = nn::util::align_up(sizeof(ShaderParam), nn::gfx::Buffer::GetBufferAlignment(pDevice, info));

    // フチ２回描画の場合はそれぞれの描画ごとにコンスタントバッファが必要なため
    // 倍量のコンスタントバッファ領域を確保する。
    if (arg.m_DoubleDrawnBorder)
    {
        constantBufferSize *= 2;
    }

    size_t  perCharacterAttributeDataSize = sizeof(detail::VertexShaderCharAttribute);

    if (arg.m_PerCharacterTransformEnabled)
    {
        perCharacterAttributeDataSize = sizeof(detail::VertexShaderCharAttributeWithTransform);
    }

    // 文字ごとに必要なコンスタントバッファのサイズを計算する。
    // 基本は一文字ごとに最大 detail::VertexShaderCharAttributeWithTransform 分必要です。
    size_t perCharacterAttributeTotalSize = nn::util::align_up(perCharacterAttributeDataSize * arg.GetCharCountMax(), nn::gfx::Buffer::GetBufferAlignment(pDevice, info));
    constantBufferSize += perCharacterAttributeTotalSize;

    // 影がある場合は影用の領域も含めて倍のサイズが必要です。
    if (arg.m_ShadowEnabled)
    {
        constantBufferSize += perCharacterAttributeTotalSize;
    }

    return constantBufferSize;
}

//---------------------------------------------------------------------------------
void
DispStringBuffer::SetConstantBuffer(GpuBuffer* pGpuBuffer)
{
    m_pConstantBuffer = pGpuBuffer;
}

//---------------------------------------------------------------------------------
void
DispStringBuffer::GetConstantBufferGpuAddress(nn::gfx::GpuAddress* pGpuAddr) const
{
    NN_SDK_ASSERT_NOT_NULL(m_pConstantBuffer);
    NN_SDK_ASSERT_NOT_NULL(pGpuAddr);

    *pGpuAddr = *m_pConstantBuffer->GetGpuAddress();
}

//----------------------------------------
void
DispStringBuffer::BuildConstantBuffer(
    const nn::util::MatrixT4x4fType& projectionMtx,
    const ConstantBufferAdditionalContent* pAdditionalContent,
    bool isDrawFromRightToLeftEnabled,
    bool isOriginToCenterEnabled)
{
    static ConstantBufferAdditionalContent defaultContent;

    NN_SDK_ASSERT_NOT_NULL(m_pConstantBuffer);
    NN_SDK_ASSERT(IsInitialized());

    if (m_pConstantBuffer == NULL)
    {
        return;
    }

    if (!IsInitialized())
    {
        return;
    }

    // 各文字共通のコンスタントバッファを作成します。
    size_t  constantBufferSize = nn::util::align_up(sizeof(DispStringBuffer::ShaderParam), m_pConstantBuffer->GetBufferAlignment());
    size_t  requiredConstantBufferSize = constantBufferSize;

    const ConstantBufferAdditionalContent& content = (pAdditionalContent)? *pAdditionalContent : defaultContent;

    // フチ２回描画のためにコンスタントバッファを倍量確保する。
    if (m_DoubleDrawnBorder)
    {
        requiredConstantBufferSize = constantBufferSize * 2;
    }

    m_ConstantBufferOffset = static_cast<uint32_t>(m_pConstantBuffer->Allocate(nn::util::align_up(requiredConstantBufferSize, m_pConstantBuffer->GetBufferAlignment())));

    nn::util::BytePtr   pConstantBuffer(m_pConstantBuffer->GetMappedPointer());
    if (pConstantBuffer.Get() != NULL)
    {
        DispStringBuffer::ShaderParam* shaderParam = static_cast<DispStringBuffer::ShaderParam*>(pConstantBuffer.Advance(m_ConstantBufferOffset).Get());

        if (m_DoubleDrawnBorder)
        {
            // フチ二回描画用コンスタントバッファ作成
            // 最初にフチ用、次に中の文字用のコンスタントバッファを作成する。
            nn::font::ConstantBufferAdditionalContent borderDrawContent = content;

            // nw 実装では RGB のみコピーしている。
            // nn 環境では直接メンバー変数にアクセス出来ないため同等の実装としておく。
            nn::util::Float4    whiteColor = content.GetInterpolateBlackFloat();
            whiteColor.v[3] = borderDrawContent.GetInterpolateWhiteFloat().v[3];
            borderDrawContent.SetInterpolateWhite(whiteColor);

            borderDrawContent.SetInterpolateAlpha(std::numeric_limits<uint8_t>::max());

            whiteColor = content.GetShadowInterpolateBlackFloat();
            whiteColor.v[3] = borderDrawContent.GetShadowInterpolateWhiteFloat().v[3];
            borderDrawContent.SetShadowInterpolateWhite(whiteColor);

            BuildCommonConstantBufferData(*shaderParam, projectionMtx, borderDrawContent);

            pConstantBuffer.Reset(m_pConstantBuffer->GetMappedPointer());
            shaderParam = static_cast<DispStringBuffer::ShaderParam*>(pConstantBuffer.Advance(m_ConstantBufferOffset + constantBufferSize).Get());
            BuildCommonConstantBufferData(*shaderParam, projectionMtx, content);
        }
        else
        {
            // フチ二回描画なしの通常のコンスタントバッファ作成。
            BuildCommonConstantBufferData(*shaderParam, projectionMtx, content);
        }
    }

    // 個別の文字を描画するためのコンスタントバッファを作成します。
    BuildPerCharacterAttributeConstantBuffer(
        content,
        isDrawFromRightToLeftEnabled,
        isOriginToCenterEnabled);

    m_DrawContentFlags = content.m_Flags;

    if (content.m_pShadowParam != NULL)
    {
        m_DrawContentFlags |= ConstantBufferAdditionalContent::Flags_ShadowEnabled;
    }

    if (content.m_pPerCharacterTransformInfos != NULL)
    {
        m_DrawContentFlags |= ConstantBufferAdditionalContent::Flags_PerCharacterTransformEnabled;
    }
}

//----------------------------------------
void
DispStringBuffer::BuildCommonConstantBufferData(
    DispStringBuffer::ShaderParam&  shaderParam,
    const nn::util::MatrixT4x4fType& projectionMtx,
    const ConstantBufferAdditionalContent& content) const
{
    NN_SDK_ASSERT(IsInitialized());

    if (!IsInitialized())
    {
        return;
    }

    nn::util::MatrixT4x4fType wvp;
    if (content.m_pViewMatrix)
    {
        nn::util::MatrixT4x4fType matrix4x4;
        nn::util::MatrixConvert(&matrix4x4, *content.m_pViewMatrix);
        nn::util::MatrixMultiply(&wvp, matrix4x4, projectionMtx);
    }
    else
    {
        wvp = projectionMtx;
    }

    if (content.m_pLocalMatrix)
    {
        nn::util::MatrixT4x4fType matrix4x4;
        nn::util::MatrixConvert(&matrix4x4, *content.m_pLocalMatrix);
        nn::util::MatrixMultiply(&wvp, matrix4x4, wvp);
    }
    nn::util::MatrixStore(reinterpret_cast<nn::util::FloatT4x4*>(&shaderParam.m_Mtx), wvp);

    const float TO_FLOAT = 1.0f / std::numeric_limits<uint8_t>::max();

    nn::util::Float4 black = NN_UTIL_FLOAT_4_INITIALIZER(
        content.m_InterpolateBlackColor.v[0],
        content.m_InterpolateBlackColor.v[1],
        content.m_InterpolateBlackColor.v[2],
        0.0f);
    nn::util::Float4 white = NN_UTIL_FLOAT_4_INITIALIZER(
        content.m_InterpolateWhiteColor.v[0],
        content.m_InterpolateWhiteColor.v[1],
        content.m_InterpolateWhiteColor.v[2],
        content.m_InterpolateWhiteColor.v[3] * (content.m_InterpolateWhiteAlpha * TO_FLOAT));
    nn::util::Float4 shadowBlack = NN_UTIL_FLOAT_4_INITIALIZER(
        content.m_ShadowInterpolateBlackColor.v[0],
        content.m_ShadowInterpolateBlackColor.v[1],
        content.m_ShadowInterpolateBlackColor.v[2],
        0.0f);
    nn::util::Float4 shadowWhite = NN_UTIL_FLOAT_4_INITIALIZER(
        content.m_ShadowInterpolateWhiteColor.v[0],
        content.m_ShadowInterpolateWhiteColor.v[1],
        content.m_ShadowInterpolateWhiteColor.v[2],
        content.m_ShadowInterpolateWhiteColor.v[3] * (content.m_ShadowInterpolateWhiteAlpha * TO_FLOAT));

    shaderParam.m_InterpolateOffset[0] = black.v[0];
    shaderParam.m_InterpolateOffset[1] = black.v[1];
    shaderParam.m_InterpolateOffset[2] = black.v[2];
    shaderParam.m_InterpolateOffset[3] = black.v[3];
    shaderParam.m_InterpolateWidth[0] = white.v[0] - black.v[0];
    shaderParam.m_InterpolateWidth[1] = white.v[1] - black.v[1];
    shaderParam.m_InterpolateWidth[2] = white.v[2] - black.v[2];
    shaderParam.m_InterpolateWidth[3] = white.v[3] - black.v[3];

    shaderParam.m_ShadowInterpolateOffset[0] = shadowBlack.v[0];
    shaderParam.m_ShadowInterpolateOffset[1] = shadowBlack.v[1];
    shaderParam.m_ShadowInterpolateOffset[2] = shadowBlack.v[2];
    shaderParam.m_ShadowInterpolateOffset[3] = shadowBlack.v[3];
    shaderParam.m_ShadowInterpolateWidth[0] = shadowWhite.v[0] - shadowBlack.v[0];
    shaderParam.m_ShadowInterpolateWidth[1] = shadowWhite.v[1] - shadowBlack.v[1];
    shaderParam.m_ShadowInterpolateWidth[2] = shadowWhite.v[2] - shadowBlack.v[2];
    shaderParam.m_ShadowInterpolateWidth[3] = shadowWhite.v[3] - shadowBlack.v[3];

    SwapEndian(shaderParam.m_Mtx[0][0]);
    SwapEndian(shaderParam.m_Mtx[0][1]);
    SwapEndian(shaderParam.m_Mtx[0][2]);
    SwapEndian(shaderParam.m_Mtx[0][3]);
    SwapEndian(shaderParam.m_Mtx[1][0]);
    SwapEndian(shaderParam.m_Mtx[1][1]);
    SwapEndian(shaderParam.m_Mtx[1][2]);
    SwapEndian(shaderParam.m_Mtx[1][3]);
    SwapEndian(shaderParam.m_Mtx[2][0]);
    SwapEndian(shaderParam.m_Mtx[2][1]);
    SwapEndian(shaderParam.m_Mtx[2][2]);
    SwapEndian(shaderParam.m_Mtx[2][3]);
    SwapEndian(shaderParam.m_Mtx[3][0]);
    SwapEndian(shaderParam.m_Mtx[3][1]);
    SwapEndian(shaderParam.m_Mtx[3][2]);
    SwapEndian(shaderParam.m_Mtx[3][3]);
    SwapEndian(shaderParam.m_InterpolateWidth[0]);
    SwapEndian(shaderParam.m_InterpolateWidth[1]);
    SwapEndian(shaderParam.m_InterpolateWidth[2]);
    SwapEndian(shaderParam.m_InterpolateWidth[3]);
    SwapEndian(shaderParam.m_InterpolateOffset[0]);
    SwapEndian(shaderParam.m_InterpolateOffset[1]);
    SwapEndian(shaderParam.m_InterpolateOffset[2]);
    SwapEndian(shaderParam.m_InterpolateOffset[3]);
    SwapEndian(shaderParam.m_ShadowInterpolateWidth[0]);
    SwapEndian(shaderParam.m_ShadowInterpolateWidth[1]);
    SwapEndian(shaderParam.m_ShadowInterpolateWidth[2]);
    SwapEndian(shaderParam.m_ShadowInterpolateWidth[3]);
    SwapEndian(shaderParam.m_ShadowInterpolateOffset[0]);
    SwapEndian(shaderParam.m_ShadowInterpolateOffset[1]);
    SwapEndian(shaderParam.m_ShadowInterpolateOffset[2]);
    SwapEndian(shaderParam.m_ShadowInterpolateOffset[3]);
}

//----------------------------------------
void
DispStringBuffer::BuildPerCharacterAttributeConstantBuffer(
    const ConstantBufferAdditionalContent& content,
    bool isDrawFromRightToLeftEnabled,
    bool isOriginToCenterEnabled)
{
    int charCount = GetCharCount();

    size_t  perCharacterAttributeDataSize = sizeof(detail::VertexShaderCharAttribute);
    if (m_PerCharacterTransformEnabled)
    {
        perCharacterAttributeDataSize = sizeof(detail::VertexShaderCharAttributeWithTransform);
    }

    // 文字毎に必要なパラメータのメモリを確保する。
    if (content.m_pShadowParam != NULL)
    {
        // 影つきの場合は影描画用も含めて倍のサイズを確保する。
        size_t  bufferSize = nn::util::align_up(
            perCharacterAttributeDataSize * charCount * 2,
            m_pConstantBuffer->GetBufferAlignment());
        m_PerCharacterParamOffset = static_cast<uint32_t>(m_pConstantBuffer->Allocate(bufferSize));
    }
    else
    {
        size_t  bufferSize = perCharacterAttributeDataSize * charCount;
        m_PerCharacterParamOffset = static_cast<uint32_t>(m_pConstantBuffer->Allocate(nn::util::align_up(bufferSize, m_pConstantBuffer->GetBufferAlignment())));
    }

    const PerCharacterTransformInfo* pPerCharacterTransformInfos = content.m_pPerCharacterTransformInfos;
    ConstantBufferAdditionalContent::PerCharacterTransformCenter perCharacterTransformCenter = content.m_PerCharacterTransformCenter;
    float perCharacterTransformCenterOffset = content.m_PerCharacterTransformCenterOffset;

    const ShadowParameter* shadowParam = content.m_pShadowParam;

    // テクスチャの使用状態を更新する。
    // このメソッドの中で描画される文字数が決定されている。
    BuildTextureUseInfos(shadowParam != NULL);

    // 描画する際に文字ごとに必要な情報をコンスタントバッファ内に作成する。
    BuildPerCharacterParams(
        m_PerCharacterParamOffset,
        pPerCharacterTransformInfos,
        perCharacterTransformCenter,
        perCharacterTransformCenterOffset,
        shadowParam,
        isDrawFromRightToLeftEnabled,
        isOriginToCenterEnabled);
}

//----------------------------------------
void DispStringBuffer::BuildPerCharacterParams(
    ptrdiff_t   bufferOffset,
    const PerCharacterTransformInfo* pPerCharacterTransformInfos,
    const ConstantBufferAdditionalContent::PerCharacterTransformCenter perCharacterTransformCenter,
    const float perCharacterTransformCenterOffset,
    const ShadowParameter*  pShadowParam,
    bool isDrawFromRightToLeftEnabled,
    bool isOriginToCenterEnabled)
{
    nn::util::BytePtr constantBuffer(m_pConstantBuffer->GetMappedPointer());

    // コンスタントバッファへ各テクスチャごとに文字描画情報をまとめて格納するため
    // それぞれのテクスチャごとに使用されている文字数から格納場所の基点を計算する。
    uint32_t    offsetCharCount = 0;
    uint32_t    constantBufferOffsetCharCount[detail::UseTextureCountMax];
    memset(constantBufferOffsetCharCount, 0, sizeof(uint32_t) * detail::UseTextureCountMax);

    for(int i = 0; i < m_VertexBufferData.textureUseInfoPos;++i)
    {
        constantBufferOffsetCharCount[i] = offsetCharCount;
        offsetCharCount += m_VertexBufferData.textureUseInfos[i].useCount;
    }

    uint32_t    texCharCount[detail::UseTextureCountMax];
    memset(texCharCount, 0, sizeof(uint32_t) * detail::UseTextureCountMax);

    if (constantBuffer.Get() != NULL)
    {
        int charCount = GetCharCount();

        const uint32_t offsetScale = pShadowParam != NULL ? 2 : 1; // 影を使う場合はインデックスが 2 倍必要になる

        if (pShadowParam != NULL)
        {
            if (isDrawFromRightToLeftEnabled)
            {
                for (int charIndex = charCount - 1; charIndex >= 0; --charIndex)
                {
                    BuildShadowBufferPerCharacterParams(
                        bufferOffset,
                        pPerCharacterTransformInfos,
                        perCharacterTransformCenter,
                        perCharacterTransformCenterOffset,
                        pShadowParam,
                        constantBufferOffsetCharCount,
                        constantBuffer,
                        offsetScale,
                        texCharCount,
                        charIndex,
                        isOriginToCenterEnabled);
                }
            }
            else
            {
                for (int charIndex = 0; charIndex < charCount; ++charIndex)
                {
                    BuildShadowBufferPerCharacterParams(
                        bufferOffset,
                        pPerCharacterTransformInfos,
                        perCharacterTransformCenter,
                        perCharacterTransformCenterOffset,
                        pShadowParam,
                        constantBufferOffsetCharCount,
                        constantBuffer,
                        offsetScale,
                        texCharCount,
                        charIndex,
                        isOriginToCenterEnabled);
                }
            }
        }

        if (isDrawFromRightToLeftEnabled)
        {
            for (int charIndex = charCount - 1; charIndex >= 0; --charIndex)
            {
                BuildCharacterBufferPerCharacterParams(
                    bufferOffset,
                    pPerCharacterTransformInfos,
                    perCharacterTransformCenter,
                    perCharacterTransformCenterOffset,
                    constantBufferOffsetCharCount,
                    constantBuffer,
                    offsetScale,
                    texCharCount,
                    charIndex,
                    isOriginToCenterEnabled);
            }
        }
        else
        {
            for (int charIndex = 0; charIndex < charCount; ++charIndex)
            {
                BuildCharacterBufferPerCharacterParams(
                    bufferOffset,
                    pPerCharacterTransformInfos,
                    perCharacterTransformCenter,
                    perCharacterTransformCenterOffset,
                    constantBufferOffsetCharCount,
                    constantBuffer,
                    offsetScale,
                    texCharCount,
                    charIndex,
                    isOriginToCenterEnabled);
            }
        }
    }
}

void DispStringBuffer::BuildShadowBufferPerCharacterParams(
    ptrdiff_t   bufferOffset,
    const PerCharacterTransformInfo* pPerCharacterTransformInfos,
    const ConstantBufferAdditionalContent::PerCharacterTransformCenter perCharacterTransformCenter,
    const float perCharacterTransformCenterOffset,
    const ShadowParameter*  pShadowParam,
    const uint32_t* pConstantBufferOffsetCharCount,
    nn::util::BytePtr constantBuffer,
    uint32_t offsetScale,
    uint32_t* pTexCharCount,
    int charIndex,
    bool isOriginToCenterEnabled)
{
    CharAttribute* pAttr = &GetCharAttributes()[charIndex];

    // 使用しているテクスチャと描画済みの文字数からコンスタントバッファにデータを
    // 書き込む位置を計算する。
    const uint32_t    useTexIndex = m_pCharUseTextureIndices[charIndex];
    const uint32_t    charAttrWriteOffset = pConstantBufferOffsetCharCount[useTexIndex] * offsetScale + pTexCharCount[useTexIndex];
    pTexCharCount[useTexIndex]++;

    // ランダムアクセスするため、文字毎にポインタを再計算する必要がある。
    void* pParams = detail::CalculateVertexShaderCharAttributePtr(constantBuffer, bufferOffset, charAttrWriteOffset, pPerCharacterTransformInfos != NULL);

    const float width = pAttr->pos.v[0] * pShadowParam->shadowScale.v[0];
    const float height = pAttr->pos.v[1] * pShadowParam->shadowScale.v[1];
    const float x = (isOriginToCenterEnabled ? -width / 2.0f : pAttr->pos.v[2]) + pShadowParam->shadowOffset.v[0];
    const float y = pAttr->pos.v[3] - pShadowParam->shadowOffset.v[1] + (pAttr->pos.v[1] - height);
    nn::util::Unorm8x4 upperColor = pShadowParam->shadowUpperColor;
    upperColor.v[3] = upperColor.v[3] * pAttr->shadowAlpha / 255;
    nn::util::Unorm8x4 lowerColor = pShadowParam->shadowLowerColor;
    lowerColor.v[3] = lowerColor.v[3] * pAttr->shadowAlpha / 255;
    // italicOffset は斜体にしたときの文字ごとの傾きのオフセット値。
    // rectangleTopXOffset は傾けた文字を適切な位置に移動させるオフセット値。
    const float rectangleTopXOffset = static_cast<float>(pAttr->italicOffset) + pShadowParam->shadowItalicOffset * pShadowParam->shadowScale.v[0];
    const float italicOffset = rectangleTopXOffset * pAttr->pos.v[1] / m_FontHeight;

    if (pPerCharacterTransformInfos == NULL)
    {
        // 変換なし、通常描画用パラメータ設定。
        static_cast<detail::VertexShaderCharAttribute*>(pParams)->Set(
            x, y, width, height,
            pAttr->tex, upperColor, lowerColor, -pAttr->sheetIndex - 1,
            italicOffset, rectangleTopXOffset);
    }
    else
    {
        // 変換あり描画用パラメータ設定。
        NN_SDK_ASSERT(m_PerCharacterTransformEnabled);

        nn::util::Float4    rotateMatrixAndCenterX;
        nn::util::Float4    rotateMatrixAndCenterY;
        nn::util::Float4    translate;

        CalculatePerCharacterTransform(
            rotateMatrixAndCenterX, rotateMatrixAndCenterY, translate,
            x, y, width, height, italicOffset, rectangleTopXOffset,
            pPerCharacterTransformInfos[charIndex],
            perCharacterTransformCenter,
            perCharacterTransformCenterOffset,
            isOriginToCenterEnabled,
            -pShadowParam->shadowOffset.v[1]);

        if (m_PerCharacterTransformAutoShadowAlpha)
        {
            // 影のアルファを文字のアルファに追従させる
            upperColor.v[3] = static_cast<uint8_t>(upperColor.v[3] * pPerCharacterTransformInfos[charIndex].LT[3] / 255);
            lowerColor.v[3] = static_cast<uint8_t>(lowerColor.v[3] * pPerCharacterTransformInfos[charIndex].LB[3] / 255);
        }

        static_cast<detail::VertexShaderCharAttributeWithTransform*>(pParams)->Set(
            x, y, width, height,
            pAttr->tex, upperColor, lowerColor, -pAttr->sheetIndex - 1,
            italicOffset, rectangleTopXOffset,
            rotateMatrixAndCenterX, rotateMatrixAndCenterY, translate);
    }
}

void DispStringBuffer::BuildCharacterBufferPerCharacterParams(
    ptrdiff_t   bufferOffset,
    const PerCharacterTransformInfo* pPerCharacterTransformInfos,
    const ConstantBufferAdditionalContent::PerCharacterTransformCenter perCharacterTransformCenter,
    const float perCharacterTransformCenterOffset,
    const uint32_t* pConstantBufferOffsetCharCount,
    nn::util::BytePtr constantBuffer,
    uint32_t offsetScale,
     uint32_t* pTexCharCount,
    int charIndex,
    bool isOriginToCenterEnabled)
{
    CharAttribute* pAttr = &GetCharAttributes()[charIndex];

    // 使用しているテクスチャと描画済みの文字数からコンスタントバッファにデータを
    // 書き込む位置を計算する。
    const uint32_t    useTexIndex = m_pCharUseTextureIndices[charIndex];
    const uint32_t    charAttrWriteOffset = pConstantBufferOffsetCharCount[useTexIndex] * offsetScale + pTexCharCount[useTexIndex];
    pTexCharCount[useTexIndex]++;

    // ランダムアクセスするため、文字毎にポインタを再計算する必要がある。
    void* pParams = detail::CalculateVertexShaderCharAttributePtr(constantBuffer, bufferOffset, charAttrWriteOffset, pPerCharacterTransformInfos != NULL);

    {
        const float width = pAttr->pos.v[0];
        const float height = pAttr->pos.v[1];
        const float x = isOriginToCenterEnabled ? -width / 2.0f : pAttr->pos.v[2];
        const float y = pAttr->pos.v[3];
        const nn::util::Unorm8x4& upperColor = pAttr->color[detail::TextColor_Start];
        const nn::util::Unorm8x4& lowerColor = pAttr->color[detail::TextColor_End];
        // italicOffset は斜体にしたときの文字ごとの傾きのオフセット値。
        // rectangleTopXOffset は傾けた文字を適切な位置に移動させるオフセット値。
        const float rectangleTopXOffset = static_cast<float>(pAttr->italicOffset);
        const float italicOffset = rectangleTopXOffset * pAttr->pos.v[1] / m_FontHeight;

        if (pPerCharacterTransformInfos == NULL)
        {
            // 変換なし、通常描画用パラメータ設定。
            static_cast<detail::VertexShaderCharAttribute*>(pParams)->Set(
                x, y, width, height,
                pAttr->tex, upperColor, lowerColor, pAttr->sheetIndex,
                italicOffset, rectangleTopXOffset);
        }
        else
        {
            // 変換あり描画用パラメータ設定。
            NN_SDK_ASSERT(m_PerCharacterTransformEnabled);

            nn::util::Float4    rotateMatrixAndCenterX;
            nn::util::Float4    rotateMatrixAndCenterY;
            nn::util::Float4    translate;

            CalculatePerCharacterTransform(
                rotateMatrixAndCenterX, rotateMatrixAndCenterY, translate,
                x, y, width, height, italicOffset, rectangleTopXOffset,
                pPerCharacterTransformInfos[charIndex],
                perCharacterTransformCenter,
                perCharacterTransformCenterOffset,
                isOriginToCenterEnabled,
                0.0f);

            const nn::util::Unorm8x4 animUpperColor =
            { {
                static_cast<uint8_t>(upperColor.v[0] * pPerCharacterTransformInfos[charIndex].LT[0] / 255),
                static_cast<uint8_t>(upperColor.v[1] * pPerCharacterTransformInfos[charIndex].LT[1] / 255),
                static_cast<uint8_t>(upperColor.v[2] * pPerCharacterTransformInfos[charIndex].LT[2] / 255),
                static_cast<uint8_t>(upperColor.v[3] * pPerCharacterTransformInfos[charIndex].LT[3] / 255)
            } };
            const nn::util::Unorm8x4 animLowerColor =
            { {
                static_cast<uint8_t>(lowerColor.v[0] * pPerCharacterTransformInfos[charIndex].LB[0] / 255),
                static_cast<uint8_t>(lowerColor.v[1] * pPerCharacterTransformInfos[charIndex].LB[1] / 255),
                static_cast<uint8_t>(lowerColor.v[2] * pPerCharacterTransformInfos[charIndex].LB[2] / 255),
                static_cast<uint8_t>(lowerColor.v[3] * pPerCharacterTransformInfos[charIndex].LB[3] / 255)
            } };

            static_cast<detail::VertexShaderCharAttributeWithTransform*>(pParams)->Set(
                x, y, width, height,
                pAttr->tex, animUpperColor, animLowerColor, pAttr->sheetIndex,
                italicOffset, rectangleTopXOffset,
                rotateMatrixAndCenterX, rotateMatrixAndCenterY, translate);
        }
    }
}

void
DispStringBuffer::CalculatePerCharacterTransform(
    nn::util::Float4& outRotateMatrixAndCenterX,
    nn::util::Float4& outRotateMatrixAndCenterY,
    nn::util::Float4& outTranslate,
    const float x,
    const float y,
    const float width,
    const float height,
    const float italicOffset,
    const float rectangleTopXOffset,
    const PerCharacterTransformInfo& perCharacterTransformInfo,
    const ConstantBufferAdditionalContent::PerCharacterTransformCenter perCharacterTransformCenter,
    const float perCharacterTransformCenterOffset,
    bool isOriginToCenterEnabled,
    float originToCenterShadowOffset) const
{
    NN_UNUSED(italicOffset);

    // 回転および平行移動を計算
    const float centerX = (x + x + width + rectangleTopXOffset) / 2.0f;
    float centerY;

    if (isOriginToCenterEnabled)
    {
        centerY = y;
    }
    else
    {
        switch (perCharacterTransformCenter)
        {
        case ConstantBufferAdditionalContent::PerCharacterTransformCenter_Center:
            centerY = (y + y + height) / 2.0f;
            break;
        case ConstantBufferAdditionalContent::PerCharacterTransformCenter_Bottom:
            centerY = y + height;
            break;
        default:
            centerY = 0.0f;
            NN_SDK_ASSERT(false, "Invalid value of letterMotionCenter.");
            break;
        }
        centerY += -perCharacterTransformCenterOffset; // フォントは Y 軸がレイアウトと逆なのでオフセットの正負を反転する
    }

    const float matrix00 = perCharacterTransformInfo.RotationCos[1] * perCharacterTransformInfo.RotationCos[2];
    const float matrix01 = -perCharacterTransformInfo.RotationCos[1] * perCharacterTransformInfo.RotationSin[2];
    const float matrix10 = perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationCos[2] + perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[2];
    const float matrix11 = -perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationSin[2] + perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationCos[2];
    const float matrix20 = -perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationCos[2] + perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[2];
    const float matrix21 = perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationSin[2] + perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationCos[2];

    // 座標変換に必要なパラメータを書き込む。
    /*
        [ RX.x  RX.y  RX.z ][ S.x  0    0 ]   [ RX.x * S.x  RX.y * S.y  RX.z ]
        [ RY.x  RY.y  RY.z ][ 0    S.y  0 ] = [ RY.x * S.x  RY.y * S.y  RY.z ]
        [ 0     0     1    ][ 0    0    1 ]   [ 0           0           1    ]
    */
    outRotateMatrixAndCenterX.x = matrix00 * perCharacterTransformInfo.Scale[0];
    outRotateMatrixAndCenterX.y = matrix10 * perCharacterTransformInfo.Scale[1];
    outRotateMatrixAndCenterX.z = matrix20;
    outRotateMatrixAndCenterX.w = centerX;
    outRotateMatrixAndCenterY.x = matrix01 * perCharacterTransformInfo.Scale[0];
    outRotateMatrixAndCenterY.y = matrix11 * perCharacterTransformInfo.Scale[1];
    outRotateMatrixAndCenterY.z = matrix21;
    outRotateMatrixAndCenterY.w = centerY;

    outTranslate.x = perCharacterTransformInfo.Translation[0];
    if (isOriginToCenterEnabled)
    {
        // 回転原点 Y がずれたままになっていたため、
        // 円形配置の機能を使ったときには本来の Y 座標の位置に戻す。
        // 影のオフセットまで戻ってしまうため、別途加算する。
        outTranslate.y = perCharacterTransformInfo.Translation[1] - centerY + originToCenterShadowOffset;
    }
    else
    {
        outTranslate.y = perCharacterTransformInfo.Translation[1];
    }
    outTranslate.z = perCharacterTransformInfo.Translation[2];
}

//----------------------------------------
void
DispStringBuffer::BuildTextureUseInfos(bool shadowEnabled)
{
    NN_UNUSED(shadowEnabled);

    uint32_t  textureUseInfoPos = 0;

    // 利用回数(dispStringBuffer->textureUseInfos[i].useCount) をカウントします。
    uint32_t  count = GetCharCount();
    detail::TextureUseInfo* pTextureUseInfos = m_VertexBufferData.textureUseInfos;
    for (uint32_t  i = 0; i < count; ++i)
    {
        const TextureObject* pTexObj = GetCharAttributes()[i].GetTexObj();
        // 既に登録されているか探す
        bool isNew = true;
        uint8_t textureUseInfoIndex = 0;
        for ( textureUseInfoIndex = 0; textureUseInfoIndex < textureUseInfoPos; ++textureUseInfoIndex )
        {
            if (pTexObj == pTextureUseInfos[textureUseInfoIndex].pTexObj)
            {
                isNew = false;
                pTextureUseInfos[textureUseInfoIndex].useCount++;
                break;
            }
        }
        if (isNew)
        {
            detail::TextureUseInfo* pInfo = &pTextureUseInfos[textureUseInfoPos];
            pInfo->pTexObj = pTexObj;
            pInfo->useCount = 1;
            pInfo->bitFlags = 0;

            if (pTexObj->IsColorBlackWhiteInterpolationEnabled())
            {
                pInfo->bitFlags |= detail::BitFlags_ColorBlackWhiteInterpolationEnabled ;
            }

            if (GetCharAttributes()[i].GetFlags() & CharAttribute::Flags_BorderEffect)
            {
                pInfo->bitFlags |= detail::BitFlags_BorderEffectEnabled;
            }

            ++textureUseInfoPos;
            // 上限に達したら抜ける
            if (textureUseInfoPos >= detail::UseTextureCountMax)
            {
                NN_DETAIL_FONT_WARN("texture num exceeds limit\n");
                break;
            }
        }
        m_pCharUseTextureIndices[i] = textureUseInfoIndex;
    }

    m_VertexBufferData.textureUseInfoPos = textureUseInfoPos;
}

//----------------------------------------
void DispStringBuffer::SetFontHeight(float fontHeight)
{
    m_FontHeight = fontHeight;
}

//----------------------------------------
bool DispStringBuffer::CompareCopiedInstanceTest(const DispStringBuffer& target) const
{
    if (m_CharCountMax != target.m_CharCountMax ||
        m_ShadowEnabled != target.m_ShadowEnabled ||
        m_DoubleDrawnBorder != target.m_DoubleDrawnBorder ||
        m_PerCharacterTransformEnabled != target.m_PerCharacterTransformEnabled ||
        m_PerCharacterTransformAutoShadowAlpha != target.m_PerCharacterTransformAutoShadowAlpha ||
        m_FontHeight != target.m_FontHeight)
    {
        return false;
    }

    return true;
}


}   // namespace font
}   // namespace nn
