﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>

#include <nn/font/font_CharWriter.h>
#include <nn/font/font_DispStringBuffer.h>
#include <nn/util/util_Vector.h>
#include <nn/font/detail/font_Log.h>

namespace nn {
namespace font {

/* =======================================================================
        public
   ======================================================================== */

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

CharWriter::CharWriter()
:   m_GradationMode(GradationMode_None),
    m_FixedWidth(0),
    m_ItalicRatio(0.0f),
    m_pFont(NULL),
    m_pDispStringBuffer(NULL),
    m_IsWidthFixed(false)
{
    const nn::util::Unorm8x4 white = { { std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max(), std::numeric_limits<uint8_t>::max() } };
    SetTextColor(white);                              // m_TextColor, mVertexColor
    SetShadowAlpha(255);                              // m_ShadowAlpha
    SetScale(1, 1);                                   // m_Scale
    SetCursor(0, 0, 0);                               // m_CursorPos
}

CharWriter::~CharWriter()
{
}

/* ------------------------------------------------------------------------
        文字サイズ
   ------------------------------------------------------------------------ */

void
CharWriter::SetFontSize(
    float width,
    float height
)
{
    NN_SDK_ASSERT_NOT_NULL(m_pFont);
    NN_FONT_MIN_ASSERT(m_pFont->GetWidth(),  1);
    NN_FONT_MIN_ASSERT(m_pFont->GetHeight(), 1);
    SetScale(
        width  / m_pFont->GetWidth(),
        height / m_pFont->GetHeight()
    );
}

void
CharWriter::SetFontSize(float height)
{
    NN_SDK_ASSERT_NOT_NULL(m_pFont);
    NN_FONT_MIN_ASSERT(m_pFont->GetHeight(), 1);
    const float scale = height / m_pFont->GetHeight();
    SetScale(scale);
}

float
CharWriter::GetFontWidth() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFont);
    return m_pFont->GetWidth() * m_Scale.v[0];
}

float
CharWriter::GetFontHeight() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFont);
    return m_pFont->GetHeight() * m_Scale.v[1];
}

float
CharWriter::GetFontAscent() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFont);
    return m_pFont->GetAscent() * m_Scale.v[1];
}

float
CharWriter::GetFontDescent() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pFont);
    return m_pFont->GetDescent() * m_Scale.v[1];
}

/* ------------------------------------------------------------------------
        文字描画
   ------------------------------------------------------------------------ */

float
CharWriter::PrintGlyph(const Glyph& glyph)
{
    const Glyph::GlyphCharWidths& widths = glyph.widths;

    float width;
    float left;
    if (m_IsWidthFixed)
    {
        float margin = (m_FixedWidth - widths.charWidth * m_Scale.v[0]) / 2;

        width   = m_FixedWidth;
        left    = margin + widths.left * m_Scale.v[0];
    }
    else
    {
        width   = widths.charWidth * m_Scale.v[0];
        left    = widths.left      * m_Scale.v[0];
    }

    PrintGlyph(m_CursorPos.v[0] + left, glyph);

    m_CursorPos.v[0] += width;

    return width;
}

float
CharWriter::Print(uint32_t code)
{
    NN_SDK_ASSERT_NOT_NULL(m_pFont);
    NN_SDK_ASSERT(code != Font::InvalidCharacterCode);

    Glyph glyph;
    m_pFont->GetGlyph(&glyph, code);

    return PrintGlyph(glyph);
}

void
CharWriter::DrawGlyph(const Glyph& glyph)
{
    NN_SDK_ASSERT_NOT_NULL(&glyph);
    PrintGlyph(m_CursorPos.v[0], glyph);
    m_CursorPos.v[0] += glyph.widths.glyphWidth * m_Scale.v[0];
}


/* =======================================================================
        private
   ======================================================================== */


void
CharWriter::PrintGlyph(
    float           x,
    const Glyph&  glyph
)
{
    NN_SDK_ASSERT_NOT_NULL(&glyph);
    NN_FONT_MIN_ASSERT(glyph.texWidth, 1);
    NN_FONT_MIN_ASSERT(glyph.texHeight, 1);

    float texLeft;
    float texRight;

    // cellYは左上原点の値が入っているので、これをOpenGLの左下原点の
    // テクスチャ座標に変換してセットする。
    float texTop;
    float texBottom;

    float px;
    float py;
    float width;
    float height;

    if (m_pFont->IsExtraMarginEnabled())
    {
        const float margin = 0.5f;
        texLeft   = (1.0f * glyph.cellX - margin) / glyph.texWidth;
        texRight  = (1.0f * (glyph.cellX + glyph.widths.rawWidth) + margin)
                              / glyph.texWidth;
        texTop    = 1.f - (static_cast<float>(glyph.cellY) - margin) / glyph.texHeight;
        texBottom = 1.f - (static_cast<float>(glyph.cellY + glyph.rawHeight) + margin) / glyph.texHeight;

        px = x - margin * m_Scale.v[0];
        py = m_CursorPos.v[1] - margin * m_Scale.v[1];
        width  = (glyph.widths.glyphWidth + margin * 2.0f) * m_Scale.v[0];
        height = (glyph.height + margin * 2.0f) * m_Scale.v[1];
    }
    else
    {
        texLeft   = (1.0f * glyph.cellX) / glyph.texWidth;
        texRight  = (1.0f * (glyph.cellX + glyph.widths.rawWidth))
                              / glyph.texWidth;

        texTop    = 1.f - static_cast<float>(glyph.cellY) / glyph.texHeight;
        texBottom = 1.f - static_cast<float>(glyph.cellY + glyph.rawHeight) / glyph.texHeight;

        px = x;
        py = m_CursorPos.v[1];
        width  = glyph.widths.glyphWidth * m_Scale.v[0];
        height = glyph.height * m_Scale.v[1];
    }

    {
        const int  charIdx = m_pDispStringBuffer->GetCharCount();

        if (charIdx >= m_pDispStringBuffer->GetCharCountMax())
        {
            NN_DETAIL_FONT_WARN("nn::font::CharWriter : Vertex Buffer Over.\n");
            return;
        }

        m_pDispStringBuffer->SetCharCount(charIdx + 1);

        CharAttribute* pCharAttrs =
            &m_pDispStringBuffer->GetCharAttributes()[charIdx];

        // ポジションのセット
        // カーソルが進んだら下に行くようにyを反転する
        pCharAttrs->pos.v[0] = width;
        pCharAttrs->pos.v[1] = height;
        pCharAttrs->pos.v[2] = px;
        pCharAttrs->pos.v[3] = py;

        // カラーのセット
        for (int i = 0; i < detail::TextColor_MaxTextColor; ++i)
        {
            pCharAttrs->color[i] = m_TextColors[i];
        }

        // 影のアルファのセット
        pCharAttrs->shadowAlpha = m_ShadowAlpha;

        // テクスチャ座標のセット
        pCharAttrs->tex.v[0] = texLeft;
        pCharAttrs->tex.v[1] = texTop;
        pCharAttrs->tex.v[2] = texRight;
        pCharAttrs->tex.v[3] = texBottom;

        pCharAttrs->italicOffset = static_cast<int16_t >(
            m_ItalicRatio * m_Scale.v[0] * m_pFont->GetWidth());
        {
            // 斜体の計算で必要になるので、
            // フォント全体の高さ情報を DispStringBuffer に渡しておく。
            // 拡張マージンが有効であればその値も加味する。
            const float margin = 0.5f;
            float fontHeight = GetFontHeight();
            if (m_pFont->IsExtraMarginEnabled())
            {
                fontHeight += (margin * 2.0f) * m_Scale.v[1];
            }
            m_pDispStringBuffer->SetFontHeight(fontHeight);
        }

        pCharAttrs->sheetIndex = glyph.sheetIndex;

        // テクスチャオブジェクトへのポインタのセット
        NN_SDK_ASSERT_NOT_NULL(glyph.pTextureObject);
        uint8_t  flags = 0;
        if (m_pFont->IsBorderEffectEnabled())
        {
            flags |= CharAttribute::Flags_BorderEffect;
        }
        pCharAttrs->SetTexObjAndFlags(glyph.pTextureObject, flags);
    }
}

void
CharWriter::StartPrint()
{
    NN_SDK_ASSERT_NOT_NULL(m_pDispStringBuffer);

    m_pDispStringBuffer->SetCharCount(0);
}

void
CharWriter::EndPrint() const
{
    NN_SDK_ASSERT_NOT_NULL(m_pDispStringBuffer);
}


/* ------------------------------------------------------------------------
        static
   ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------
        描画準備
   ------------------------------------------------------------------------ */

size_t
CharWriter::GetDispStringBufferSize(uint32_t  charCount)
{
    DispStringBuffer::InitializeArg arg;
    arg.SetCharCountMax(charCount);
    return sizeof(DispStringBuffer) + DispStringBuffer::GetRequiredDrawBufferSize(arg);
}

size_t
CharWriter::GetConstantBufferSize(nn::gfx::Device* pDevice, int charCount, bool shadowEnabled)
{
    DispStringBuffer::InitializeArg arg;
    arg.SetCharCountMax(charCount);
    arg.SetShadowEnabled(shadowEnabled);

    return DispStringBuffer::GetRequiredConstantBufferSize(pDevice, arg);
}

DispStringBuffer*
CharWriter::InitializeDispStringBuffer(
    nn::gfx::Device* pDevice,
    void* pDrawBuffer,
    uint32_t  charCount,
    bool shadowEnabled /* = false */
)
{
    NN_SDK_ASSERT_NOT_NULL(pDrawBuffer);

    DispStringBuffer* result = new (pDrawBuffer) DispStringBuffer();

    DispStringBuffer::InitializeArg arg;
    arg.SetCharCountMax(charCount);
    arg.SetDrawBuffer(nn::util::BytePtr(pDrawBuffer).Advance(sizeof(DispStringBuffer)).Get());
    arg.SetShadowEnabled(shadowEnabled);

    result->Initialize(pDevice, arg);
    return result;
}

}   // namespace font
}   // namespace nn
