﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/detail/res/atk_CacheManager.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/fnd/basis/atkfnd_Memory.h>
#include <nn/atk/viewer/detail/res/atk_ResItemInfo.h>
#include <nn/util/util_StringUtil.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_CACHE_DEBUG
//#define NN_ATK_ENABLE_CACHE_DUMP_DEBUG
//#define NN_ATK_ENABLE_CACHE_DUMP_DEBUG_HASHCODE
#endif

namespace nn {
namespace atk {
namespace viewer {
namespace detail {

uint32_t CacheManager::InvalidId = std::numeric_limits<uint32_t>::max();

//----------------------------------------------------------
CacheManager::InitializeArgs::InitializeArgs() NN_NOEXCEPT :
buffer(NULL),
bufferLength(0),
dataBufferAllocator(NULL),
baseId(InvalidId),
maxEntries(0),
maxName(0)
{
}

//----------------------------------------------------------
CacheManager::Item::Item(
    CacheManager& owner,
    uint32_t id,
    ResName& name,
    bool isOverrideOriginal,
    size_t bufferSize,
    size_t dataSize,
    const void* data) NN_NOEXCEPT :
m_Owner(owner),
m_IsFreezed(false),
m_IsDisposed(false),
m_IsOverrideOriginal(isOverrideOriginal),
m_Id(id),
m_ReferenceCount(1),
m_DataType(ResDataType_Unknown),
m_DataSize(dataSize),
m_BufferSize(bufferSize),
m_UserParameter(0),
m_pPlaybackBuffer(nullptr)
{
    NN_SDK_ASSERT_NOT_NULL(data);

    m_Reference.value = this;
    m_UnfreezedReference.value = this;
    m_UnusedReference.value = this;

    m_Name.set_ptr(name.GetName());
    m_DataOffset.set_ptr(data);
    m_HashCode.SetEmpty();
}

//----------------------------------------------------------
void
CacheManager::Item::Freeze() NN_NOEXCEPT
{
    if(m_IsFreezed)
    {
        return;
    }

    m_IsFreezed = true;

    m_Owner.m_UnfreezedItems.erase(m_Owner.m_UnfreezedItems.iterator_to(GetUnfreezedReference()));

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    m_Owner.DumpItems("Item::Freeze");
#endif
}

//----------------------------------------------------------
void
CacheManager::Item::Unfreeze() NN_NOEXCEPT
{
    if(!m_IsFreezed)
    {
        return;
    }

    m_IsFreezed = false;

    m_Owner.m_UnfreezedItems.push_back(GetUnfreezedReference());

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    m_Owner.DumpItems("Item::Unfreeze");
#endif
}

//----------------------------------------------------------
void
CacheManager::Item::IncrementReferenceCount() NN_NOEXCEPT
{
    ++m_ReferenceCount;

    if(m_ReferenceCount == 1)
    {
        m_Owner.m_UnusedItems.erase(m_Owner.m_UnfreezedItems.iterator_to(GetUnusedReference()));
    }
}

//----------------------------------------------------------
void
CacheManager::Item::DecrementReferenceCount() NN_NOEXCEPT
{
    if(m_ReferenceCount == 0)
    {
        NN_SDK_ASSERT(false, "invalid CacheManager::Item reference count.\n");
        return;
    }

    --m_ReferenceCount;

    if(m_ReferenceCount == 0)
    {
        m_Owner.m_UnusedItems.push_back(GetUnusedReference());
    }
}

//----------------------------------------------------------
void
CacheManager::Item::ClearReferenceCount() NN_NOEXCEPT
{
    if(m_ReferenceCount == 0)
    {
        return;
    }

    m_ReferenceCount = 0;
    m_Owner.m_UnusedItems.push_back(GetUnusedReference());
}

//----------------------------------------------------------
void
CacheManager::Item::Dispose() NN_NOEXCEPT
{
    if(m_IsDisposed)
    {
        return;
    }

    if(m_ReferenceCount == 0)
    {
        m_Owner.m_UnusedItems.erase(m_Owner.m_UnusedItems.iterator_to(GetUnusedReference()));
    }

    m_IsDisposed = true;

    m_Owner.m_MemoryUsage -= GetBufferSize();
    m_Owner.m_DataBufferAllocator->Free(this);
    if (m_pPlaybackBuffer != nullptr)
    {
        m_Owner.m_PlaybackBufferAllocator->Free(m_pPlaybackBuffer);
        m_pPlaybackBuffer = nullptr;
    }

    if (!m_IsOverrideOriginal)
    {
        if(m_Owner.m_IdTable != NULL && m_Owner.m_BaseId != InvalidId)
        {
            int index = m_Id - m_Owner.m_BaseId;
            NN_SDK_ASSERT(index < m_Owner.m_MaxItemCount);

            m_Owner.m_IdTable[index] = NULL;
        }
    }
}

//----------------------------------------------------------
CacheManager::CacheManager() NN_NOEXCEPT :
m_DataBufferAllocator(NULL),
m_ItemDictionaryData(NULL),
m_MaxItemCount(0),
m_ValidItemCount(0),
m_MaxName(0),
m_MemoryUsage(0),
m_BaseId(InvalidId),
m_IdTable(NULL)
{
}

//----------------------------------------------------------
Result
CacheManager::Initialize(const InitializeArgs& args) NN_NOEXCEPT
{
    if((args.buffer == NULL && args.bufferLength > 0) ||
        args.bufferLength < GetRequiredMemorySize(args.maxEntries) ||
        args.dataBufferAllocator == NULL ||
        args.playbackBufferAllocator == NULL ||
        args.maxEntries < 0 ||
        args.maxName == 0)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return Result(ResultType_Failed);
    }

    NN_SDK_ASSERT(m_MemoryUsage == 0);

    atk::detail::fnd::FrameHeap heap;
    heap.Initialize(args.buffer, args.bufferLength);

    if(args.buffer == NULL)
    {
        m_ItemDictionaryData = NULL;
        m_IdTable = NULL;
    }
    else
    {
        // IDテーブルを初期化します。
        if(args.baseId < InvalidId)
        {
            m_IdTable = reinterpret_cast<Item**>(heap.Alloc(GetIDTableSize(args.maxEntries)));
            // 新規追加アイテムについては、実用上利用されないと思われる、InitialBaseId から始めます
            if(args.baseId < InitialBaseId)
            {
                m_BaseId = InitialBaseId;
            }
            else
            {
                m_BaseId = args.baseId;
            }

            for(int32_t index = 0; index < args.maxEntries; ++index)
            {
                m_IdTable[index] = NULL;
            }
        }

        // 辞書を初期化します。
        size_t itemDictionaryBufferLength = heap.GetFreeLength();
        void* itemDictionaryBuffer = heap.Alloc(itemDictionaryBufferLength);

        m_ItemDictionaryData = PatriciaDictionary::CreateData(
            itemDictionaryBuffer,
            itemDictionaryBufferLength,
            args.maxEntries);
    }

    m_DataBufferAllocator = args.dataBufferAllocator;
    m_PlaybackBufferAllocator = args.playbackBufferAllocator;
    m_MaxItemCount = args.maxEntries;
    m_ValidItemCount = 0;
    m_MaxName = args.maxName;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
CacheManager::Finalize() NN_NOEXCEPT
{
    RemoveAllItems();
    m_ItemDictionaryData = NULL;
    m_IdTable = NULL;
    m_BaseId = InvalidId;
    m_MemoryUsage = 0;

    m_DataBufferAllocator = NULL;
    m_PlaybackBufferAllocator = nullptr;
    m_MaxItemCount = 0;
    m_ValidItemCount = 0;
    m_MaxName = 0;
}

//----------------------------------------------------------
size_t
CacheManager::GetRequiredMemorySize(int maxEntries) const NN_NOEXCEPT
{
    if(maxEntries == 0)
    {
        return 0;
    }

    return nn::util::align_up(
        PatriciaDictionary::GetRequiredSize(maxEntries) + GetIDTableSize(maxEntries),
        atk::detail::fnd::MemoryTraits::DefaultAlignment);
}

//----------------------------------------------------------
size_t
CacheManager::GetRequiredMemorySizeForItem(int maxName, size_t maxDataLength) const NN_NOEXCEPT
{
    return nn::util::align_up(
        sizeof(Item) + ResName::GetRequiredSize(maxName) + maxDataLength,
        atk::detail::fnd::MemoryTraits::DefaultAlignment );
}

//----------------------------------------------------------
bool
CacheManager::TryGetFreeID(uint32_t* id) const NN_NOEXCEPT
{
    if(m_IdTable == NULL || m_BaseId == InvalidId)
    {
        return false;
    }

    for(int32_t index = 0; index < m_MaxItemCount; ++index)
    {
        if(m_IdTable[index] == NULL)
        {
            if(id != NULL)
            {
                *id = m_BaseId + index;
            }

            return true;
        }
    }

    return false;
}

//----------------------------------------------------------
const void*
CacheManager::GetData(
    const char* name,
    uint32_t* dataType /*= NULL*/,
    bool allowUnfreezedData /*= false*/) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "CacheManager is not initialized.\n");

    if(dataType != NULL)
    {
        *dataType = ResDataType_Unknown;
    }

    Item* item = GetItemImpl(name);

    if(item == NULL ||
        (!allowUnfreezedData && !item->IsFreezed()))
    {
        return NULL;
    }

    uint32_t cachedDataType = item->GetDataType();

    if(dataType != NULL)
    {
        *dataType = cachedDataType;
    }

    return const_cast<const Item*>(item)->GetData();
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetItem(const char* name) NN_NOEXCEPT
{
    return GetItemImpl(name);
}

//----------------------------------------------------------
const CacheManager::Item*
CacheManager::GetItem(const char* name) const NN_NOEXCEPT
{
    return GetItemImpl(name);
}

//----------------------------------------------------------
const CacheManager::Item*
CacheManager::GetItem(int index) const NN_NOEXCEPT
{
    if(index >= m_ValidItemCount)
    {
        return NULL;
    }

    return m_ItemDictionaryData->node[index + 1].ofsData.to_ptr<Item>();
}

//----------------------------------------------------------
const CacheManager::Item*
CacheManager::GetItemFromID(uint32_t id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IdTable != NULL && m_BaseId != InvalidId);

    if(id < m_BaseId)
    {
        return NULL;
    }

    int32_t index = id - m_BaseId;

    if(index < 0 || m_MaxItemCount <= index)
    {
        return NULL;
    }

    return m_IdTable[index];
}

//----------------------------------------------------------
Result
CacheManager::AddItem(
    uint32_t id,
    const char* name,
    bool isOverrideOriginal,
    uint32_t dataType,
    const void* data,
    size_t dataSize,
    bool isFileEntry) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "CacheManager is not initialized.\n");

    if(name == NULL || name[0] == '\0' || data == NULL || dataSize == 0)
    {
        return Result(ResultType_Failed);
    }

    Result result = CanAddItem();

    if(result.IsFailed())
    {
        return result;
    }

    Item* newItem = NULL;
    result = CreateItem(
        newItem,
        id,
        name,
        isOverrideOriginal,
        data,
        dataSize,
        isFileEntry);

    if(result.IsFailed())
    {
        return result;
    }

    NN_SDK_ASSERT_NOT_NULL(newItem);
    newItem->SetDataType(dataType);

    return newItem != NULL ? AddItem(*newItem) : Result(ResultType_Failed);
}

//----------------------------------------------------------
Result
CacheManager::AddItem(Item& item) NN_NOEXCEPT
{
    Result result = CanAddItem();
    if(result.IsFailed())
    {
        return result;
    }

    int32_t index = FindEmptyNodeIndex();

    if(index < 0)
    {
        return Result(ResultType_CacheOverFlow);
    }

    PatriciaDictionaryData::Node& node = m_ItemDictionaryData->node[index];

    node.ofsName.set_ptr(item.GetName().GetName());
    node.ofsData.set_ptr(&item);

    NN_SDK_ASSERT(node.ofsName.to_ptr() != NULL, "invalid name.\n");

    // 有効インデックスを更新します。
    m_ItemDictionaryData->dataCount = std::max(index, m_ItemDictionaryData->dataCount);
    ++m_ValidItemCount;

    // （最も最近使用されたアイテムとして）最後尾に追加します。
    m_Items.push_back(item.GetReference());

    // 非フリーズアイテムは別途リストに追加します。
    if(!item.IsFreezed())
    {
        m_UnfreezedItems.push_back(item.GetUnfreezedReference());
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] CacheManager::AddItem() : %d/%d : index=%3d, Name='%s'\n",
        m_ValidItemCount,
        m_MaxItemCount,
        index,
        node.ofsName.to_ptr()
        );
#endif

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::AddItem");
#endif

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
CacheManager::RemoveItem(const char* name, Action action /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "CacheManager is not initialized.\n");

    if(name == NULL || name[0] == '0')
    {
        return Result(ResultType_Failed);
    }

    PatriciaDictionary* dictionary = GetPatriciaDictionary();

    if(dictionary == NULL)
    {
        return Result(ResultType_Failed);
    }

    PatriciaDictionary::Node* node = dictionary->FindNode(name, std::strlen(name));

    if(node == NULL)
    {
        return Result(ResultType_False);
    }

    RemoveItem(*node, action);
    return Result(ResultType_True);
}

//----------------------------------------------------------
void
CacheManager::RemoveAllItems(Action action /*= NULL*/) NN_NOEXCEPT
{
    if(!IsInitialized())
    {
        return;
    }

    if(m_ItemDictionaryData == NULL)
    {
        return;
    }

    // 辞書を一括リセットします。
    for(int32_t index = 1; index <= m_ItemDictionaryData->dataCount; ++index)
    {
        PatriciaDictionary::ResetNode(m_ItemDictionaryData->node[index]);
    }

    m_ItemDictionaryData->dataCount = 1;

    // アイテムデータを削除します。
    for(Item::ReferenceList::iterator it = m_Items.begin(); it != m_Items.end(); ++it)
    {
        NN_SDK_ASSERT_NOT_NULL(it->value);

        // 辞書は一括リセットしたので、DisposeItem() します。
        DisposeItem(*(it->value), action);
    }

    m_Items.clear();
    m_UnfreezedItems.clear();

    m_ValidItemCount = 0;

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] CacheManager::RemoveAllItems() : %d/%d\n",
        m_ValidItemCount,
        m_MaxItemCount
        );
#endif

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::RemoveAllItems");
#endif
}

//----------------------------------------------------------
void
CacheManager::RemoveGarbages() NN_NOEXCEPT
{
    while(!m_UnusedItems.empty())
    {
        NN_SDK_ASSERT_NOT_NULL(m_UnusedItems.begin()->value);
        RemoveItem(m_UnusedItems.begin()->value->GetName().GetName());
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] CacheManager::RemoveGarbages() : %d/%d\n",
        m_ValidItemCount,
        m_MaxItemCount
        );
#endif
}

//----------------------------------------------------------
void
CacheManager::ClearAllItemReferenceCounts() NN_NOEXCEPT
{
    for(Item::ReferenceList::const_iterator it = Item::ReferenceList::const_iterator(m_Items.begin()); it != Item::ReferenceList::const_iterator(m_Items.end()); ++it)
    {
        NN_SDK_ASSERT_NOT_NULL(it->value);
        it->value->ClearReferenceCount();
    }
}

//----------------------------------------------------------
void
CacheManager::UpdateDictionary() NN_NOEXCEPT
{
    if(GetPatriciaDictionary() == NULL)
    {
        return;
    }

    GetPatriciaDictionary()->Build();

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::UpdateDictionary");
#endif
}

//----------------------------------------------------------
Result
CacheManager::RebuildDictionary(void* buffer, size_t bufferLength, int maxEntries, void** oldBuffer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT(IsInitialized(), "CacheManager is not initialized.\n");

    int32_t currentDataCount = 0;

    if(m_ItemDictionaryData != NULL)
    {
        currentDataCount = m_ItemDictionaryData->dataCount;
    }

    // 削除後の欠番が考えられるので、
    // 有効アイテムのインデックスまでの領域を保持する必要があります。
    NN_SDK_ASSERT(maxEntries >= currentDataCount);

    // 現在、利用可能なバッファ長
    size_t currentUsableLength = GetRequiredMemorySize(currentDataCount);

    if(currentUsableLength > bufferLength)
    {
        return Result(ResultType_OutOfMemory);
    }

    PatriciaDictionaryData* newData =
        PatriciaDictionary::CreateData(buffer, bufferLength, maxEntries);
    NN_SDK_ASSERT_NOT_NULL(newData);

    if(m_ItemDictionaryData != NULL)
    {
        newData->size = m_ItemDictionaryData->size;
        newData->dataCount = m_ItemDictionaryData->dataCount;

        for(auto index = 0; index <= m_ItemDictionaryData->dataCount; ++index)
        {
            PatriciaDictionaryData::Node& srcNode = m_ItemDictionaryData->node[index];
            PatriciaDictionaryData::Node& destNode = newData->node[index];

            destNode.refBit = srcNode.refBit;
            destNode.idxLeft = srcNode.idxLeft;
            destNode.idxRight = srcNode.idxRight;

            destNode.ofsName.set_ptr(srcNode.ofsName.to_ptr());
            NN_SDK_ASSERT(srcNode.ofsName.to_ptr() == destNode.ofsName.to_ptr(), "failed to offset.\n");

#if !defined(NN_SDK_BUILD_RELEASE)
            if(index == 0)
            {
                NN_SDK_ASSERT(srcNode.ofsName.to_ptr() == NULL, "failed to offset.\n");
                NN_SDK_ASSERT(destNode.ofsName.to_ptr() == NULL, "failed to offset.\n");
            }
            else
            {
                NN_SDK_ASSERT(srcNode.ofsName.to_ptr() != NULL, "failed to offset.\n");
                NN_SDK_ASSERT(destNode.ofsName.to_ptr() != NULL, "failed to offset.\n");
            }
#endif

            destNode.ofsData.set_ptr(srcNode.ofsData.to_ptr());
            NN_SDK_ASSERT(srcNode.ofsData.to_ptr() == destNode.ofsData.to_ptr(), "failed to offset.\n");
        }
    }

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] CacheManager::RebuildDictionary() : %d/%d : OldMaxItemCount=%d\n",
        m_ValidItemCount,
        maxEntries,
        m_MaxItemCount
        );
#endif

    if(oldBuffer != NULL)
    {
        *oldBuffer = m_ItemDictionaryData;
    }

    m_ItemDictionaryData = newData;
    m_MaxItemCount = maxEntries;

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::RemoveItem");
#endif

    return Result(ResultType_True);
}

//----------------------------------------------------------
Result
CacheManager::CreateItem(
    Item*& item,
    uint32_t id,
    const char* name,
    bool isOverrideOriginal,
    const void* data,
    size_t dataSize,
    bool isFileEntry) NN_NOEXCEPT
{
    if(m_ItemDictionaryData == NULL)
    {
        NN_SDK_ASSERT(false, "CacheManager is not initialized.\n");
        return Result(ResultType_NotInitialized);
    }

    // アイテムバッファを確保します。
    // バッファの内訳は以下の通りです。
    //  - Item
    //  - ResNameData (可変長)
    //  - アイテムデータ (可変長)
    int nameLength = static_cast<int>(std::strlen(name));

    if(nameLength >= m_MaxName)
    {
        return Result(ResultType_NameTooLong);
    }

    size_t resNameLength = ResName::GetRequiredSize(nameLength);
    size_t itemBufferSize = sizeof(Item) + resNameLength + dataSize;

    void* itemBuffer = m_DataBufferAllocator->Alloc(itemBufferSize);
    if(itemBuffer == NULL)
    {
        return Result(ResultType_OutOfMemory);
    }

    // 名前をコピーします。
    ResNameData* resNameData = reinterpret_cast<ResNameData*>(
        util::BytePtr(itemBuffer, sizeof(Item)).Get());

    resNameData->len = nameLength;
    util::Strlcpy(resNameData->str, name, nameLength + 1);

    // データをコピーします。
    // HACK : ※Item の直後に data が参照する ResName 等の実体が続いていることが前提です。
    void* resData = util::BytePtr(resNameData, resNameLength).Get();

    if(data != NULL)
    {
        std::memcpy(resData, data, dataSize);
    }

    // Item（名前とデータのコンテナ）を生成します。
    item = new(itemBuffer) Item(
        *this,
        id,
        *reinterpret_cast<ResName*>(resNameData),
        isOverrideOriginal,
        itemBufferSize,
        dataSize,
        resData);

    m_MemoryUsage += item->GetBufferSize();

    if (isFileEntry)
    {
        void* pPlaybackBuffer = m_PlaybackBufferAllocator->Alloc(itemBufferSize, Alignments::WaveBuffer);
        if (pPlaybackBuffer == nullptr)
        {
            return Result(ResultType_OutOfMemory);
        }
        item->SetPlaybackBuffer(pPlaybackBuffer);
    }

    if (!isOverrideOriginal)
    {
        if(m_IdTable != NULL && m_BaseId != InvalidId)
        {
            int32_t index = id - m_BaseId;
            NN_SDK_ASSERT(index < m_MaxItemCount);

            m_IdTable[index] = item;
        }
    }

    return Result(ResultType_True);
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetFirstUnfreezedItem(FilterFunc filter /*= NULL*/) const NN_NOEXCEPT
{
    if(m_UnfreezedItems.empty())
    {
        return NULL;
    }

    if(filter == NULL)
    {
        NN_SDK_ASSERT_NOT_NULL(m_UnfreezedItems.rbegin()->value);
        return m_UnfreezedItems.rbegin()->value;
    }

    for(Item::ReferenceList::const_reverse_iterator it = m_UnfreezedItems.rbegin(); it != m_UnfreezedItems.rend(); ++it)
    {
        Item* item = it->value;
        NN_SDK_ASSERT_NOT_NULL(item);

        if(filter(*item))
        {
            return item;
        }
    }

    return NULL;
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetFirstUnusedItem(FilterFunc filter /*= NULL*/) const NN_NOEXCEPT
{
    if(m_UnusedItems.empty())
    {
        return NULL;
    }

    if(filter == NULL)
    {
        return m_UnusedItems.begin()->value;
    }

    for(Item::ReferenceList::const_iterator it = m_UnusedItems.begin(); it != m_UnusedItems.end(); ++it)
    {
        Item* item = it->value;
        NN_SDK_ASSERT_NOT_NULL(item);

        if(filter(*item))
        {
            return item;
        }
    }

    return NULL;
}

//----------------------------------------------------------
Result
CacheManager::CanAddItem() const NN_NOEXCEPT
{
    if(m_ItemDictionaryData == NULL)
    {
        return Result(ResultType_CacheOverFlow);
    }

    return m_ValidItemCount <= m_MaxItemCount ?
        Result(ResultType_True) : Result(ResultType_CacheOverFlow);
}

//----------------------------------------------------------
int32_t
CacheManager::FindEmptyNodeIndex() NN_NOEXCEPT
{
    if(m_ItemDictionaryData == NULL)
    {
        return -1;
    }

    for(int32_t index = 1; index <= m_MaxItemCount; ++index)
    {
        if( m_ItemDictionaryData->node[index].ofsName.to_ptr() == NULL)
        {
            return index;
        }
    }

    return -1;
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetItemImpl(const char* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "CacheManager is not initialized.\n");

    if(name == NULL || name[0] == '\0')
    {
        return NULL;
    }

    if(m_ItemDictionaryData == NULL)
    {
        return NULL;
    }

    PatriciaDictionary* dictionary = GetPatriciaDictionary();

    if(dictionary == NULL)
    {
        return NULL;
    }

    Item* result = reinterpret_cast<Item*>(dictionary->Find(name));

    if(result == NULL)
    {
        return NULL;
    }

    return result;
}

//----------------------------------------------------------
void
CacheManager::RemoveItem(PatriciaDictionary::Node& node, Action action /*= NULL*/) NN_NOEXCEPT
{
    Item* item = node.ofsData.to_ptr<Item>();

    if(item == NULL)
    {
        NN_SDK_ASSERT(false, "unexpected error.\n");
        return;
    }

    --m_ValidItemCount;
    PatriciaDictionary::ResetNode(node);

    // 正しく索引できない可能性があるので、ノードを削除したら必ず更新します。
    UpdateDictionary();

    if(!item->IsFreezed())
    {
        m_UnfreezedItems.erase(m_UnfreezedItems.iterator_to(item->GetUnfreezedReference()));
    }

    m_Items.erase(m_Items.iterator_to(item->GetReference()));

#if defined(NN_ATK_ENABLE_CACHE_DEBUG)
    NN_DETAIL_ATK_INFO(
        "[sndedit] CacheManager::RemoveItem() : %d/%d : Name='%s'\n",
        m_ValidItemCount,
        m_MaxItemCount,
        item->GetName().GetName()
        );
#endif

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::RemoveItem");
#endif

    DisposeItem(*item, action);
}

//----------------------------------------------------------
void
CacheManager::DisposeItem(Item& item, Action action /*= NULL*/) NN_NOEXCEPT
{
    if(action != NULL)
    {
        action(item);
    }

    // HACK : Dispose() 内部で解放され、m_MemoryUsage から使用メモリ量が差し引かれます。
    item.Dispose();
}

//----------------------------------------------------------
void
CacheManager::DumpItems(const char* tag) const NN_NOEXCEPT
{
    (void)tag;

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG)
    NN_DETAIL_ATK_INFO("[%s] ----------------\n", tag);

    NN_DETAIL_ATK_INFO("Items:\n");
    NN_DETAIL_ATK_INFO("   0 : Left=%3d, Right=%3d\n", m_ItemDictionaryData->node[0].idxLeft, m_ItemDictionaryData->node[0].idxRight);

    static const uint32_t hashStringBufferLength = Hash32::Size * 4 + 1;
    char hashString[hashStringBufferLength];

    for(int32_t index = 1; index <= m_MaxItemCount; ++index)
    {
        PatriciaDictionaryData::Node& node = m_ItemDictionaryData->node[index];

        if( node.ofsName.to_ptr() == NULL)
        {
            continue;
        }

        Item* item = node.ofsData.to_ptr<Item>();
        NN_SDK_ASSERT_NOT_NULL(item);

#if defined(NN_ATK_ENABLE_CACHE_DUMP_DEBUG_HASHCODE)
        item->GetHashCode().ToString(hashString, hashStringBufferLength);
#else
        hashString[0] = '\0';
#endif

        NN_DETAIL_ATK_INFO(
            " %3d : Left=%3d, Right=%3d, HashCode=%s, RefCount=%3d, Key=%s\n",
            index,
            node.idxLeft,
            node.idxRight,
            hashString,
            item->GetReferenceCount(),
            node.ofsName.to_ptr()
            );
    }

    NN_DETAIL_ATK_INFO("UnfreezedItems:\n");

    uint32_t index = 0;

    for(Item::ReferenceList::const_iterator it = m_UnfreezedItems.begin(); it != m_UnfreezedItems.end(); ++it)
    {
        NN_DETAIL_ATK_INFO(" %3d : %s\n", index, it->value->GetName().GetName());
        ++index;
    }

    NN_DETAIL_ATK_INFO("-------------------------------\n");
#endif
}

} // namespace nn::atk::viewer::detail
} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
