﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/atk_UpdateCachesHelper.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

namespace nn {
namespace atk {
namespace viewer {

NN_DEFINE_STATIC_CONSTANT( const size_t UpdateCachesHelper::DefaultThreadStackSize );

//----------------------------------------------------------
UpdateCachesHelper::UpdateCachesHelper() NN_NOEXCEPT :
m_SoundArchiveEditor(NULL),
m_IsExitThread(false)
{
    m_ThreadMainDelegate.Initialize(this, &UpdateCachesHelper::ThreadMain);
}

//----------------------------------------------------------
Result
UpdateCachesHelper::Start(
    SoundArchiveEditor* soundArchiveEditor,
    void* threadStack,
    size_t threadStackSize,
    AffinityMask affinityMask /*= AffinityMask_CoreDefault*/,
    uint32_t priority /*= atk::detail::fnd::Thread::DEFAULT_THREAD_PRIORITY*/,
    int idealCoreNumber /*= 0*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(soundArchiveEditor);
    NN_SDK_ASSERT_NOT_NULL(threadStack);
    NN_SDK_ASSERT(threadStackSize > 0);

    if(m_Thread.GetState() == atk::detail::fnd::Thread::State_Running)
    {
        return Result(ResultType_False);
    }

    m_IsExitThread = false;

    m_SoundArchiveEditor = soundArchiveEditor;
    m_UpdateCachesCanceller.Reset(NULL);

    atk::detail::fnd::Thread::RunArgs args;
    args.name = "nn::atk::viewer::UpdateChachesHelper::Thread";
    args.stack = threadStack;
    args.stackSize = threadStackSize;
    args.idealCoreNumber = idealCoreNumber;
    args.affinityMask = static_cast<atk::detail::fnd::Thread::AffinityMask>(affinityMask);
    args.priority = priority;
    args.handler = &m_ThreadMainDelegate;

    if(!m_Thread.Run(args))
    {
        Stop();
        return Result(ResultType_Failed);
    }

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
UpdateCachesHelper::Stop() NN_NOEXCEPT
{
    m_IsExitThread = true;
    m_UpdateCachesCanceller.Cancel();

    m_Thread.WaitForExit();
    m_Thread.Release();

    m_SoundArchiveEditor = NULL;
}

#if defined(NN_ATK_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION)
//----------------------------------------------------------
int
UpdateCachesHelper::GetThreadCoreNumber() const NN_NOEXCEPT
{
    return m_Thread.GetCoreNumber();
}
#endif

//----------------------------------------------------------
uint32_t
UpdateCachesHelper::ThreadMain(void* parameter) NN_NOEXCEPT
{
    (void)parameter;
    while(!m_IsExitThread)
    {
        m_SoundArchiveEditor->UpdateCaches(&m_UpdateCachesCanceller);

        atk::detail::fnd::Thread::Sleep(
            atk::detail::fnd::TimeSpan::FromMilliSeconds(200)
            );
    }

    return 0;
}

} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
