﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/viewer/atk_SoundArchiveEditor.h>

#ifdef NN_ATK_CONFIG_ENABLE_DEV

#include <nn/atk/fnd/os/atkfnd_CriticalSection.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>
#include <nn/atk/viewer/detail/atk_SoundEditConnection.h>
#include <nn/atk/viewer/detail/res/atk_CacheManager.h>
#include <nn/atk/viewer/detail/res/atk_ResItemInfo.h>

#if !defined(NN_SDK_BUILD_RELEASE)
//#define NN_ATK_ENABLE_ERROR_DEBUG
#endif

namespace nn {
namespace atk {
namespace viewer {

NN_DEFINE_STATIC_CONSTANT( const int SoundArchiveEditor::DefaultEditableItemCountMax );
NN_DEFINE_STATIC_CONSTANT( const int SoundArchiveEditor::DefaultEditableFileCountMax );
NN_DEFINE_STATIC_CONSTANT( const int SoundArchiveEditor::DefaultItemNameLengthMax );

//----------------------------------------------------------
SoundArchiveEditor::SoundArchiveEditor() NN_NOEXCEPT :
m_MaxItemName(0),
m_TotalBufferLength(0),
m_EditConnection(NULL)
{
}

//----------------------------------------------------------
SoundArchiveEditor::~SoundArchiveEditor() NN_NOEXCEPT
{
    // Finalize() 済みであることを確認します。
    NN_SDK_ASSERT(!IsInitialized(), "SoundArchiveEditor is not finalized.\n");
}

//----------------------------------------------------------
Result
SoundArchiveEditor::Initialize(
    void* buffer,
    size_t bufferLength,
    SoundArchive* soundArchive,
    SoundArchivePlayer* soundArchivePlayer,
    const Configs& configs) NN_NOEXCEPT
{
    if(buffer == NULL ||
        bufferLength == 0 ||
        soundArchive == NULL ||
        soundArchivePlayer == NULL ||
        (configs.maxEditableItems == 0 && !configs.isEditableItemsCountMaxExpandable) ||
        (configs.maxEditableFiles == 0 && !configs.isEditableFilesCountMaxExpandable) ||
        configs.maxItemName == 0)
    {
        NN_SDK_ASSERT(false, "invalid arguments.\n");
        return Result(ResultType_Failed);
    }

    Result result = Result(ResultType_True);

    atk::detail::fnd::FrameHeap frameHeap;
    frameHeap.Initialize(buffer, bufferLength);

    m_TotalBufferLength = frameHeap.GetFreeLength();

    {
        size_t editControllerBufferSize = m_EditController.GetRequiredMemorySize(configs.maxItemName);
        void* editControllerBuffer = frameHeap.Alloc(editControllerBufferSize);

        if(editControllerBuffer == NULL)
        {
            SetLastError(ResultType_OutOfMemory);
            Finalize();
            return Result(ResultType_OutOfMemory);
        }

        detail::SoundArchiveEditController::InitializeArgs args;
        args.errorProvider = this;
        args.buffer = editControllerBuffer;
        args.bufferLength = editControllerBufferSize;
        args.maxItemName = configs.maxItemName;
        args.resourceManager = &m_ResourceManager;
        args.soundArchive = soundArchive;
        args.soundArchivePlayer = soundArchivePlayer;

        result = m_EditController.Initialize(args);

        if(!result.IsTrue())
        {
            SetLastError(result);
            Finalize();
            return result;
        }
    }

    int soundCount = 0;
    if (soundArchive->IsAvailable())
    {
        soundCount = soundArchive->GetSoundCount();
    }

    result = m_ResourceManager.Initialize(
        frameHeap,
        soundArchive->GetSoundIdFromIndex(soundCount),
        configs.maxEditableItems,
        configs.maxEditableFiles,
        configs.maxItemName,
        configs.isEditableItemsCountMaxExpandable,
        configs.isEditableFilesCountMaxExpandable);

    if(result.IsFailed())
    {
        SetLastError(result);
        Finalize();
        return result;
    }

    result = m_ParameterEditor.Initialize(&m_EditController, &m_ResourceManager, &soundArchivePlayer->GetSoundArchiveManager());

    if(!result.IsTrue())
    {
        SetLastError(result);
        Finalize();
        return result;
    }

    result = m_FileEditor.Initialize(&m_EditController, soundArchivePlayer);

    if(!result.IsTrue())
    {
        SetLastError(result);
        Finalize();
        return result;
    }

    m_MaxItemName = configs.maxItemName;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundArchiveEditor::Finalize() NN_NOEXCEPT
{
    detail_Stop();

    m_ParameterEditor.Finalize();
    m_FileEditor.Finalize();

    m_EditController.Finalize();
    m_ResourceManager.Finalize();

    m_MaxItemName = 0;
    m_TotalBufferLength = 0;
}

//----------------------------------------------------------
size_t
SoundArchiveEditor::GetMemoryUsage() const NN_NOEXCEPT
{
    return m_EditController.GetTotalMemoryUsage();
}

//----------------------------------------------------------
size_t
SoundArchiveEditor::GetMemoryMax() const NN_NOEXCEPT
{
    return m_TotalBufferLength;
}

//----------------------------------------------------------
Result
SoundArchiveEditor::detail_Start(detail::SoundEditConnection& connection) NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_ConnectionLockObject);

    detail_Stop();

    if(connection.IsOpened())
    {
        return Result(ResultType_False);
    }

    // インゲーム編集の準備を行います。
    if(m_ParameterEditor.IsInitialized())
    {
        m_ParameterEditor.Start();
    }

    if(m_FileEditor.IsInitialized())
    {
        m_FileEditor.Start();
    }

    Result result = m_EditController.Start(&connection);

    if(!result.IsTrue())
    {
        detail_Stop();
        return result;
    }

    m_EditConnection = &connection;

    return Result(ResultType_True);
}

//----------------------------------------------------------
void
SoundArchiveEditor::detail_Stop() NN_NOEXCEPT
{
    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_ConnectionLockObject);

    m_EditConnection = NULL;

    if(m_ParameterEditor.IsInitialized())
    {
        m_ParameterEditor.Stop();
    }

    if(m_FileEditor.IsInitialized())
    {
        m_FileEditor.Stop();
    }

    m_EditController.Stop();
}

//----------------------------------------------------------
Result
SoundArchiveEditor::UpdateCaches(ICanceller* canceller /*= NULL*/, void* cancellerParam /*= NULL*/) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditor is not initialized.\n");

    nn::atk::detail::fnd::ScopedLock<nn::atk::detail::fnd::CriticalSection> lock(m_ConnectionLockObject);

    if(!IsStarted())
    {
        return Result(ResultType_False);
    }

    return m_EditController.UpdateCaches(canceller, cancellerParam);
}

//----------------------------------------------------------
void
SoundArchiveEditor::detail_UpdateAllItemCaches() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditor is not started.\n");

    m_EditController.InvokeUpdateAllItemInfos();
}

//----------------------------------------------------------
Result
SoundArchiveEditor::detail_UpdateItemCache(const char* name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    return m_EditController.MakeItemInfoEditable(name);
}

//----------------------------------------------------------
void
SoundArchiveEditor::detail_RemoveAllItemCaches() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditor is not started.\n");

    m_EditController.InvokeRemoveAllItemInfos();
}

//----------------------------------------------------------
Result
SoundArchiveEditor::detail_RemoveItemCache(const char* name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    NN_SDK_ASSERT(IsStarted(), "SoundArchiveEditor is not started.\n");

    return m_EditController.InvokeRemoveItemInfo(name);
}

//----------------------------------------------------------
CacheState
SoundArchiveEditor::GetItemCacheState(const char* name) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized(), "SoundArchiveEditor is not initialized.\n");

    if(!IsStarted())
    {
        return CacheState_None;
    }

    return m_EditController.GetItemCacheState(name);
}

//----------------------------------------------------------
void
SoundArchiveEditor::SetLastError(ResultType value) NN_NOEXCEPT
{
    m_LastError = Result(value);

#if defined(NN_ATK_ENABLE_ERROR_DEBUG)
    NN_DETAIL_ATK_INFO("[sndedit] SoundArchiveEditor::SetLastError '%s'.\n", Result(value).ToString());
#endif
}

} // namespace nn::atk::viewer
} // namespace nn::atk
} // namespace nn

#endif // NN_ATK_CONFIG_ENABLE_DEV
