﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_FinalMix.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

#include <nn/atk/fnd/basis/atkfnd_WorkBufferAllocator.h>

namespace
{
    //  FinalMix は bus を 1 つしか持ちません。
    const int FinalMixBusCount = 1;
    const int FinalMixBusIndex = 0;
}

namespace nn {
namespace atk {


size_t FinalMix::GetRequiredMemorySize(bool isEffectEnabled) NN_NOEXCEPT
{
    return OutputMixer::GetRequiredMemorySize( FinalMixBusCount, isEffectEnabled );
}
bool FinalMix::Initialize(nn::audio::AudioRendererConfig* pConfig, int channelCount, bool isEffectEnabled, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pConfig );
    NN_SDK_ASSERT_MINMAX( channelCount, 1, nn::atk::ChannelIndex_Count );
    NN_SDK_ASSERT_ALIGNED( buffer, OutputMixer::RequiredAlignment );

    detail::fnd::WorkBufferAllocator allocator(buffer, bufferSize);
    size_t requiredSize = OutputMixer::GetRequiredMemorySize(FinalMixBusCount, isEffectEnabled);
    void* mixerBuffer = allocator.Allocate(requiredSize, OutputMixer::RequiredAlignment);
    OutputMixer::Initialize( FinalMixBusCount, isEffectEnabled, mixerBuffer, OutputMixer::GetRequiredMemorySize(FinalMixBusCount, isEffectEnabled) );

    m_ChannelCount = channelCount;
    m_ReferenceCount = 0;
    return nn::audio::AcquireFinalMix( pConfig, &m_FinalMix, GetBusCount() * GetChannelCount() );
}
void FinalMix::Finalize(nn::audio::AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pConfig );
    NN_SDK_ASSERT_EQUAL( static_cast<uint32_t>( m_ReferenceCount ), 0u );   //  FinalMix に接続されている Voice がないことを確認します。

    nn::audio::ReleaseFinalMix( pConfig, &m_FinalMix );

    OutputMixer::Finalize();
}
bool FinalMix::AppendEffect(EffectBase* pEffect, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::AppendEffect( pEffect, FinalMixBusIndex, buffer, bufferSize );
}
bool FinalMix::AppendEffect(EffectAux* pEffect, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::AppendEffect( pEffect, FinalMixBusIndex, buffer, bufferSize );
}
bool FinalMix::RemoveEffect(EffectBase* pEffect) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::RemoveEffect( pEffect, FinalMixBusIndex );
}
bool FinalMix::RemoveEffect(EffectAux* pEffect) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::RemoveEffect( pEffect, FinalMixBusIndex );
}
bool FinalMix::ClearEffect() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsEffectEnabled() );
    return OutputMixer::ClearEffect( FinalMixBusIndex );
}
bool FinalMix::IsEffectEnabled() const NN_NOEXCEPT
{
    return OutputMixer::IsEffectEnabled();
}
OutputReceiver::ReceiverType FinalMix::GetReceiverType() const NN_NOEXCEPT
{
    return ReceiverType::ReceiverType_FinalMix;
}
int FinalMix::GetChannelCount() const NN_NOEXCEPT
{
    return m_ChannelCount;
}
int FinalMix::GetBusCount() const NN_NOEXCEPT
{
    return FinalMixBusCount;
}
void FinalMix::AddReferenceCount(int value) NN_NOEXCEPT
{
    m_ReferenceCount += value;
}
bool FinalMix::IsSoundSendClampEnabled(int bus) const NN_NOEXCEPT
{
    NN_UNUSED(bus);
    return true;
}

} // namespace nn::atk
} // namespace nn
