﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/io/atkfnd_SimpleRingBuffer.h>

#include <new>
#include <cstring>
#include <nn/nn_SdkAssert.h>

#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

/* ========================================================================
    static
   ======================================================================== */

namespace
{

using namespace nn::atk::detail::fnd;

const int BufferAlignment = 8;

inline size_t
GetMin(
    const size_t a,
    const size_t b
) NN_NOEXCEPT
{
    return a < b ? a: b;
}

size_t
GetReadableBytes(
    const SimpleRingBufferRange&  range,
    size_t                           bufSize
) NN_NOEXCEPT
{
    if (range.start <= range.end)
    {
        return range.end - range.start;
    }
    else
    {
        return range.end + bufSize - range.start;
    }
}

size_t
GetSequenceReadableBytes(
    const SimpleRingBufferRange&  range,
    size_t                           bufSize
) NN_NOEXCEPT
{
    if (range.start <= range.end)
    {
        return range.end - range.start;
    }
    else
    {
        return bufSize - range.start;
    }
}

size_t
GetWritableBytes(
    const SimpleRingBufferRange&  range,
    size_t                           bufSize
) NN_NOEXCEPT
{
    uintptr_t start = (range.start + bufSize - BufferAlignment) % bufSize;

    if (start >= range.end)
    {
        return start - range.end;
    }
    else
    {
        return start + bufSize - range.end;
    }
}

size_t
GetSequenceWritableBytes(
    const SimpleRingBufferRange&  range,
    size_t                           bufSize
) NN_NOEXCEPT
{
    uintptr_t start = (range.start + bufSize - BufferAlignment) % bufSize;

    if (start >= range.end)
    {
        return start - range.end;
    }
    else
    {
        return bufSize - range.end;
    }
}

}   // namespace


/* ========================================================================
    外部
   ======================================================================== */

namespace nn {
namespace atk {
namespace detail {
namespace fnd {

//---------------------------------------------------------------------------
void
SimpleRingBuffer::Init(
    void*   buf,
    size_t     size
) NN_NOEXCEPT
{
    uintptr_t bufStart = reinterpret_cast<uintptr_t>(buf);
    uintptr_t bufEnd = bufStart + size;

    bufStart = RoundUp(bufStart, BufferAlignment);
    bufEnd = RoundDown(bufEnd, BufferAlignment );

    NN_SDK_ASSERT(bufStart < bufEnd);
    NN_SDK_ASSERT(bufEnd - bufStart > sizeof(SimpleRingBufferHeader) + BufferAlignment);

    m_pHeader = new( reinterpret_cast<void*>(bufStart) ) SimpleRingBufferHeader();
    m_pHeader->bufferSize = bufEnd - bufStart - sizeof(SimpleRingBufferHeader);

    m_pHeader->range.start = 0;
    m_pHeader->range.end   = 0;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::Discard() NN_NOEXCEPT
{
    ScopedLock<LockObject> lock(m_pHeader->lockObject);

    m_pHeader->range.start = 0;
    m_pHeader->range.end   = 0;
}


//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::GetReadableBytes() NN_NOEXCEPT
{
    SimpleRingBufferRange range;

    {
        ScopedLock<LockObject> lock(m_pHeader->lockObject);

        range = m_pHeader->range;
    }

    return ::GetReadableBytes(range, m_pHeader->bufferSize);
}


//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::GetWritableBytes() NN_NOEXCEPT
{
    SimpleRingBufferRange range;

    {
        ScopedLock<LockObject> lock(m_pHeader->lockObject);

        range = m_pHeader->range;
    }

    return ::GetWritableBytes(range, m_pHeader->bufferSize);
}

//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::Read(
    void*       buf,
    size_t         size
) NN_NOEXCEPT
{
    SimpleRingBufferRange range;

    BeginRead(&range);

    size_t ret = ContinueRead(&range, buf, size);

    EndRead(range);

    return ret;
}

//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::Write(
    const void*     buf,
    size_t             size
) NN_NOEXCEPT
{
    SimpleRingBufferRange range;

    BeginWrite(&range);

    size_t ret = ContinueWrite(&range, buf, size);

    EndWrite(range);

    return ret;
}

//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::Skip(
    size_t         size
) NN_NOEXCEPT
{
    SimpleRingBufferRange range;

    BeginSkip(&range);

    size_t ret = ContinueSkip(&range, size);

    EndSkip(range);

    return ret;
}

//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::Peek(
    void*       buf,
    size_t         size
) NN_NOEXCEPT
{
    SimpleRingBufferRange range;

    range = m_pHeader->range;
    size_t ret = ContinueRead(&range, buf, size);

    return ret;
}

//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginRead(SimpleRingBufferRange* pRange) NN_NOEXCEPT
{
    // データ開始・終了位置を取得
    ScopedLock<LockObject> lock(m_pHeader->lockObject);

    *pRange = m_pHeader->range;
}


//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::ContinueRead(
    SimpleRingBufferRange*    pRange,
    void*                     buf,
    size_t                       size
) NN_NOEXCEPT
{
    size_t readableBytes = ::GetReadableBytes(*pRange, m_pHeader->bufferSize);
    if (readableBytes == 0)
    {
        return 0;
    }

    uint8_t* pDst = static_cast<uint8_t*>(buf);
    const uint8_t *const pSrc = reinterpret_cast<uint8_t*>(m_pHeader) + sizeof(SimpleRingBufferHeader);

    size_t restSize = GetMin(readableBytes, size);

    size_t ret = restSize;

    while (restSize > 0)
    {
        const size_t readByte = GetMin(GetSequenceReadableBytes(*pRange, m_pHeader->bufferSize), restSize);
        (void)std::memcpy(pDst, pSrc + pRange->start, readByte);
        pRange->start = (pRange->start + readByte) % m_pHeader->bufferSize;
        pDst += readByte;
        restSize -= readByte;
    }

    return ret;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::EndRead(const SimpleRingBufferRange& range) NN_NOEXCEPT
{
    // 読み込み開始位置を更新
    ScopedLock<LockObject> lock(m_pHeader->lockObject);

    m_pHeader->range.start = range.start;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginWrite(SimpleRingBufferRange* pRange) NN_NOEXCEPT
{
    // データ開始・終了位置を取得
    ScopedLock<LockObject> lock(m_pHeader->lockObject);

    *pRange = m_pHeader->range;
}

/*
    end ポイントが、buf + bufferSize でないようにすること。このときは、buf を指すようにする。

    書き込むときは、end は start と同じ値にならないようにすること。常に end は常に start に対して 4 バイト以上空きを取ることになる。
*/
size_t
SimpleRingBuffer::ContinueWrite(
    SimpleRingBufferRange*    pRange,
    const void*               buf,
    size_t                       size
) NN_NOEXCEPT
{
    size_t writableBytes = ::GetWritableBytes(*pRange, m_pHeader->bufferSize);
    if (writableBytes == 0)
    {
        return 0;
    }

    uint8_t *const pDst = reinterpret_cast<uint8_t*>(m_pHeader) + sizeof(SimpleRingBufferHeader);
    const uint8_t* pSrc = static_cast<const uint8_t*>(buf);

    size_t restSize = GetMin(writableBytes, size);

    size_t ret = restSize;

    while (restSize > 0)
    {
        const size_t writeByte = GetMin(GetSequenceWritableBytes(*pRange, m_pHeader->bufferSize), restSize);
        (void)std::memcpy(pDst + pRange->end, pSrc, writeByte);
        pRange->end = (pRange->end + writeByte) % m_pHeader->bufferSize;
        pSrc += writeByte;
        restSize -= writeByte;
    }

    return ret;
}

//---------------------------------------------------------------------------
void
SimpleRingBuffer::EndWrite(const SimpleRingBufferRange& range) NN_NOEXCEPT
{
    // 書き込み開始位置を更新
    ScopedLock<LockObject> lock(m_pHeader->lockObject);

    m_pHeader->range.end = range.end;
}

//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginSkip(SimpleRingBufferRange* pRange) NN_NOEXCEPT
{
    // データ開始・終了位置を取得
    ScopedLock<LockObject> lock(m_pHeader->lockObject);

    *pRange = m_pHeader->range;
}


//---------------------------------------------------------------------------
size_t
SimpleRingBuffer::ContinueSkip(
    SimpleRingBufferRange*    pRange,
    size_t                       size
) NN_NOEXCEPT
{
    size_t readableBytes = ::GetReadableBytes(*pRange, m_pHeader->bufferSize);
    if (readableBytes == 0)
    {
        return 0;
    }

    size_t restSize = GetMin(readableBytes, size);
    size_t ret = restSize;

    while (restSize > 0)
    {
        const size_t readByte = GetMin(GetSequenceReadableBytes(*pRange, m_pHeader->bufferSize), restSize);
        pRange->start = (pRange->start + readByte) % m_pHeader->bufferSize;
        restSize -= readByte;
    }

    return ret;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::EndSkip(const SimpleRingBufferRange& range) NN_NOEXCEPT
{
    // 読み込み開始位置を更新
    ScopedLock<LockObject> lock(m_pHeader->lockObject);

    m_pHeader->range.start = range.start;
}

}   // namespace fnd
}   // namespace detail
}   // namespace atk
}   // namespace nn
