﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_EffectDelay.h>
#include <nn/atk/fnd/basis/atkfnd_Inlines.h>
#include <nn/atk/atk_HardwareManager.h>
#include <nn/atk/detail/atk_Macro.h>

namespace
{
int ConvertDelayChannelModeToInt(nn::atk::EffectDelay::ChannelMode channelMode) NN_NOEXCEPT
{
    switch(channelMode)
    {
    case nn::atk::EffectBase::ChannelMode_1Ch:
        return 1;
    case nn::atk::EffectBase::ChannelMode_2Ch:
        return 2;
    case nn::atk::EffectBase::ChannelMode_4Ch:
        return 4;
    case nn::atk::EffectBase::ChannelMode_6Ch:
        return 6;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool IsChannelModeSupported(nn::atk::EffectBase::ChannelMode channelMode) NN_NOEXCEPT
{
    return channelMode == nn::atk::EffectBase::ChannelMode_1Ch || channelMode == nn::atk::EffectBase::ChannelMode_2Ch
        || channelMode == nn::atk::EffectBase::ChannelMode_4Ch || channelMode == nn::atk::EffectBase::ChannelMode_6Ch;
}

int ConvertDelaySampleRateToInt(nn::atk::EffectDelay::SampleRate sampleRate) NN_NOEXCEPT
{
    switch(sampleRate)
    {
    case nn::atk::EffectBase::SampleRate_32000:
        return 32000;
    case nn::atk::EffectBase::SampleRate_48000:
        return 48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
}

namespace nn {
namespace atk {

NN_DEFINE_STATIC_CONSTANT(const int EffectDelay::ChannelSettingCountMax);

// 最大値・最小値の定義
const float EffectDelay::InGainMin = 0.0f;
const float EffectDelay::InGainMax = 1.0f;
const float EffectDelay::FeedbackGainMin = 0.0f;
const float EffectDelay::FeedbackGainMax = 1.0f;
const float EffectDelay::DryGainMin = 0.0f;
const float EffectDelay::DryGainMax = 1.0f;
const float EffectDelay::LowPassAmountMin = 0.0f;
const float EffectDelay::LowPassAmountMax = 1.0f;
const float EffectDelay::ChannelSpreadMin = 0.0f;
const float EffectDelay::ChannelSpreadMax = 1.0f;

// デフォルト値の定義
const int64_t EffectDelay::DefaultDelayTimeMaxMilliSeconds = 2000;
const int64_t EffectDelay::DefaultDelayTimeMilliSeconds = 160;
const float EffectDelay::DefaultInGain = 0.5f;
const float EffectDelay::DefaultFeedbackGain = 0.4f;
const float EffectDelay::DefaultDryGain = 0.5f;
const float EffectDelay::DefaultLowPassAmount = 0.8f;
const float EffectDelay::DefaultChannelSpread = 0.3f;

EffectDelay::EffectDelay() NN_NOEXCEPT
    : EffectBase()
    , m_AudioRendererUpdateCountWhenAddedDelay(0)
    , m_ChannelMode(ChannelMode_4Ch)
{
    ResetChannelIndex();
}

EffectDelay::~EffectDelay() NN_NOEXCEPT
{
}

size_t EffectDelay::GetRequiredMemSize() const NN_NOEXCEPT
{
    int maxChannelCount = ConvertDelayChannelModeToInt(m_ChannelMode);
    int sampleRate = ConvertDelaySampleRateToInt(m_SampleRate);
    return nn::audio::GetRequiredBufferSizeForDelay(GetDelayTimeMax(), sampleRate, maxChannelCount);
}

nn::TimeSpan EffectDelay::GetDefaultDelayTimeMax() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DefaultDelayTimeMaxMilliSeconds);
}

nn::TimeSpan EffectDelay::GetDefaultDelayTime() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(DefaultDelayTimeMilliSeconds);
}

bool EffectDelay::AddEffect(nn::audio::AudioRendererConfig* pConfig, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(m_IsActive)
    {
        return false;
    }

    int channelCount = ConvertDelayChannelModeToInt(m_ChannelMode);
    int sampleRate = ConvertDelaySampleRateToInt(m_SampleRate);
    NN_SDK_ASSERT_NOT_NULL(m_EffectBuffer);
    NN_SDK_REQUIRES_ALIGNED( m_EffectBuffer, nn::audio::BufferAlignSize );
    NN_SDK_ASSERT_GREATER_EQUAL(m_EffectBufferSize, nn::audio::GetRequiredBufferSizeForDelay(GetDelayTimeMax(), sampleRate, channelCount));
    NN_UNUSED(sampleRate);

    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        m_AudioRendererUpdateCountWhenAddedDelay = detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();

        nn::Result result;
        switch( pOutputMixer->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
            {
                result = nn::audio::AddDelay( pConfig, &m_DelayType, m_EffectBuffer, m_EffectBufferSize, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance(), GetDelayTimeMax(), channelCount );
                break;
            }
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
            {
                result = nn::audio::AddDelay( pConfig, &m_DelayType, m_EffectBuffer, m_EffectBufferSize, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance(), GetDelayTimeMax(), channelCount );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        if( result.IsFailure() )
        {
            return false;
        }
    }

    m_IsActive = true;

    // 初期化後に DelayType へパラメータを反映させる
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayTime(&m_DelayType, m_Param.delayTime);
        nn::audio::SetDelayInGain(&m_DelayType, m_Param.inGain);
        nn::audio::SetDelayFeedbackGain(&m_DelayType, m_Param.feedbackGain);
        nn::audio::SetDelayDryGain(&m_DelayType, m_Param.dryGain);
        nn::audio::SetDelayLowPassAmount(&m_DelayType, m_Param.lowPassAmount);
        nn::audio::SetDelayChannelSpread(&m_DelayType, m_Param.channelSpread);
        nn::audio::SetDelayEnabled(&m_DelayType, m_Param.isEnabled);
    }

    return true;
}

void EffectDelay::SetEffectInputOutput(const int8_t* input, const int8_t* output, int inputCount, int outputCount) NN_NOEXCEPT
{
    // delay は入出力のチャンネル数が同じ
    NN_SDK_ASSERT_EQUAL(inputCount, outputCount);
    NN_SDK_ASSERT_NOT_NULL(input);
    NN_SDK_ASSERT_NOT_NULL(output);
    NN_SDK_ASSERT_GREATER(inputCount, 0);

    int channelCount = ConvertDelayChannelModeToInt(m_ChannelMode);
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        NN_SDK_ASSERT_MINMAX(channelCount, 0, nn::audio::GetDelayChannelCountMax(&m_DelayType));
    }
    // 設定した Delay のチャンネル数が与えられたバッファ数よりも少ないことを確認する
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, inputCount);

    NN_UNUSED(inputCount);
    NN_UNUSED(outputCount);
    NN_UNUSED(output);

    // ディレイのチャンネル設定を適用する
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelSettingCountMax);
    int8_t channelBus[ChannelSettingCountMax];
    for (int i = 0; i < channelCount; i++)
    {
        // m_ChannelSetting[i] が input の範囲外の配列を指していないか確認する
        NN_SDK_ASSERT_RANGE(m_ChannelSetting[i], 0, inputCount);
        channelBus[i] = input[m_ChannelSetting[i]];
    }

    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayInputOutput(&m_DelayType, channelBus, channelBus, channelCount);
    }
}

void EffectDelay::RemoveEffect(nn::audio::AudioRendererConfig* pConfig, nn::atk::OutputMixer* pOutputMixer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutputMixer);
    if(!m_IsActive)
    {
        return;
    }

    {
#ifdef NN_ATK_CONFIG_ENABLE_VOICE_COMMAND
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
#endif
        NN_SDK_ASSERT(nn::audio::IsDelayRemovable(&m_DelayType));

        void* returnedBuffer = nullptr;
        switch( pOutputMixer->GetReceiverType() )
        {
            case OutputReceiver::ReceiverType::ReceiverType_SubMix:
            {
                returnedBuffer = nn::audio::RemoveDelay( pConfig, &m_DelayType, reinterpret_cast<SubMix*>( pOutputMixer )->GetAudioSubMixInstance() );
                break;
            }
            case OutputReceiver::ReceiverType::ReceiverType_FinalMix:
            {
                returnedBuffer = nn::audio::RemoveDelay( pConfig, &m_DelayType, reinterpret_cast<FinalMix*>( pOutputMixer )->GetAudioFinalMixInstance() );
                break;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        NN_SDK_ASSERT_EQUAL(returnedBuffer, m_EffectBuffer);
        NN_UNUSED(returnedBuffer);
    }

    m_IsActive = false;
}

bool EffectDelay::IsRemovable() const NN_NOEXCEPT
{
    if (!m_IsActive)
    {
        return false;
    }

    //  AddDelay 直後は IsDelayRemovable が true を返すため、
    //  1 回以上 RequestUpdateAudioRenderer が呼ばれてから IsDelayRemovable で確認するようにします

    if( !( m_AudioRendererUpdateCountWhenAddedDelay < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount() ) )
    {
        return false;
    }

    detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
    if( !nn::audio::IsDelayRemovable(&m_DelayType) )
    {
        return false;
    }

    //  RequestUpdateAudioRenderer の呼び出し回数を 2 度比較している理由はコミットログを参照
    return m_AudioRendererUpdateCountWhenAddedDelay < detail::driver::HardwareManager::GetInstance().GetAudioRendererUpdateCount();
}

bool EffectDelay::IsClearable() NN_NOEXCEPT
{
    return IsRemovable();
}

EffectDelay::ChannelMode EffectDelay::GetChannelMode() const NN_NOEXCEPT
{
    return m_ChannelMode;
}

bool EffectDelay::SetChannelMode(ChannelMode mode) NN_NOEXCEPT
{
    ChannelIndex defaultChannelSetting[ChannelModeCountMax];
    for (int i = 0; i < ChannelModeCountMax; i++)
    {
        defaultChannelSetting[i] = static_cast<ChannelIndex>(i);
    }
    return SetChannelIndex(defaultChannelSetting, mode);
}

void EffectDelay::GetChannelIndex(ChannelIndex* pChannel, int channelCount) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);
    int effectChannelCount = ConvertDelayChannelModeToInt(m_ChannelMode);
    NN_SDK_REQUIRES_GREATER_EQUAL(channelCount, effectChannelCount);
    NN_UNUSED(channelCount);
    for(int i = 0; i < effectChannelCount; i++)
    {
        pChannel[i] = static_cast<ChannelIndex>(m_ChannelSetting[i]);
    }
}

bool EffectDelay::SetChannelIndex( const ChannelIndex* pChannel, ChannelMode channelMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChannel);

    int channelCount = ConvertChannelModeToInt(channelMode);
    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelModeCountMax);
    for (int i = 0; i < channelCount; i++)
    {
        NN_SDK_REQUIRES_RANGE(pChannel[i], 0, nn::atk::ChannelIndex_Count);
        for (int j = i + 1; j < channelCount; j++)
        {
            NN_SDK_REQUIRES_NOT_EQUAL(pChannel[i], pChannel[j]);
        }
    }

    if (m_IsActive)
    {
        return false;
    }

    if (IsChannelModeSupported(channelMode))
    {
        m_ChannelMode = channelMode;
    }
    else
    {
        NN_ATK_WARNING("unsupported delay channel mode.");
        return false;
    }

    NN_SDK_ASSERT_LESS_EQUAL(channelCount, ChannelSettingCountMax);
    for(int i = 0; i < channelCount; i++)
    {
        m_ChannelSetting[i] = pChannel[i];
    }

    return true;
}

nn::TimeSpan EffectDelay::GetDelayTimeMax() const NN_NOEXCEPT
{
    return m_Param.delayTimeMax;
}

bool EffectDelay::SetDelayTimeMax( const nn::TimeSpan delayTimeMax ) NN_NOEXCEPT
{
    if(m_IsActive)
    {
        return false;
    }

    m_Param.delayTimeMax = delayTimeMax;
    return true;
}

nn::TimeSpan EffectDelay::GetDelayTime() const NN_NOEXCEPT
{
    return m_Param.delayTime;
}

void EffectDelay::SetDelayTime( const nn::TimeSpan delayTime ) NN_NOEXCEPT
{
    m_Param.delayTime = nn::atk::detail::fnd::Clamp(delayTime, nn::TimeSpan::FromMilliSeconds(0), GetDelayTimeMax());
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayTime(&m_DelayType, m_Param.delayTime);
    }
}

float EffectDelay::GetInGain() const NN_NOEXCEPT
{
    return m_Param.inGain;
}

void EffectDelay::SetInGain( const float inGain ) NN_NOEXCEPT
{
    m_Param.inGain = nn::atk::detail::fnd::Clamp(inGain, InGainMin, InGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayInGain(&m_DelayType, m_Param.inGain);
    }
}

float EffectDelay::GetFeedbackGain() const NN_NOEXCEPT
{
    return m_Param.feedbackGain;
}

void EffectDelay::SetFeedbackGain( const float feedbackGain ) NN_NOEXCEPT
{
    m_Param.feedbackGain = nn::atk::detail::fnd::Clamp(feedbackGain, FeedbackGainMin, FeedbackGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayFeedbackGain(&m_DelayType, m_Param.feedbackGain);
    }
}

float EffectDelay::GetDryGain() const NN_NOEXCEPT
{
    return m_Param.dryGain;
}

void EffectDelay::SetDryGain( float dryGain ) NN_NOEXCEPT
{
    m_Param.dryGain = nn::atk::detail::fnd::Clamp(dryGain, DryGainMin, DryGainMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayDryGain(&m_DelayType, m_Param.dryGain);
    }
}

float EffectDelay::GetLowPassAmount() const NN_NOEXCEPT
{
    return m_Param.lowPassAmount;
}

void EffectDelay::SetLowPassAmount( float lowPassAmount ) NN_NOEXCEPT
{
    m_Param.lowPassAmount = nn::atk::detail::fnd::Clamp(lowPassAmount, LowPassAmountMin, LowPassAmountMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayLowPassAmount(&m_DelayType, m_Param.lowPassAmount);
    }
}

float EffectDelay::GetChannelSpread() const NN_NOEXCEPT
{
    return m_Param.channelSpread;
}

void EffectDelay::SetChannelSpread( float channelSpread ) NN_NOEXCEPT
{
    m_Param.channelSpread = nn::atk::detail::fnd::Clamp(channelSpread, ChannelSpreadMin, ChannelSpreadMax);
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayChannelSpread(&m_DelayType, m_Param.channelSpread);
    }
}

bool EffectDelay::IsEnabled() const NN_NOEXCEPT
{
    return m_Param.isEnabled;
}

void EffectDelay::SetEnabled( bool isEnabled ) NN_NOEXCEPT
{
    m_Param.isEnabled = isEnabled;
    if(m_IsActive)
    {
        detail::driver::HardwareManager::UpdateAudioRendererScopedLock lock;
        nn::audio::SetDelayEnabled(&m_DelayType, m_Param.isEnabled);
    }
}

void EffectDelay::ResetChannelIndex() NN_NOEXCEPT
{
    for(int i = 0; i < ChannelSettingCountMax; i++)
    {
        m_ChannelSetting[i] = static_cast<ChannelIndex>(i);
    }
}

EffectDelay::DelayParam::DelayParam() NN_NOEXCEPT
    : delayTimeMax(GetDefaultDelayTimeMax())
    , delayTime(GetDefaultDelayTime())
    , inGain(DefaultInGain)
    , feedbackGain(DefaultFeedbackGain)
    , dryGain(DefaultDryGain)
    , lowPassAmount(DefaultLowPassAmount)
    , channelSpread(DefaultChannelSpread)
    , isEnabled(false)
{
}

}}
