﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveSoundPlayer.h>
#include <nn/atk/atk_WaveFileReader.h>
#include <nn/atk/atk_DisposeCallbackManager.h>
#include <nn/atk/atk_SoundStartable.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_SoundDataManager.h>

namespace nn { namespace atk { namespace detail { namespace driver {

NN_DEFINE_STATIC_CONSTANT( const int WaveSoundPlayer::PauseReleaseValue );
NN_DEFINE_STATIC_CONSTANT( const int WaveSoundPlayer::MuteReleaseValue );
NN_DEFINE_STATIC_CONSTANT( const int WaveSoundPlayer::DefaultPriority );

WaveSoundPlayer::WaveSoundPlayer() NN_NOEXCEPT
: m_IsInitialized(false)
, m_IsRegisterPlayerCallback(false)
, m_pLoaderManager(NULL)
, m_pLoader(NULL)
{
}

WaveSoundPlayer::~WaveSoundPlayer() NN_NOEXCEPT
{
    Finalize();
}

void WaveSoundPlayer::Initialize(OutputReceiver* pOutputReceiver) NN_NOEXCEPT
{
    BasicSoundPlayer::Initialize( pOutputReceiver );

    m_ReleasePriorityFixFlag = false;
    m_PanRange     = 1.0f;

    m_Priority     = DefaultPriority;

    m_pWsdFile = NULL;
    m_pWaveFile = NULL;
    m_WaveSoundIndex = -1;

    m_DelayCount = 0;
    m_UpdateType = UpdateType_AudioFrame;

    m_WaveSoundInfo.pitch        = 1.0f;
    m_WaveSoundInfo.pan          = 64;
    m_WaveSoundInfo.surroundPan  = 0;
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        m_WaveSoundInfo.fxSend[ i ] = 0;
    }
    m_WaveSoundInfo.mainSend     = 127;

    m_LfoParam.Initialize();

    m_WavePlayFlag = false;
    m_pChannel = NULL;
    m_WaveType = WaveType_Invalid;
    m_ResState = ResState_Invalid;
    m_IsRegisterPlayerCallback = false;
    m_IsInitialized = true;
}

void WaveSoundPlayer::Finalize() NN_NOEXCEPT
{
    FinishPlayer();

    if (IsActive())
    {
        DisposeCallbackManager::GetInstance().UnregisterDisposeCallback( this );
        CloseChannel();
        SetActiveFlag(false);
    }

    if (m_IsInitialized)
    {
        BasicSoundPlayer::Finalize();
        m_IsInitialized = false;
    }

    FreeLoader();
}

void WaveSoundPlayer::Prepare(const StartInfo& info, const PrepareArg& arg) NN_NOEXCEPT
{
    if ( IsActive() )
    {
        FinishPlayer();
    }

    // StartInfo
    {
        m_WaveSoundIndex = info.index;
        m_StartOffsetType = info.startOffsetType;
        m_StartOffset = info.startOffset;
        m_DelayCount = info.delayCount != 0 ? info.delayCount : ToDelayCount(info.delayTime);
        m_WaveSoundParameterFlag = info.waveSoundParameterFlag;
        m_Release = info.release;
        if(m_WaveSoundParameterFlag & nn::atk::SoundStartable::StartInfo::WaveSoundInfo::EnableParameterFlagBit_ContextCalculationSkipMode)
        {
            m_IsContextCalculationSkipMode = info.isContextCalculationSkipMode;
        }
        else
        {
            m_IsContextCalculationSkipMode = false;
        }
        m_LoopInfo = info.loopInfo;
        m_UpdateType = info.updateType;
    }

    // PrepareArg
    {
        m_pWsdFile = arg.wsdFile;
        m_pWaveFile = arg.waveFile;
        m_WaveType = arg.waveType;
    }

    m_ResState = ResState_Assigned;
    SetActiveFlag(true);

    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
    SoundThread::GetInstance().RegisterPlayerCallback( this );
    m_IsRegisterPlayerCallback = true;
}

void WaveSoundPlayer::PrepareForPlayerHeap(const PrepareArg& arg) NN_NOEXCEPT
{
    if ( IsActive() )
    {
        FinishPlayer();
    }

    // PrepareArg
    {
        m_pWsdFile = arg.wsdFile;
        m_pWaveFile = arg.waveFile;
        m_WaveType = arg.waveType;
    }

    m_ResState = ResState_Assigned;
    SetActiveFlag(true);
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
}

void WaveSoundPlayer::RequestLoad(const StartInfo& info, const WaveSoundLoader::Arg& arg) NN_NOEXCEPT
{
    // StartInfo
    {
        m_WaveSoundIndex = info.index;
        m_StartOffsetType = info.startOffsetType;
        m_StartOffset = info.startOffset;
        m_DelayCount = info.delayCount != 0 ? info.delayCount : ToDelayCount(info.delayTime);
        m_Release = info.release;
        if(m_WaveSoundParameterFlag & nn::atk::SoundStartable::StartInfo::WaveSoundInfo::EnableParameterFlagBit_ContextCalculationSkipMode)
        {
            m_IsContextCalculationSkipMode = info.isContextCalculationSkipMode;
        }
        else
        {
            m_IsContextCalculationSkipMode = false;
        }
        m_LoopInfo = info.loopInfo;
        m_UpdateType = info.updateType;
    }

    // Loader::Arg
    m_LoaderArg = arg;  // 構造体コピー
    m_ResState = ResState_ReceiveLoadReq;

    SoundThread::GetInstance().RegisterPlayerCallback( this );
    m_IsRegisterPlayerCallback = true;
}

void WaveSoundPlayer::Start() NN_NOEXCEPT
{
    SetStartedFlag(true);
}

void WaveSoundPlayer::Stop() NN_NOEXCEPT
{
    FinishPlayer();
}

void WaveSoundPlayer::Pause( bool flag ) NN_NOEXCEPT
{
    SetPauseFlag(flag);

    // チャンネルを一時停止
    if ( IsChannelActive() )
    {
        if ( m_pChannel->IsPause() != flag )
        {
            m_pChannel->Pause( flag );
        }
    }
}

void WaveSoundPlayer::SetPanRange( float panRange ) NN_NOEXCEPT
{
    m_PanRange = panRange;
}

void WaveSoundPlayer::SetChannelPriority( int priority ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( priority >= nn::atk::ChannelPriorityMin && priority <= nn::atk::ChannelPriorityMax );
    m_Priority = static_cast<uint8_t>( priority );
}

void WaveSoundPlayer::SetReleasePriorityFix( bool fix ) NN_NOEXCEPT
{
    m_ReleasePriorityFixFlag = fix;
}

void WaveSoundPlayer::InvalidateData( const void* start, const void* end ) NN_NOEXCEPT
{
    if ( IsActive() )
    {
        const void* current = m_pWsdFile;
        if ( start <= current && current <= end )
        {
            FinishPlayer();
        }
    }
}

void WaveSoundPlayer::FinishPlayer() NN_NOEXCEPT
{
    SetFinishFlag(true);
    if (m_IsRegisterPlayerCallback)
    {
        SoundThread::GetInstance().UnregisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = false;
    }
    if (IsStarted())
    {
        SetStartedFlag(false);
    }
}

position_t WaveSoundPlayer::GetPlaySamplePosition(bool isOriginalSamplePosition) const NN_NOEXCEPT
{
    if ( m_pChannel == NULL ) return 0;

    return m_pChannel->GetCurrentPlayingSample(isOriginalSamplePosition);
}

void WaveSoundPlayer::Update() NN_NOEXCEPT
{
    // データロード状態の管理
    {
        switch (m_ResState)
        {
        case ResState_Assigned:
            break;
        case ResState_ReceiveLoadReq:
            if (TryAllocLoader())
            {
                m_pLoader->Initialize(m_LoaderArg);
            }
            else
            {
                return; // リトライする
            }
            // /* FALLTHROUGH */
        case ResState_AppendLoadTask:
            NN_SDK_ASSERT_NOT_NULL(m_pLoader);
            if (m_pLoader->TryWait())
            {
                if (m_pLoader->IsLoadSuccess())
                {
                    // タスク完了したので、プリペア
                    PrepareArg arg;
                    arg.wsdFile = m_pLoader->GetWsdFile();
                    arg.waveFile = m_pLoader->GetWaveFile();
                    // arg.waveType はデフォルト (0 = WaveType_Nwwav) のまま
                    PrepareForPlayerHeap(arg);
                }
                else
                {
                    // データロードに失敗したので終了
                    FinishPlayer();
                    return;
                }
            }
            else
            {
                return; // リトライする
            }
            break;
        default:
            NN_SDK_ASSERT(false);
            break;
        }
    }

    if (m_DelayCount > 0)
    {
        --m_DelayCount;
        return;
    }

    // フラグチェック
    {
        if ( !IsActive() )
        {
            return;
        }
        if ( !IsStarted() )
        {
            return;
        }
    }

    // 波形再生
    if ( !IsPause() )
    {
        // 波形終端終了チェック
        if ( ( m_WavePlayFlag ) && ( m_pChannel == NULL ) )
        {
            FinishPlayer();
            return;
        }

        // 波形再生
        if ( ! m_WavePlayFlag )
        {
            if ( ! StartChannel() )
            {
                FinishPlayer();
                return;
            }
        }
    }

    UpdateChannel();
}

bool WaveSoundPlayer::TryAllocLoader() NN_NOEXCEPT
{
    if (m_pLoaderManager == NULL)
    {
        return false;
    }

    WaveSoundLoader* loader = m_pLoaderManager->Alloc();
    if (loader == NULL)
    {
        return false;
    }

    m_pLoader = loader;
    m_ResState = ResState_AppendLoadTask;
    return true;
}

void WaveSoundPlayer::FreeLoader() NN_NOEXCEPT
{
    if (m_pLoader)
    {
        m_pLoaderManager->Free(m_pLoader);
        m_pLoader = NULL;
    }
}

bool WaveSoundPlayer::StartChannel() NN_NOEXCEPT
{
    const int priority = GetChannelPriority() + DefaultPriority;

    WaveInfo waveInfo;
    {
        // NN_DETAIL_ATK_INFO("--%20s-- ", GetThName());NN_DETAIL_ATK_INFO("WaveSoundPlayer(%p)::%s %c%c%c%c\n", this, __FUNCTION__,
        //         ((char*)(m_pWaveFile))[0],
        //         ((char*)(m_pWaveFile))[1],
        //         ((char*)(m_pWaveFile))[2],
        //         ((char*)(m_pWaveFile))[3]);
        detail::WaveFileReader reader( m_pWaveFile, m_WaveType );
        if ( ! reader.ReadWaveInfo( &waveInfo ) )
        {
            return false;
        }
    }

    {
        // StartInfo でのループ設定が指定されている場合、その値を使用する
        if(m_LoopInfo.loopFlagBit & nn::atk::SoundStartable::StartInfo::LoopInfo::EnableParameterFlagBit_LoopEnabled)
        {
            waveInfo.loopFlag = m_LoopInfo.isLoopEnabled;
        }
    }

    // 開始オフセット
    position_t startOffsetSamples = 0;
    if ( m_StartOffsetType == StartOffsetType_Sample )
    {
        startOffsetSamples = static_cast<uint32_t>( m_StartOffset );
    }
    else if ( m_StartOffsetType == StartOffsetType_Millisec )
    {
        startOffsetSamples = static_cast<uint32_t>(
            static_cast<int64_t>( m_StartOffset ) * waveInfo.sampleRate / 1000
        );
    }

    // 開始オフセットが範囲外なので再生せずに終了
    if ( startOffsetSamples > waveInfo.loopEndFrame )
    {
        return false;
    }

    Channel* channel;
    channel = Channel::AllocChannel(
        std::min( static_cast<int>( waveInfo.channelCount ), 2 ),
        priority,
        ChannelCallbackFunc,
        this
    );

    // 鳴らせなかったらエラーでサウンド終了
    if ( channel == NULL )
    {
        return false;
    }

    // ウェーブサウンド情報取得
    int release;
    {
        WaveSoundFileReader reader( m_pWsdFile );
        if ( ! reader.ReadWaveSoundInfo( &m_WaveSoundInfo, m_WaveSoundIndex ) )
        {
            return false;
        }

        release = m_WaveSoundInfo.adshr.GetRelease();
        // StartInfo でリリース値が指定されている場合はそれを用いる
        if(m_WaveSoundParameterFlag & nn::atk::SoundStartable::StartInfo::WaveSoundInfo::EnableParameterFlagBit_Release)
        {
            release = m_Release;
        }
    }

    // エンベロープ設定
    channel->SetAttack( m_WaveSoundInfo.adshr.GetAttack() );
    channel->SetHold( m_WaveSoundInfo.adshr.GetHold() );
    channel->SetDecay( m_WaveSoundInfo.adshr.GetDecay() );
    channel->SetSustain( m_WaveSoundInfo.adshr.GetSustain() );
    channel->SetRelease( release );

    channel->SetReleasePriorityFix( m_ReleasePriorityFixFlag );
    channel->SetUpdateType( m_UpdateType );
    channel->SetOutputReceiver( GetOutputReceiver() );

    channel->Start( waveInfo, -1, startOffsetSamples, m_IsContextCalculationSkipMode );

    // チャンネルリストへの結合
    m_pChannel = channel;

    m_WavePlayFlag = true;

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         CloseChannel

  Description:  チャンネルを閉じます

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void WaveSoundPlayer::CloseChannel() NN_NOEXCEPT
{
    // Release
    if ( IsChannelActive() )
    {
        UpdateChannel();
        m_pChannel->Release();
    }

    // Free
    if ( m_pChannel != NULL )
    {
        Channel::DetachChannel( m_pChannel );
        m_pChannel = NULL;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         UpdateChannel

  Description:  トラックが保持しているチャンネルのパラメータを更新します

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void WaveSoundPlayer::UpdateChannel() NN_NOEXCEPT
{
    if ( m_pChannel == NULL ) return;

    // volume
    float volume = 1.0f;
    volume *= GetVolume();

    float pitchRatio = 1.0f;
    pitchRatio *= GetPitch();
    pitchRatio *= m_WaveSoundInfo.pitch;

    // lpf freq
    float lpfFreq = static_cast<float>( m_WaveSoundInfo.lpfFreq - 64 ) / 64.0f;
    lpfFreq += GetLpfFreq();

    // biquad filter
    int biquadType = static_cast<int>( m_WaveSoundInfo.biquadType );
    float biquadValue = static_cast<float>( m_WaveSoundInfo.biquadValue ) / 127.0f;
    int handleBiquadType = GetBiquadFilterType();
    if ( handleBiquadType != static_cast<int>(BiquadFilterType_Inherit) )
    {
        biquadType  = handleBiquadType;
        biquadValue = GetBiquadFilterValue();
    }

    // データの値を算出する (TODO:キャッシュできそうなので、キャッシュする)
    float panBase = 0.0f;
    if ( m_WaveSoundInfo.pan <= 1 ) // pan の 1 と 2 は同じ値になる
    {
        panBase = static_cast<float>( static_cast<int>( m_WaveSoundInfo.pan ) - 63 ) / 63.0f;
    }
    else
    {
        panBase = static_cast<float>( static_cast<int>( m_WaveSoundInfo.pan ) - 64 ) / 63.0f;
    }
    panBase *= m_PanRange;

    float spanBase = 0.0f;
    if ( m_WaveSoundInfo.surroundPan <= 63 )
    {
        spanBase = static_cast<float>( m_WaveSoundInfo.surroundPan ) / 63.0f;
    }
    else
    {
        spanBase = static_cast<float>( m_WaveSoundInfo.surroundPan + 1 ) / 64.0f;
            // NOTE: y = (1/64) * x + 1/64 = (x/64) + (1/64) = (x+1)/64
    }

    float mainSendBase = static_cast<float>( m_WaveSoundInfo.mainSend ) / 127.0f - 1.0f;
    float fxSendBase[AuxBus_Count];
    for ( int i = 0; i < AuxBus_Count; i++ )
    {
        fxSendBase[i] = static_cast<float>( m_WaveSoundInfo.fxSend[i] ) / 127.0f;
    }

    // TV 出力向けパラメータ (データの値と重ね合わせる)
    OutputParam tvParam = GetTvParam();
    {
        tvParam.pan += panBase;
        tvParam.span += spanBase;
        tvParam.send[Util::GetSubMixBusFromMainBus()] += mainSendBase;
        for ( int i = 0; i < AuxBus_Count; i++ )
        {
            tvParam.send[Util::GetSubMixBus(i)] += fxSendBase[i];
        }
    }

    // Channel インスタンスへの反映
    m_pChannel->SetPanMode( GetPanMode() );
    m_pChannel->SetPanCurve( GetPanCurve() );
    m_pChannel->SetUserVolume( volume );
    m_pChannel->SetUserPitchRatio( pitchRatio );
    m_pChannel->SetLfoParam( m_LfoParam );
    m_pChannel->SetUserLpfFreq( lpfFreq );
    m_pChannel->SetBiquadFilter( biquadType, biquadValue );
    m_pChannel->SetOutputLine( GetOutputLine() );

    m_pChannel->SetTvParam( tvParam );
    if(GetTvAdditionalParamAddr() != nullptr)
    {
        m_pChannel->SetTvAdditionalParam( *GetTvAdditionalParamAddr() );
    }
}

/*--------------------------------------------------------------------------------*
  Name:         ChannelCallbackFunc

  Description:  チャンネルから呼びだされるコールバック関数

  Arguments:    dropChannel - チャンネルポインタ
                status - チャンネルコールバックステータス
                userData - トラックポインタを格納したコールバックユーザーデータ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void WaveSoundPlayer::ChannelCallbackFunc(
    Channel* dropChannel,
    Channel::ChannelCallbackStatus /* status */,
    void* userData
) NN_NOEXCEPT
{
    WaveSoundPlayer* player = reinterpret_cast<WaveSoundPlayer*>( userData );

    NN_SDK_ASSERT_NOT_NULL( dropChannel );
    NN_SDK_ASSERT_NOT_NULL( player );
    NN_SDK_ASSERT( dropChannel == player->m_pChannel );
#if defined(NN_SDK_BUILD_RELEASE)
    NN_UNUSED( dropChannel );
#endif

    // チャンネル参照の切断
    player->m_pChannel = NULL;
}

}}}} // namespace nn::atk::detail::driver

