﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveSoundLoader.h>
#include <nn/atk/atk_SoundDataManager.h>
#include <nn/atk/atk_SoundPlayer.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/detail/atk_Macro.h>

#include <nn/atk/atk_WarningCallback.h>

namespace nn { namespace atk { namespace detail { namespace driver {

WaveSoundLoader::~WaveSoundLoader() NN_NOEXCEPT
{
    m_Task.Wait();
    m_FreePlayerHeapTask.Wait();
}

void WaveSoundLoader::Initialize(const Arg& arg) NN_NOEXCEPT
{
    m_Task.Wait();
    m_FreePlayerHeapTask.Wait();

    NN_SDK_ASSERT(SoundSystem::detail_IsTaskThreadEnabled(), "Task thread is not enabled.");
    m_Task.Initialize();
    m_Task.m_Arg = arg;
    m_Task.m_pPlayerHeap = NULL;
    m_Task.m_pPlayerHeapDataManager = &m_PlayerHeapDataManager;

    m_FreePlayerHeapTask.Initialize();
    m_FreePlayerHeapTask.m_Arg = arg;
    m_FreePlayerHeapTask.m_pPlayerHeap = nullptr;
    m_FreePlayerHeapTask.m_pPlayerHeapDataManager = &m_PlayerHeapDataManager;
}

void WaveSoundLoader::Finalize() NN_NOEXCEPT
{
    m_FreePlayerHeapTask.m_pPlayerHeap = m_Task.m_pPlayerHeap;
    TaskManager::GetInstance().AppendTask(&m_FreePlayerHeapTask, TaskManager::TaskPriority_Middle);
}

bool WaveSoundLoader::TryWait() NN_NOEXCEPT
{
    // プレイヤーヒープの確保
    if (m_Task.TryAllocPlayerHeap() == false)
    {
        return false;
    }

    // データのアペンド
    Task::Status status = m_Task.GetStatus();
    switch (status)
    {
    case Task::Status_Free:
        TaskManager::GetInstance().AppendTask(&m_Task, TaskManager::TaskPriority_Middle);
        return false;
    case Task::Status_Done:
    case Task::Status_Cancel:
        return true;
    default:
        return false;
    }
}

bool WaveSoundLoader::IsInUse() NN_NOEXCEPT
{
    if (!m_Task.TryWait())
    {
        return true;
    }
    if (!m_FreePlayerHeapTask.TryWait())
    {
        return true;
    }

    return false;
}

void WaveSoundLoader::DataLoadTask::Initialize() NN_NOEXCEPT
{
    Task::InitializeStatus();
    m_Data.Initialize();
    m_IsLoadSuccess = false;
}

bool WaveSoundLoader::DataLoadTask::TryAllocPlayerHeap() NN_NOEXCEPT
{
    if (m_pPlayerHeap == NULL)
    {
        NN_SDK_ASSERT_NOT_NULL(m_Arg.soundPlayer);
        m_pPlayerHeap = m_Arg.soundPlayer->detail_AllocPlayerHeap();
        if (m_pPlayerHeap == NULL)
        {
            return false;
        }
    }
    return true;
}

void WaveSoundLoader::DataLoadTask::Execute(TaskProfileLogger& logger) NN_NOEXCEPT
{
    NN_UNUSED( logger );
    NN_SDK_ASSERT_NOT_NULL(m_pPlayerHeap);
    NN_SDK_ASSERT_NOT_NULL(m_pPlayerHeapDataManager);

    m_pPlayerHeapDataManager->Initialize(m_Arg.soundArchive);

    if ( ( m_Arg.loadInfoWsd.address == NULL ) &&
         ( m_Arg.loadInfoWsd.itemId != SoundArchive::InvalidId ) )
    {
        SoundArchive::ItemId soundId = m_Arg.loadInfoWsd.itemId;

        if ( ! m_pPlayerHeapDataManager->LoadData(
                    soundId, m_pPlayerHeap, SoundArchiveLoader::LoadFlag_Wsd) )
        {
            NN_ATK_WARNING("failed to load WSD(%08x) to PlayerHeap", soundId );
            PlayerHeapForWsdWarningCallbackInfo info(soundId);
            nn::atk::SoundSystem::CallWarningCallback(WarningId_PlayerHeap_WaveSoundFileLoadFailed, &info);
            m_pPlayerHeap->SetState(PlayerHeap::State_TaskFinished);
            m_IsLoadSuccess = false;
            return;
        }
        m_Arg.loadInfoWsd.address =
            m_pPlayerHeapDataManager->detail_GetFileAddressByItemId( soundId );
    }

    NN_SDK_ASSERT_NOT_NULL(m_Arg.loadInfoWsd.address);

    // SoundDataManager 側で波形アーカイブがロード済みか？
    const void* waveFile = Util::GetWaveFileOfWaveSound(
            m_Arg.loadInfoWsd.address, m_Arg.index,
            *m_Arg.soundArchive, *m_Arg.soundDataManager );
    // 本来であれば WaveSound は SoundArchive や SoundDataManager に依存したくないが、
    // プレイヤーヒープ利用時のみ、我慢する。
    if ( waveFile == NULL )
    {
        // 未ロードなら、ここでロードする
        if ( ! m_pPlayerHeapDataManager->detail_LoadWaveArchiveByWaveSoundFile(
                    m_Arg.loadInfoWsd.address, m_Arg.index, m_pPlayerHeap ) )
        {
            NN_ATK_WARNING("failed to load WSD(%08x)'s WARC to PlayerHeap", m_Arg.loadInfoWsd.itemId );
            PlayerHeapForWsdWarningCallbackInfo info(m_Arg.loadInfoWsd.itemId);
            nn::atk::SoundSystem::CallWarningCallback(WarningId_PlayerHeap_WaveFileLoadFailed, &info);
            m_pPlayerHeap->SetState(PlayerHeap::State_TaskFinished);
            m_IsLoadSuccess = false;
            return;
        }
        waveFile = Util::GetWaveFileOfWaveSound(
            m_Arg.loadInfoWsd.address,
            m_Arg.index,
            *m_Arg.soundArchive,
            *m_pPlayerHeapDataManager);
    }

    m_Data.wsdFile = m_Arg.loadInfoWsd.address;
    m_Data.waveFile = waveFile;

    m_pPlayerHeap->SetState(PlayerHeap::State_TaskFinished);
    m_IsLoadSuccess = true;
}

void WaveSoundLoader::FreePlayerHeapTask::Initialize() NN_NOEXCEPT
{
    Task::InitializeStatus();
}

void WaveSoundLoader::FreePlayerHeapTask::Execute(TaskProfileLogger& logger) NN_NOEXCEPT
{
    NN_UNUSED( logger );
    if (m_pPlayerHeap)
    {
        m_pPlayerHeap->Clear();
        m_Arg.soundPlayer->detail_FreePlayerHeap(m_pPlayerHeap);
    }

    NN_SDK_ASSERT_NOT_NULL(m_pPlayerHeapDataManager);
    m_pPlayerHeapDataManager->Finalize();
}

}}}} // namespace nn::atk::detail::driver

