﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_WaveSoundFile.h>

#include <nn/atk/atk_ElementType.h>

namespace nn {
namespace atk {
namespace detail {

namespace {

const uint8_t WsdDefaultPan            = 64;   // 中央
const int8_t WsdDefaultSurroundPan     = 0;    // 中央
const float WsdDefaultPitch            = 1.0f;
const uint8_t WsdDefaultMainSend       = 127;  // 最大
const uint8_t WsdDefaultFxSend         = 0;    // 最小
const AdshrCurve WsdDefaultAdshrCurve(
    127,    // uint8_t attack
    127,    // uint8_t decay
    127,    // uint8_t sustain
    127,    // uint8_t hold
    127     // uint8_t release
);
const uint8_t WsdDefaultLpfFreq     = 64;   // フィルタがかかっていない
const uint8_t WsdDefaultBiquadType  = 0;    // フィルタ未使用
const uint8_t WsdDefaultBiquadValue = 0;    // フィルタがかかっていない
const uint8_t WsdDefaultKey         = 64;   // TODO: おそらく中央
const uint8_t WsdDefaultVolume      = 96;

enum WaveSoundInfoBitFlagWsd
{
    WaveSoundInfoBitFlagWsd_Pan = 0x00,
    WaveSoundInfoBitFlagWsd_Pitch,
    WaveSoundInfoBitFlagWsd_Filter,
    WaveSoundInfoBitFlagWsd_Send = 0x08,
    WaveSoundInfoBitFlagWsd_Envelope,
    WaveSoundInfoBitFlagWsd_Randomizer  // 未実装
};

enum NoteInfoBitFlag
{
    NoteInfoBitFlag_Key = 0x00,
    NoteInfoBitFlag_Volume,
    NoteInfoBitFlag_Pan,
    NoteInfoBitFlag_Pitch,
    NoteInfoBitFlag_Send = 0x08,
    NoteInfoBitFlag_Envelope,
    NoteInfoBitFlag_Randomizer,   // 未実装
    NoteInfoBitFlag_Lfo           // 未実装
};

struct SendValueWsd
{
    uint8_t mainSend;
    Util::Table<uint8_t,uint8_t> fxSend;
};

} // anonymous namespace


//
// WaveSoundFile::FileHeader
//
const WaveSoundFile::InfoBlock* WaveSoundFile::FileHeader::GetInfoBlock() const NN_NOEXCEPT
{
    return reinterpret_cast<const InfoBlock*>( GetBlock( ElementType_WaveSoundFile_InfoBlock ) );
}


//
// WaveSoundFile::InfoBlockBody
//

const WaveSoundFile::WaveSoundData&
WaveSoundFile::InfoBlockBody::GetWaveSoundData( uint32_t index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( index < GetWaveSoundCount() );
    const void* pWaveSoundData =
        GetWaveSoundDataReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_WaveSoundMetaData );
    NN_SDK_ASSERT_NOT_NULL( pWaveSoundData );
    return *reinterpret_cast<const WaveSoundData*>( pWaveSoundData );
}
const Util::ReferenceTable&
WaveSoundFile::InfoBlockBody::GetWaveSoundDataReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toWaveSoundDataReferenceTable.offset ).Get() );
}
const Util::WaveIdTable&
WaveSoundFile::InfoBlockBody::GetWaveIdTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::WaveIdTable*>(
            util::ConstBytePtr( this, toWaveIdTable.offset ).Get() );
}


//
// WaveSoundFile::WaveSoundData
//
const WaveSoundFile::WaveSoundInfo&
WaveSoundFile::WaveSoundData::GetWaveSoundInfo() const NN_NOEXCEPT
{
    NN_SDK_ASSERT( toWaveSoundInfo.IsValidTypeId(
                ElementType_WaveSoundFile_WaveSoundInfo ) );

    return *reinterpret_cast<const WaveSoundInfo*>(
            util::ConstBytePtr( this, toWaveSoundInfo.offset ).Get() );
}

const Util::ReferenceTable&
WaveSoundFile::WaveSoundData::GetTrackInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toTrackInfoReferenceTable.offset ).Get() );
}

const Util::ReferenceTable&
WaveSoundFile::WaveSoundData::GetNoteInfoReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toNoteInfoReferenceTable.offset ).Get() );
}

const WaveSoundFile::TrackInfo&
WaveSoundFile::WaveSoundData::GetTrackInfo( uint32_t index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( index < GetTrackCount() );

    const void* pTrackInfo =
        GetTrackInfoReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_TrackInfo );
    NN_SDK_ASSERT_NOT_NULL( pTrackInfo );

    return *reinterpret_cast<const TrackInfo*>( pTrackInfo );
}

const WaveSoundFile::NoteInfo&
WaveSoundFile::WaveSoundData::GetNoteInfo( uint32_t index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( index < GetNoteCount() );

    const void* pNoteInfo =
        GetNoteInfoReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_NoteInfo );
    NN_SDK_ASSERT_NOT_NULL( pNoteInfo );

    return *reinterpret_cast<const NoteInfo*>( pNoteInfo );
}


//
// WaveSoundFile::WaveSoundInfo
//
uint8_t WaveSoundFile::WaveSoundInfo::GetPan() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlagWsd_Pan );
    if ( result == false ) return WsdDefaultPan;
    return Util::DevideBy8bit( value, 0 );
}
int8_t WaveSoundFile::WaveSoundInfo::GetSurroundPan() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlagWsd_Pan );
    if ( result == false ) return WsdDefaultSurroundPan;
    return static_cast<int8_t>( Util::DevideBy8bit( value, 1 ) );
}
float WaveSoundFile::WaveSoundInfo::GetPitch() const NN_NOEXCEPT
{
    float value;
    bool result = optionParameter.GetValueF32( &value, WaveSoundInfoBitFlagWsd_Pitch );
    if ( result == false ) return WsdDefaultPitch;
    return value;
}

void WaveSoundFile::WaveSoundInfo::GetSendValue(
        uint8_t* mainSend, uint8_t* fxSend, uint8_t fxSendCount ) const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlagWsd_Send );
    if ( result == false )
    {
        *mainSend = WsdDefaultMainSend;
        for ( int i = 0; i < fxSendCount; i++ )
        {
            fxSend[ i ] = WsdDefaultFxSend;
        }
        return;
    }

    const SendValueWsd& sendValue = *reinterpret_cast<const SendValueWsd*>(
            util::ConstBytePtr( this, value ).Get() );

    *mainSend = sendValue.mainSend;
    int countSize = sendValue.fxSend.count > AuxBus_Count ?
        AuxBus_Count : sendValue.fxSend.count;
    for ( int i = 0; i < countSize; i++ )
    {
        fxSend[ i ] = sendValue.fxSend.item[ i ];
    }
}

const AdshrCurve& WaveSoundFile::WaveSoundInfo::GetAdshrCurve() const NN_NOEXCEPT
{
    uint32_t offsetToReference;
    bool result = optionParameter.GetValue( &offsetToReference, WaveSoundInfoBitFlagWsd_Envelope );
    if ( result == false ) return WsdDefaultAdshrCurve;

    const Util::Reference& ref = *reinterpret_cast<const Util::Reference*>(
            util::ConstBytePtr( this, offsetToReference ).Get() );
    return *reinterpret_cast<const AdshrCurve*>(
            util::ConstBytePtr( &ref, ref.offset ).Get() );
}

uint8_t WaveSoundFile::WaveSoundInfo::GetLpfFreq() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlagWsd_Filter );
    if ( result == false ) return WsdDefaultLpfFreq;
    return Util::DevideBy8bit( value, 0 );
}

uint8_t WaveSoundFile::WaveSoundInfo::GetBiquadType() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlagWsd_Filter );
    if ( result == false ) return WsdDefaultBiquadType;
    return Util::DevideBy8bit( value, 1 );
}

uint8_t WaveSoundFile::WaveSoundInfo::GetBiquadValue() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, WaveSoundInfoBitFlagWsd_Filter );
    if ( result == false ) return WsdDefaultBiquadValue;
    return Util::DevideBy8bit( value, 2 );
}


//
// WaveSoundFile::TrackInfo
//
const Util::ReferenceTable&
WaveSoundFile::TrackInfo::GetNoteEventReferenceTable() const NN_NOEXCEPT
{
    return *reinterpret_cast<const Util::ReferenceTable*>(
            util::ConstBytePtr( this, toNoteEventReferenceTable.offset ).Get() );
}

const WaveSoundFile::NoteEvent&
WaveSoundFile::TrackInfo::GetNoteEvent( uint32_t index ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT( index < GetNoteEventCount() );

    const void* pNoteEvent =
        GetNoteEventReferenceTable().GetReferedItem(
                index,
                ElementType_WaveSoundFile_NoteEvent );
    NN_SDK_ASSERT_NOT_NULL( pNoteEvent );

    return *reinterpret_cast<const NoteEvent*>( pNoteEvent );
}


//
// WaveSoundFile::NoteInfo
//
uint8_t WaveSoundFile::NoteInfo::GetOriginalKey() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, NoteInfoBitFlag_Key );
    if ( result == false ) return WsdDefaultKey;
    return Util::DevideBy8bit( value, 0 );
}

uint8_t WaveSoundFile::NoteInfo::GetVolume() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, NoteInfoBitFlag_Volume );
    if ( result == false ) return WsdDefaultVolume;
    return Util::DevideBy8bit( value, 0 );
}

uint8_t WaveSoundFile::NoteInfo::GetPan() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, NoteInfoBitFlag_Pan );
    if ( result == false ) return WsdDefaultPan;
    return Util::DevideBy8bit( value, 0 );
}
uint8_t WaveSoundFile::NoteInfo::GetSurroundPan() const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, NoteInfoBitFlag_Pan );
    if ( result == false ) return WsdDefaultSurroundPan;
    return static_cast<int8_t>( Util::DevideBy8bit( value, 1 ) );
}
float WaveSoundFile::NoteInfo::GetPitch() const NN_NOEXCEPT
{
    float value;
    bool result = optionParameter.GetValueF32( &value, NoteInfoBitFlag_Pitch );
    if ( result == false ) return WsdDefaultPitch;
    return value;
}

void WaveSoundFile::NoteInfo::GetSendValue( uint8_t* mainSend, uint8_t* fxSend[], uint8_t fxSendCount ) const NN_NOEXCEPT
{
    uint32_t value;
    bool result = optionParameter.GetValue( &value, NoteInfoBitFlag_Send );
    if ( result == false )
    {
        *mainSend = WsdDefaultMainSend;
        for ( int i = 0; i < fxSendCount; i++ )
        {
            *fxSend[i] = WsdDefaultFxSend;
        }
        return;
    }

    const SendValueWsd& sendValue = *reinterpret_cast<const SendValueWsd*>(
            util::ConstBytePtr( this, value ).Get() );

    NN_SDK_ASSERT( fxSendCount <= sendValue.fxSend.count );
    *mainSend = sendValue.mainSend;
    int countSize = sendValue.fxSend.count > AuxBus_Count ?
        AuxBus_Count : sendValue.fxSend.count;
    for ( int i = 0; i < countSize; i++ )
    {
        *fxSend[ i ] = sendValue.fxSend.item[ i ];
    }
}

const AdshrCurve& WaveSoundFile::NoteInfo::GetAdshrCurve() const NN_NOEXCEPT
{
    uint32_t offsetToReference;
    bool result = optionParameter.GetValue( &offsetToReference, NoteInfoBitFlag_Envelope );
    if ( result == false ) return WsdDefaultAdshrCurve;

    const Util::Reference& ref = *reinterpret_cast<const Util::Reference*>(
            util::ConstBytePtr( this, offsetToReference ).Get() );
    return *reinterpret_cast<const AdshrCurve*>(
            util::ConstBytePtr( &ref, ref.offset ).Get() );
}


} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
