﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_InstancePool.h>

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BytePtr.h>

namespace nn {
namespace atk {
namespace detail {

/*--------------------------------------------------------------------------------*
  Name:         CreateImpl

  Description:  メモリプールの作成

  Arguments:    buffer - バッファの開始アドレス
                size - バッファのサイズ
                objSize - オブジェクトのサイズ

  Returns:      None.
 *--------------------------------------------------------------------------------*/
int PoolImpl::CreateImpl( void* buffer, size_t size, size_t objSize ) NN_NOEXCEPT
{
    return CreateImpl( buffer, size, objSize, DefaultAlignment );
}

/*--------------------------------------------------------------------------------*
  Name:         CreateImpl

  Description:  メモリプールの作成

  Arguments:    buffer - バッファの開始アドレス
                size - バッファのサイズ
                objSize - オブジェクトのサイズ
                alignment - オブジェクトのアライメント

  Returns:      None.
 *--------------------------------------------------------------------------------*/
int PoolImpl::CreateImpl( void* buffer, size_t size, size_t objSize, size_t alignment ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( buffer );

    char* ptr = static_cast<char*>( util::BytePtr( buffer ).AlignUp( alignment ).Get() );
    objSize = nn::util::align_up( objSize, alignment );
    int objectCount = static_cast<int>( ( size - ( ptr - static_cast<char*>(buffer) ) ) / objSize );

    for ( auto i = 0; i < objectCount; i++ )
    {
        PoolImpl *head = reinterpret_cast<PoolImpl*>( ptr );
        head->m_pNext = m_pNext;
        m_pNext = head;
        ptr += objSize;
    }
    m_pBuffer = buffer;
    m_BufferSize = size;
    return objectCount;
}


/*--------------------------------------------------------------------------------*
  Name:         DestroyImpl

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void PoolImpl::DestroyImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( m_pBuffer );

    void* begin = m_pBuffer;
    void* end = static_cast<void*>( static_cast<char*>( m_pBuffer ) + m_BufferSize );
    PoolImpl* ptr = m_pNext;
    PoolImpl* prev = this;
    while ( ptr != NULL )
    {
        if ( ( begin <= ptr ) && ( ptr < end ) )
        {
            prev->m_pNext = ptr->m_pNext;
        }
        else
        {
            prev = ptr;
        }
        ptr = ptr->m_pNext;
    }
}

/*--------------------------------------------------------------------------------*
  Name:         CountImpl

  Description:  空き領域数をカウント

  Arguments:    None.

  Returns:      空き領域数
 *--------------------------------------------------------------------------------*/
int PoolImpl::CountImpl() const NN_NOEXCEPT
{
    int count = 0;
    for ( PoolImpl* ptr = m_pNext; ptr != NULL; ptr = ptr->m_pNext )
    {
        ++count;
    }
    return count;
}

/*--------------------------------------------------------------------------------*
  Name:         AllocImpl

  Description:  プールからメモリ領域を確保

  Arguments:    None.

  Returns:      メモリ領域の開始アドレス
 *--------------------------------------------------------------------------------*/
void* PoolImpl::AllocImpl() NN_NOEXCEPT
{
    if ( m_pNext == NULL ) return NULL;
    PoolImpl* head = m_pNext;
    m_pNext = head->m_pNext;
    return head;
}

/*--------------------------------------------------------------------------------*
  Name:         FreeImpl

  Description:  メモリ領域をプールに開放

  Arguments:    ptr - メモリ領域の開始アドレス

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void PoolImpl::FreeImpl( void* ptr ) NN_NOEXCEPT
{
    PoolImpl* head = reinterpret_cast<PoolImpl*>( ptr );
    head->m_pNext = m_pNext;
    m_pNext = head;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn

