﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/atk_TaskThread.h>
#include <nn/atk/atk_TaskManager.h>
#include <nn/atk/atk_DriverCommand.h>
#include <nn/atk/fnd/os/atkfnd_ScopedLock.h>

namespace nn {
namespace atk {
namespace detail {

/*--------------------------------------------------------------------------------*
  Name:         TaskThread

  Description:  コンストラクタ

  Arguments:    なし

  Returns:      なし
 *--------------------------------------------------------------------------------*/
#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif

TaskThread::TaskThread() NN_NOEXCEPT
: m_IsFinished( false )
, m_IsCreated( false )
, m_FsPriority( FsPriority_Normal )
{
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop) // disable:4355
#endif

TaskThread::~TaskThread() NN_NOEXCEPT
{
    if ( m_IsCreated )
    {
        Destroy();
    }
}

bool TaskThread::Create( int priority, void* stackBase, size_t stackSize, int idealCoreNumber, uint32_t affinityMask, FsPriority fsPriority ) NN_NOEXCEPT
{
    // 多重初期化チェック
    if ( m_IsCreated )
    {
        Destroy();
    }

    m_IsFinished = false;


    nn::atk::detail::fnd::Thread::RunArgs args;
    args.name = "nn::atk::detail::TaskThread";
    args.stack = stackBase;
    args.stackSize = stackSize;
    args.idealCoreNumber = idealCoreNumber;
    args.affinityMask = static_cast<nn::atk::detail::fnd::Thread::AffinityMask>(affinityMask);
    args.fsPriority = static_cast<nn::atk::detail::fnd::Thread::FsPriority>(fsPriority);
    args.priority = priority;
    args.param = nullptr;
    args.handler = this;

    bool result = m_Thread.Run( args );

    if ( ! result )
    {
        NN_DETAIL_ATK_INFO("TaskThread::Create Failed: OSCreateThread is Failed\n");
        return false;
    }

    m_FsPriority = fsPriority;
    m_IsCreated = true;

    return true;
}

/*--------------------------------------------------------------------------------*
  Name:         Destroy

  Description:  タスクスレッドを終了する

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
void TaskThread::Destroy() NN_NOEXCEPT
{
    if ( ! m_IsCreated )
    {
        return;
    }

    // サウンドスレッド終了メッセージ送信
    m_IsFinished = true;
    TaskManager::GetInstance().CancelWaitTask();

    // スレッドの終了を待つ
    m_Thread.WaitForExit();
    m_Thread.Release();

    m_IsCreated = false;
}

/*--------------------------------------------------------------------------------*
  Name:         ThreadHandlerProc

  Description:  タスクスレッドプロシージャ

  Arguments:    None.

  Returns:      None.
 *--------------------------------------------------------------------------------*/
uint32_t TaskThread::Run(void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    while ( ! m_IsFinished )
    {
        TaskManager::GetInstance().WaitTask();
        if ( m_IsFinished )
        {
            break;
        }

        {
            // NOTE: タスクスレッドのロック。
            //
            // SoundSystem::(Try)LockDataLoadThread 関数で外部からロックされることによって
            // いったんスレッドが停止します。
            //
            // タスクスレッドはストリームデータのロードや、
            // プレイヤーヒープへのデータロードを行いますが、
            // 「スリープ中はファイルアクセス禁止」の制約に沿うため、
            // このロックが追加されました。
            fnd::ScopedLock<fnd::CriticalSection> scopeLock(m_CriticalSection);
            TaskManager::GetInstance().ExecuteTask();
        }

        DriverCommand::GetInstanceForTaskThread().RecvCommandReply();
    }

    return 0;
}

TaskThread& TaskThread::GetInstance() NN_NOEXCEPT
{
    static TaskThread instance;
    return instance;
}

} // namespace nn::atk::detail
} // namespace nn::atk
} // namespace nn
